/*
  Liste complte des nombres premiers qui tiennent sur 16 bits

  (c) Olivier Pirson --- http://www.opimedia.be/ --- olivier.pirson.opi@gmail.com
  Dbut le 1er octobre 2011
  8 octobre 2011
*/

#include <stdbool.h>



/*********************
 * Macros constantes *
 *********************/
/* Nombre maximum de facteurs > 1 dont le produit tient dans un nat32 */
#define NAT32_MAX_NB_FACTORS 31

/* Format utilis pour reprsenter un nat32 dans printf() et les fonctions semblables */
#define NAT32_FORMAT "lu"



/********
 * Type *
 ********/
/*
  Naturel (entier non sign) sur 32 bits

  Pre: sizeof(nat32) == 4

  Changer de type en fonction du compilateur !
*/
typedef unsigned long nat32;



/**************
 * Constantes *
 **************/
/* Plus grand nombre premier qui tient sur 16 bits */
extern const unsigned int PRIMES16_MAX;

/* Nombre de nombres premiers qui tiennent sur 16 bits */
extern const unsigned int PRIMES16_NB;



/************
 * Variable *
 ************/
/*
  Liste complte des nombres premiers qui tiennent sur 16 bits, suivie d'un 0

  Pre: doit tre initialise avec primes16_init()
*/
extern unsigned int *primes16;



/**************
 * Prototypes *
 **************/
/*
  Renvoie le produit des nb premiers nombres de array (modulo 2^32)

  Pre: array: tableau d'au moins nb nat32
*/
nat32 nat32_array_to_nat32_product(nat32 *array, unsigned int nb);


/*
  Renvoie le nombre (total) de facteurs premiers de n
  et place dans factors ces facteurs premiers (par ordre croissant)

  Ex. : nat32_to_nat32_array_primefactors(120, *array) renvoie 5 et place {2, 2, 2, 3, 5} dans array

  Pre: n > 0
       array: tableau de nat32 suffisament grand pour contenir le nombre de facteurs et un 0 final
*/
unsigned int nat32_to_nat32_array_primefactors(nat32 n, nat32 *array);


/*
  Renvoie le nombre (total) de facteurs premiers impairs de n
  et place dans factors ces facteurs premiers impairs (par ordre croissant)

  Ex. : nat32_to_nat32_array_primefactors(120, *array) renvoie 5 et place {3, 5} dans array

  Pre: n > 0
       array: tableau de nat32 suffisament grand pour contenir le nombre de facteurs et un 0 final
*/
unsigned int nat32_to_nat32_array_primefactors_odd(nat32 n, nat32 *array);


/*
  Renvoie sigma(n) c.--d. la somme des diviseurs de n

  Pre: n > 0
*/
nat32 nat32_to_nat32_sigma(nat32 n);


/*
  Renvoie sigma_odd(n) c.--d. la somme des diviseurs impairs de n

  Pre: n > 0
*/
nat32 nat32_to_nat32_sigma_odd(nat32 n);


/*
  Renvoie sigma(nat32_array_to_nat32_product(factors))
  c.--d. la somme des diviseurs de nat32_array_to_nat32_product(factors)

  Pre: factors: liste de nombres premiers par ordre croissant, suivie d'un 0
*/
nat32 primefactors_to_nat32_sigma(nat32 *factors);


/*
  Libre la mmoire utilise par primes16
  (si la mmoire tait dj libre alors n'a aucun effet)
*/
void primes16_free();


/*
  Initialise primes16
  (si tait dj initialis alors n'a aucun effet)
*/
void primes16_init();


/*
  Envoie sur la sortie standard la liste des nb premiers nombre de array.
  Chaque nombre est prcd de before et suivi de after

  Pre: array: tableau d'au moins nb nat32
       before: non NULL
       after: non NULL
*/
void print_nat32_array(nat32 *array, unsigned int nb, char *before, char *after);


/*
  Renvoie true si n est dans primes16,
  false sinon

  Result: bool
*/
bool ui_in_primes16(unsigned int n);
