#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "primes16.h"



/**************
 * Constantes *
 **************/
const unsigned int PRIMES16_MAX = 65521;

const unsigned int PRIMES16_NB = 6542;



/************
 * Variable *
 ************/
unsigned int *primes16 = NULL;



/************
 * Fonction *
 ************/
nat32 nat32_array_to_nat32_product(nat32 *array, unsigned int nb) {
  nat32 prod = 1;

  while ( nb-- ) {
    prod *= *(array++);
  }

  return prod;
}


unsigned int nat32_to_nat32_array_primefactors(nat32 n, nat32 *array) {
  nat32 mu = 0;

  unsigned int *p = primes16 + 1;

  assert(0 < n);

  while ( !(n&1) ) {  /* n est pair */
    n /= 2;
    array[mu++] = 2;
  }

  while ( (*p < n) && *p ) {  /* il reste des nombres premiers de primes16  tester */
    if ( ui_in_primes16(n) ) {  /* n est premier */
      break;
    }

    while ( n%*p == 0 ) {  /* *p est le plus petit diviseur de n */
      n /= *p;
      array[mu++] = *p;

      assert( mu <= NAT32_MAX_NB_FACTORS );
    }
    p++;
  }

  if ( n > 1 ) {  /* n est premier */
      array[mu++] = n;
  }

  array[mu] = 0;

  return mu;
}


unsigned int nat32_to_nat32_array_primefactors_odd(nat32 n, nat32 *array) {
  nat32 mu = 0;

  unsigned int *p = primes16 + 1;

  assert(0 < n);

  while ( !(n&1) ) {  /* n est pair */
    n /= 2;
  }

  while ( (*p < n) && *p ) {  /* il reste des nombres premiers de primes16  tester */
    if ( ui_in_primes16(n) ) {  /* n est premier */
      break;
    }

    while ( n%*p == 0 ) {  /* *p est le plus petit diviseur de n */
      n /= *p;
      array[mu++] = *p;

      assert( mu <= NAT32_MAX_NB_FACTORS );
    }
    p++;
  }

  if ( n > 1 ) {  /* n est premier */
      array[mu++] = n;
  }

  array[mu] = 0;

  return mu;
}


nat32 nat32_to_nat32_sigma(nat32 n) {
  nat32 factors[NAT32_MAX_NB_FACTORS + 1];

  assert(n > 0);

  nat32_to_nat32_array_primefactors(n, factors);

  return primefactors_to_nat32_sigma(factors);
}


nat32 nat32_to_nat32_sigma_odd(nat32 n) {
  nat32 factors[NAT32_MAX_NB_FACTORS + 1];

  assert(n > 0);

  nat32_to_nat32_array_primefactors_odd(n, factors);

  return primefactors_to_nat32_sigma(factors);
}


nat32 primefactors_to_nat32_sigma(nat32 *factors) {
  nat32
    pow = 1,
    sigma = 1,
    sum = 1;

  while ( *factors ) {  /* il reste des facteurs premiers */
    pow *= *factors;
    sum += pow;

    if ( *factors != *(factors + 1) ) {  /* il est diffrent du suivant */
      sigma *= sum;
      pow = 1;
      sum = 1;
    }

    factors++;
  }

  return sigma;
}


void primes16_free() {
  if ( primes16 != NULL ) {
    free(primes16);
  }
}


void primes16_init() {
  if ( primes16 == NULL ) {
    unsigned int n = 3;  /* candidat nombre premier */

    unsigned int nb = 1;  /* nombre de nombres premiers trouvs */

    /* Vrifie les prrequis sur les types utiliss (nat32 sur 32 bits et unsigned int au moins sur 16 bits) */
    assert(sizeof(nat32) == 4);
    assert(sizeof(unsigned int) >= 2);


    primes16 = malloc((PRIMES16_NB + 1)*sizeof(unsigned int));
    primes16[0] = 2;  /* premier nombre premier (le seul pair) */

    while ( n < 65535 ) {  /* 65535 == 2^16 - 1 */
      unsigned int i = 0;
      unsigned is_prime = true;

      for ( ; i < nb - 1; i++) {
        if ( n%primes16[i] == 0 ) {  /* n est divisible par un des nombres premiers antrieurement trouvs */
          is_prime = false;

          break;
        }
      }

      if ( is_prime ) {  /* n est premier */
        primes16[nb++] = n;
      }

      n += 2;
    }

    primes16[nb] = 0;  /* ajoute un 0 en fin de tableau */


    /* Vrifie quelques valeurs */
    assert(nb == PRIMES16_NB);

    assert(primes16[0] == 2);
    assert(primes16[1] == 3);
    assert(primes16[2] == 5);
    assert(primes16[3] == 7);
    assert(primes16[4] == 11);

    assert(primes16[PRIMES16_NB - 3] == 65497);
    assert(primes16[PRIMES16_NB - 2] == 65519);
    assert(primes16[PRIMES16_NB - 1] == 65521);
    assert(primes16[PRIMES16_NB] == 0);
  }
}


void print_nat32_array(nat32 *array, unsigned int nb, char *before, char *after) {
  assert(before != NULL);
  assert(after != NULL);

  while ( nb-- ) {
    printf("%s%" NAT32_FORMAT "%s", before, *(array++), after);
  }
}


bool ui_in_primes16(unsigned int n) {
  if ( !(n&1) || (n > PRIMES16_MAX) ) {  /* n pair ou trop grand */
    return n == 2;
  }
  else {                                 /* n impair et <= PRIMES16_MAX */
    /* recherche dichotomique */
    unsigned int
      a = 1,
      b = PRIMES16_NB - 1,
      m,
      v;
    
    do {
      m = (b + a)/2;
      v = primes16[m];

      if ( n < v ) {       /*  gauche */
        b = m - 1;
      }
      else if ( n > v ) {  /*  droite */
        a = m + 1;
      } else {             /* trouv */
        return true;
      }
    } while ( a < b );

    return (n == primes16[a]) || (n == primes16[b]);
  }
}
