#!/usr/bin/env python3

#
# PHPstripDEBUG
#
# Output PHP source with stripped
#   #DEBUG...#DEBUG_END sections, comments and whitespace.
#   (Require Python 3 or better)
#
# Copyright (C) 2010, 2011, 2013, 2015, 2020 Olivier Pirson
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#

#
# olivier.pirson.opi@gmail.com
# DragonSoft DS --- http://www.opimedia.be/DS/
# Débuté le 10 octobre 2010
# v.01.00.00 --- 1er novembre 2010
# v.01.00.01 --- 14 novembre 2010 :
#   Ajout du charset pour les liens plain/text dans le fichier log.
# v.01.01.00 --- 21 novembre 2010 :
#   Ajout warning des 'debug' et 'assert(' qui restent dans le fichier destination.
#   Ajout option -q.
#   Ajout total des tailles des fichiers dans le fichier log.
#   Correction bug du fichier log avec l'option -n.
# v.01.01.01 --- 24 décembre 2010 :
#   Dans les fichiers XHTML :
#     - Ajout d'une transparence sur le panneau d'informations.
#     - Correction du titre PHPstripDEBUG qui n'apparaissait pas sous Opera.
#     - Mise en place de spécialisations pour Internet Explorer.
# v.01.01.02 --- 19 juin 2011 :
#   Remplace la propriété CSS 'display: none;' de #src par 'color: transparent;'.
#   Changement de mon adresse Internet : http://www.opimedia.be/ .
# v.01.01.03 --- 24 mars 2013 :
#   - Remplacé le code JavaScript location.hash par window.location.hash .
#   - Remplacé le <span class="white-space"> contenant les <button> par un <div class="buttons">.
# --- 1er août 2025 :
#   - remplacé www.browserchoice.eu
# v.01.01.04 --- June 19, 2020 :
#   - Some cleaning about old Python 2.
#   - Update email.
##################################################################################################
import collections
import glob
import os
import re
import subprocess
import sys
import tempfile

VERSION = '01.01.04 --- June 19, 2020'


#######################
# Constantes globales #
#######################
# Dictionnaire
#   encoding Python: charset (X)HTML
# D'autres valeurs peuvent y être ajoutées :
# cf. http://docs.python.org/py3k/library/codecs.html#standard-encodings
# et http://htmlhelp.com/tools/validator/supported-encodings.html.fr
ENCODING_CHARSET = {'cp1250': 'windows-1250',
                    'cp1251': 'windows-1251',
                    'cp1252': 'windows-1252',
                    'iso8859-1': 'ISO-8859-1',
                    'iso8859_2': 'ISO-8859-2',
                    'iso8859_3': 'ISO-8859-3',
                    'iso8859_4': 'ISO-8859-4',
                    'iso8859_5': 'ISO-8859-5',
                    'iso8859_6': 'ISO-8859-6',
                    'iso8859_7': 'ISO-8859-7',
                    'iso8859_8': 'ISO-8859-8',
                    'iso8859_9': 'ISO-8859-9',
                    'iso8859_10': 'ISO-8859-10',
                    'iso8859_13': 'ISO-8859-13',
                    'iso8859_14': 'ISO-8859-14',
                    'iso8859_15': 'ISO-8859-15',
                    'latin_1': 'ISO-8859-1',
                    'utf_8': 'UTF-8',
                    'utf_16': 'UTF-16'}


# Dictionnaire
#   caractère spécial (X)HTML ou ' ': entité (X)HTML correspondante
HMLSPECIALCHARS_AND_SPACE = {' ': '&nbsp;',
                             '"': '&quot;',
                             '&': '&amp;',
                             '<': '&lt;',
                             '>': '&gt;'}


# Valeur pour l'argument newline lors de l'ouverture des fichiers en écriture
OUTPUT_NEWLINE = '\n'


# Regexp caractère spécial (X)HTML ou ' '
RE_HMLSPECIALCHARS_AND_SPACE = re.compile('( |"|&|<|>)')


# Nom de l'exécutable PHP
PHP = 'php'


# Extension utilisée pour les fichiers XHTML
XHTML_EXT = '.htm'


###################
# Fonction privée #
###################
def _class_if(c, t):
    """Si t alors renvoie ' class="..."',
    sinon renvoie ''

    Result: string"""
    return (' class="{0}"'.format(c) if t
            else '')


def _dict_append_value(d, k, v):
    """Si k est déjà une clé du dictionnaire d
    alors ajoute v à la liste correpondante à cette clé,
    sinon assigne [v] à cette clé

    Pre: d: dictionnaire dont les valeurs sont des listes
         k: clé pour ce dictionnaire
         v: valeur à ajouter pour cette clé
    Result: /"""
    if k in d:
        d[k].append(v)
    else:
        d[k] = [v]


def _href_attr(num, errors, warnings):
    """Renvoie dans une string les attributs et leurs valeurs pour les hyperliens
    en fonction du numéro de la ligne num,
    du dictionnaire des erreurs errors
    et du dictionnaire des avertissement warnings

    Pre: num: entier
         errors: dict: dictionnaire int: list de string
         warnings: dict: dictionnaire int: list de string
    Result: string"""
    seq = [' href="#l' + str(num) + '"']

    if num in errors:
        seq.append('class="error"')
    elif num in warnings:
        seq.append('class="warning"')

    title = ([htmlspecialchars_and_space(s) for s in errors[num]] if num in errors
             else [])
    if num in warnings:
        title.extend(warnings[num])
    if title:
        seq.append('title="' + '; \n'.join(title) + '"')

    return ' '.join(seq)


def _tt_title(encoding, t=True):
    """Si t alors renvoie '<tt title="...">...</tt>'
    sinon renvoie ''
    Pre: encoding: string parmi les clés de ENCODING_CHARSET
         t: bool
    Result: string"""
    return ('<tt title="{0}">{1}</tt>'.format(ENCODING_CHARSET[encoding], encoding) if t
            else '')


#############
# Fonctions #
#############
def help_msg():
    """Affiche le message d'aide sur la sortie des erreurs
    et termine le programme par un code d'erreur 1"""
    print("""PHPstripDEBUG
  Copyright (C) 2010, 2011, 2013, 2015, 2020 Olivier Pirson
  This program comes with ABSOLUTELY NO WARRANTY.
  This is free software, and you are welcome to redistribute it
  under certain conditions; see the source for copying conditions.

  olivier.pirson.opi@gmail.com
  DragonSoft DS --- http://www.opimedia.be/DS/
     v.{0}


PHPstripDEBUG [options] [files] ...

  Output PHP source with stripped
    #DEBUG...#DEBUG_END sections, comments and whitespace.

  Options:
    -E type   Encoding type of source file (default is 'latin_1').
    -d path   Path of directory destination
                (by default, don't build destination file)
    -D type   Encoding type of destination file
                (default is encoding type of source file).
    -h path   Path of directory XHTML
                (by default, don't build XHTML file)
    -H type   Encoding type of XHTML file
                (default is encoding type of destination file).
    -n        Don't run any commands, just print list of files.
    -q        Warning if question mark '???'.
    -r        Walking in subdirectories recursively.
    -s        Sort files.

    --help    Print this message on error output and exit by error code 1.

  Encoding type : 'cp1250', 'cp1251', 'cp1252',
    'iso8859-1', 'iso8859_2', 'iso8859_3', 'iso8859_4', 'iso8859_5',
    'iso8859_6', 'iso8859_7', 'iso8859_8', 'iso8859_9', 'iso8859_10',
    'iso8859_13', 'iso8859_15',
    'latin_1', 'utf_8', 'utf_16'""".format(VERSION), file=sys.stderr)
    exit(1)


def htmlspecialchars_and_space(s):
    """Renvoie s après avoir remplacé les caractères spéciaux (X)HTML ('\"', '&', '<', '>') ou ' '
    par les entités (X)HTML correspondantes
    Pre: s: string
    Result: string"""
    assert isinstance(s, str), type(s)

    return re.sub(RE_HMLSPECIALCHARS_AND_SPACE, lambda r: HMLSPECIALCHARS_AND_SPACE[r.group(1)], s)


def interactive_all_files(pathnames, dest_path, xhtml_path, xhtml_log_f=None,
                          src_encoding='latin_1', dest_encoding='latin_1', xhtml_encoding='latin_1',
                          warn_question_mark=False,
                          overwrite=None, really_execute=True,
                          print_file=False, print_dir=False, sort_files=False, recursive=False, _depth=0):
    """Applique read_strip_save() à chaque fichier de pathnames.
    Si recursive alors parcourt récursivement les répertoires de pathnames.

    Si dest_path != None
    alors les fichiers destinations sont créés et sauvés dans le répertoire dest_path.

    Si xhtml_path != None
    alors les fichiers XHTML sont créés et sauvés dans le répertoire xhtml_path.

    Si xhtml_log_f != None
    alors y écrit un compte-rendu dans un format XHTML sur les fichiers parcourus.

    Si warn_question_mark alors warning pour les '???'.

    Si overwrite == None alors chaque fichier (destination et XHTML) qui existe déjà demandera une confirmation manuelle,
    si overwrite vaut True alors tous les fichiers (destination et XHTML) qui existent déjà seront écrasés,
    si overwrite vaut False alors tous les fichiers (destination et XHTML) qui existent déjà seront conservés.

    Si really_execute alors exécute vraiment read_strip_save(),
    sinon se contente d'afficher les informations sur la sortie standard (sans créer ni les fichiers destinations, ni les fichiers XHTML)

    Si print_file alors affiche sur la sortie standard les fichiers traités.
    Si print_dir alors affiche sur la sortie standard les répertoires parcourus.

    Si sort_files alors commence par trier pathnames.

    Renvoie une paire (nombre de fichiers traités, nombre d'erreurs rencontrées,
                       total des tailles des fichiers sources rencontrés, total des tailles des fichiers destinations rencontrés)

    Pre: pathnames: Iterable de string
         dest_path: string ou None
         xhtml_path: string ou None
         xhtml_log_f: fichier texte ouvert en écriture ou None
         src_encoding: string parmi les clés de ENCODING_CHARSET
         dest_encoding: string parmi les clés de ENCODING_CHARSET
         xhtml_encoding: string parmi les clés de ENCODING_CHARSET
         warn_question_mark: bool
         overwrite: bool ou None
         really_execute: bool
         print_file: bool
         print_dir: bool
         sort_files: bool
         recursive: bool
         _depth: int >= 0
    Result: (int >= 0, int >= 0)"""
    assert isinstance(pathnames, collections.Iterable), type(pathnames)
    assert isinstance(dest_path, str) or (dest_path is None), type(dest_path)
    assert isinstance(xhtml_path, str) or (xhtml_path is None), type(xhtml_path)
    assert isinstance(src_encoding, str) and (src_encoding in ENCODING_CHARSET), (type(src_encoding), src_encoding)
    assert isinstance(dest_encoding, str) and (dest_encoding in ENCODING_CHARSET), (type(dest_encoding), dest_encoding)
    assert isinstance(xhtml_encoding, str) and (xhtml_encoding in ENCODING_CHARSET), (type(xhtml_encoding), xhtml_encoding)
    assert isinstance(_depth, int) and (_depth >= 0), type(_depth)

    nb_errors = 0  # nombre d'erreurs rencontrées
    nb_filenames = 0  # nombre de fichiers traités
    src_total_size = 0  # total des tailles des fichiers sources rencontrés
    dest_total_size = 0  # total des tailles des fichiers sources rencontrés

    print('<ul>', file=xhtml_log_f)
    for pathname in (sorted(pathnames) if sort_files
                     else pathnames):
        assert isinstance(pathname, str), type(pathname)

        if not pathname:
            continue
        pathname = os.path.normpath(pathname)

        if os.path.isfile(pathname):  # fichier
            if os.path.isabs(pathname):
                nb_errors += 1
                print('! \'{0}\' is a absolute pathname!'.format(pathname), file=sys.stderr)
            else:
                nb_filenames += 1
                dest_filename = (os.path.normpath(dest_path + '/' + pathname) if dest_path is not None
                                 else None)
                xhtml_filename = (os.path.normpath(xhtml_path + '/' + pathname + XHTML_EXT) if xhtml_path is not None
                                  else None)

                if really_execute:
                    if dest_filename is not None:
                        path = os.path.dirname(dest_filename)
                        if not os.path.exists(path):
                            try:
                                os.makedirs(path)
                                print('[Makedir \'{0}\']'.format(path))
                            except OSError:
                                nb_errors += 1
                                print('! Makedir \'{0}\' failed!'.format(path), file=sys.stderr)
                    if xhtml_filename is not None:
                        path = os.path.dirname(xhtml_filename)
                        if not os.path.exists(path):
                            try:
                                os.makedirs(path)
                                print('[Makedir \'{0}\']'.format(path))
                            except OSError:
                                nb_errors += 1
                                print('! Makedir \'{0}\' failed!'.format(path), file=sys.stderr)

                if really_execute:
                    if dest_filename and (not overwrite) and os.path.exists(dest_filename):  # faut demander si il faut écraser le fichier destination
                        if overwrite is None:
                            r = input('\n\'{0}\' file already exist. Overwrite? ([A]ll/[Y]es/[N]o or [Enter]/[S]kip all) >>>'
                                      .format(dest_filename)).upper()
                            while r not in ('A', 'Y', 'N', '', 'S'):
                                r = input("You must answer by 'A', 'Y', 'N', '' or 'S' >>>").upper()
                            if r == 'A':
                                overwrite = True
                            elif (r == 'N') or (r == ''):
                                dest_filename = None
                            elif r == 'S':
                                dest_filename = None
                                overwrite = False
                        else:                  # skip all
                            dest_filename = None
                    if xhtml_filename and (not overwrite) and os.path.exists(xhtml_filename):  # faut demander si il faut écraser le fichier XHTML
                        if overwrite is None:
                            r = input('\n\'{0}\' file already exist. Overwrite? ([A]ll/[Y]es/[N]o or [Enter]/[S]kip all) >>>'
                                      .format(xhtml_filename)).upper()
                            while r not in ('A', 'Y', 'N', '', 'S'):
                                r = input("You must answer by 'A', 'Y', 'N', '' or 'S' >>>").upper()
                            if r == 'A':
                                overwrite = True
                            elif (r == 'N') or (r == ''):
                                xhtml_filename = None
                            elif r == 'S':
                                xhtml_filename = None
                                overwrite = False
                        else:                  # skip all
                            xhtml_filename = None

                if print_file:
                    print(' '*_depth + pathname, '-> ', end='')
                if really_execute:
                    file_ok = False  # True si l'opération pour ce fichier se déroule normale, False sinon
                    nb_file_errors = None  # None si ko, sinon nombre d'erreurs dans le fichier
                    nb_file_warnings = None  # None si ko, sinon nombre de warnings dans le fichier
                    try:
                        file_ok, nb_file_errors, nb_file_warnings = read_strip_save(pathname, dest_filename=dest_filename,
                                                                                    xhtml_filename=xhtml_filename,
                                                                                    src_encoding=src_encoding, dest_encoding=dest_encoding,
                                                                                    xhtml_encoding=xhtml_encoding,
                                                                                    warn_question_mark=warn_question_mark)
                        if file_ok:
                            src_total_size += os.stat(pathname).st_size
                            dest_total_size += os.stat(dest_filename).st_size
                        else:
                            nb_errors += 1
                    except IOError:
                        nb_errors += 1
                        print('! File operation on \'{0}\' failed: {1}!'.format(pathname, e), file=sys.stderr)
                    except Exception as e:
                        nb_errors += 1
                        print('! Operation on \'{0}\' failed: {1}!'.format(pathname, e), file=sys.stderr)
                if print_file:
                    print((dest_filename if dest_filename is not None
                           else ''), '-',
                          (xhtml_filename if xhtml_filename is not None
                           else ''))

            if really_execute:
                print("""  <li>
    <span{0}>{1}</span>
    <span{2}>&nbsp;{3}</span>
    <span{4}>&nbsp;{5}</span>&rarr;""".format((_class_if('error', nb_file_errors or not file_ok) if _class_if('error', nb_file_errors or not file_ok)
                                               else _class_if('warning', nb_file_warnings)),
                                              ('<a href="{0}" type="text/plain" charset="{1}"><tt>{2}</tt></a>'.format(pathname.replace(os.sep, '/'),
                                                                                                                       ENCODING_CHARSET[src_encoding],
                                                                                                                       pathname) if pathname is not None
                                               else '<tt>' + pathname + '</tt>'),
                                              _class_if('error', nb_file_errors),
                                              (nb_file_errors if nb_file_errors
                                               else ''),
                                              _class_if('warning', nb_file_warnings),
                                              (nb_file_warnings if nb_file_warnings
                                               else '')),
                      file=xhtml_log_f)
            else:
                print("""  <li>
    <span>{0}</span>&rarr;""".format(('<a href="{0}" type="text/plain" charset="{1}"><tt>{2}</tt></a>'.format(pathname.replace(os.sep, '/'),
                                                                                                              ENCODING_CHARSET[src_encoding],
                                                                                                              pathname) if pathname is not None
                                      else '<tt>' + pathname + '</tt>')),
                      file=xhtml_log_f)
            if dest_filename is not None:
                if os.path.exists(dest_filename):
                    print('    <a href="{0}" type="text/plain" charset="{1}"><tt>{2}</tt></a>'.format(dest_filename.replace(os.sep, '/'),
                                                                                                      ENCODING_CHARSET[dest_encoding], dest_filename),
                          file=xhtml_log_f)
                else:
                    print('    <span class="error"><tt>{}</tt></span>'.format(dest_filename),
                          file=xhtml_log_f)
            else:
                print('    &ensp; ', file=xhtml_log_f)
            print('    &mdash;', file=xhtml_log_f)
            if xhtml_filename is not None:
                if os.path.exists(xhtml_filename):
                    print('    <a href="{0}">{1}</a>'.format(xhtml_filename.replace(os.sep, '/'), xhtml_filename),
                          file=xhtml_log_f)
                else:
                    print('    <span class="error">{}</span>'.format(xhtml_filename),
                          file=xhtml_log_f)
        else:                         # répertoire
            if print_dir:
                print(' '*_depth + '+' + pathname)
            if recursive:
                print("""  <li>
    <tt>{0}</tt>""".format(pathname), file=xhtml_log_f)
                pathname = pathname + os.sep + '*'
                r = interactive_all_files(glob.glob(pathname), dest_path, xhtml_path, xhtml_log_f=xhtml_log_f,
                                          src_encoding=src_encoding, dest_encoding=dest_encoding, xhtml_encoding=xhtml_encoding,
                                          warn_question_mark=warn_question_mark,
                                          overwrite=overwrite, really_execute=really_execute,
                                          print_file=print_file, print_dir=print_dir, sort_files=sort_files, recursive=recursive, _depth=_depth + 1)
                nb_filenames += r[0]
                nb_errors += r[1]
                src_total_size += r[2]
                dest_total_size += r[3]

        print('  </li>', file=xhtml_log_f)

    print('</ul>', file=xhtml_log_f)

    return (nb_filenames, nb_errors, src_total_size, dest_total_size)


def PHP_available():
    """Renvoie True si l'exécutable 'php' est accessible,
    False sinon

    Result: bool"""
    try:
        subprocess.Popen([PHP, '-v'], stdout=subprocess.PIPE, stderr=subprocess.PIPE)

        return True
    except:
        return False


def PHP_source_to_list(src_f, dest_encoding):
    """Renvoie dans une list les lignes du fichier PHP src_f
    après les avoir colorées syntaxiquement
    (l'exécutable 'php' doit être accessible dans le PATH,
    car c'est lui qui fait le boulot).
    dest_encoding est le nom du codage de caractères pour le résultat

    Pre: src_f: fichier texte ouvert en lecture
         dest_encoding: string
    Result: list de string ou None si erreur"""
    assert PHP_available()
    assert isinstance(dest_encoding, str), type(dest_encoding)

    tmp_f = tempfile.SpooledTemporaryFile(mode='w+', encoding=dest_encoding, newline=OUTPUT_NEWLINE, suffix='PHPstripDEBUG_tmp')

    p = subprocess.Popen([PHP, '-s'], stdin=src_f, stdout=tmp_f, universal_newlines=True)
    p.communicate()
    tmp_f.seek(0)

    return ''.join(tmp_f).split('<br />')


def PHP_strip_comment(src_f, dest_f):
    """Renvoie dans le fichier dest_f le fichier PHP src_f sans les commentaires
    et espaces superflus
    (l'exécutable 'php' doit être accessible dans le PATH,
    car c'est lui qui fait le boulot)

    Pre: src_f: fichier texte ouvert en lecture
         dest_f: fichier texte ouvert en écriture
    Result: /"""
    assert PHP_available()

    p = subprocess.Popen([PHP, '-w'], stdin=src_f, stdout=dest_f, universal_newlines=True)
    p.communicate()


def read_strip_save(src_filename, dest_filename=None, xhtml_filename=None,
                    src_encoding='latin_1', dest_encoding='latin_1', xhtml_encoding='latin_1',
                    warn_question_mark=False):
    """À partir du fichier source PHP src_filename,
    enlève les sections #DEBUG...#DEBUG_END, les commentaires et les espaces superflus
    puis sauve le résultat dans dest_filename (si != None).

    De plus si xhtml_filename != None
    alors y sauve une page XHTML qui représente le source de départ
    et les transformations ainsi que quelques informations.

    Renvoie un triplet
    (True si tout c'est passé correctement ou False si une erreur est survenue durant le processus,
    nombre d'"erreurs",
    nombre de warnings)

    (L'exécutable 'php' doit être accessible dans le PATH,
    car c'est lui qui fait une partie du boulot.
    Cette accessibilité n'est pas testée)

    Pre: src_filename: string
         dest_filename: string ou None
         xhtml_filename: string ou None
         src_encoding: string parmi les clés de ENCODING_CHARSET
         dest_encoding: string parmi les clés de ENCODING_CHARSET
         xhtml_encoding: string parmi les clés de ENCODING_CHARSET
         warn_question_mark: bool
    Result: (bool, int >= 0, int >= 0)"""
    assert PHP_available()
    assert isinstance(src_filename, str), type(src_filename)
    assert isinstance(dest_filename, str) or (dest_filename is None), type(dest_filename)
    assert isinstance(xhtml_filename, str) or (xhtml_filename is None), type(xhtml_filename)
    assert isinstance(src_encoding, str) and (src_encoding in ENCODING_CHARSET), (type(src_encoding), src_encoding)
    assert isinstance(dest_encoding, str) and (dest_encoding in ENCODING_CHARSET), (type(dest_encoding), dest_encoding)
    assert isinstance(xhtml_encoding, str) and (xhtml_encoding in ENCODING_CHARSET), (type(xhtml_encoding), xhtml_encoding)

    # Ouvre le fichier source en lecture
    try:
        src_f = open(src_filename, encoding=src_encoding)  # fichier source
    except IOError:
        print('! The file \'' + src_filename + '\' cannot be opened!', file=sys.stderr)
        return (False, 0, 0)

    # Ouvre deux fichiers temporaires lecture/écriture
    dest_stripped_f = tempfile.SpooledTemporaryFile(mode='w+', encoding=dest_encoding, newline=OUTPUT_NEWLINE,
                                                    suffix='PHPstripDEBUG_tmp')  # fichier sans #DEBUG...#DEBUG_END

    dest_deleted_f = tempfile.SpooledTemporaryFile(mode='w+', encoding=dest_encoding, newline=OUTPUT_NEWLINE,
                                                   suffix='PHPstripDEBUG_tmp')  # f. avec seulement les #DEBUG...#DEBUG_END

    # Enlève les #DEBUG...#DEBUG_END
    dest_deleted_nums, begin_PHP, errors_dict, warnings_dict = strip_debug(src_f, dest_stripped_f, dest_deleted_f)
    # liste des numéros de lignes enlevées,
    # liste des numéros de lignes contenant '<?php',
    # dictionnaire des numéros de ligne erronée
    # dictionnaire des numéro de ligne avec avertissement

    ok = True  # True si pas d'erreur rencontrée, False sinon

    if dest_filename is not None:
        # Construit le fichier destination : #DEBUG...#DEBUG_END, commentaires et espaces superflus enlevés
        dest_f = open(dest_filename, mode='w', encoding=dest_encoding, newline=OUTPUT_NEWLINE)
        dest_stripped_f.seek(0)
        try:
            PHP_strip_comment(dest_stripped_f, dest_f)
        except:
            ok = False
            dest_f.writelines(dest_stripped_f)
        dest_f.close()

    # Version colorée (code XHTML) de dest_stripped_f
    dest_stripped_f.seek(0)
    try:
        dest_stripped_list = PHP_source_to_list(dest_stripped_f, dest_encoding)  # liste des lignes les #DEBUG...#DEBUG_END enlevés et colorées
    except:
        ok = False
        dest_stripped_list = [htmlspecialchars_and_space(s[:-1] if s and (s[-1] == '\n')
                                                         else s) for s in dest_stripped_f]
    dest_stripped_f.close()

    if xhtml_filename is not None:
        # Construit le fichier XHTML
        xhtml_f = open(xhtml_filename, mode='w', encoding=xhtml_encoding, newline=OUTPUT_NEWLINE)

        print("""<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
<head>
  <title>{0} &ndash; PHPstripDEBUG v.{1}</title>
  <meta http-equiv="Content-Type" content="text/html; charset={2}" />

  <style type="text/css">
a {{
    color: inherit;
    text-decoration: none;
}}
a:hover {{ text-decoration: underline; }}
.ie7 a {{ color: black; }}
.ie6 a {{ color: black; }}

a.frame {{
    background-color: #f0f0f0;
    border: solid 1px;

    -moz-border-radius: 10px;
    -webkit-border-radius: 10px;
    border-radius: 10px;

    cursor: default;
    padding: 2px 5px;
    text-decoration: none;
}}

body,
code {{
    font-family: Dina, monospace;
    font-size: 100%;
    margin: 0px;
    padding: 0px;
}}

div {{
    margin: 0px;
    padding: 0px;
    position: absolute;
}}

div.buttons {{
    display: inline;
    position: relative;
    white-space: nowrap;
}}

div#deleted {{
    color: gray;
    left: {3}em;
    white-space: nowrap;
}}

div#linenumber {{
    background-color: {4};
    border-bottom: solid 1px black;
    border-right: solid 1px black;

    -moz-border-radius-bottomright: 10px;
    -webkit-border-bottom-right-radius: 10px;
    border-bottom-right-radius: 10px;

    left: -.4em;
    padding-bottom: 2px;
    text-align: right;
    white-space: nowrap;
    width: {3}em;
}}

div#linenumber a {{
    -moz-border-radius-bottomleft: 10px;
    -webkit-border-bottom-left-radius: 10px;
    border-bottom-left-radius: 10px;

    -moz-border-radius-topleft: 10px;
    -webkit-border-top-left-radius: 10px;
    border-top-left-radius: 10px;

    padding: 0px .2em 0px .4em;
}}

div#no-javascript {{
    color: red;
    position: relative;
}}


div#panel-infos {{
    background-color: {4};
    border-bottom: solid 1px;
    border-left: solid 1px;

    -moz-border-radius-bottomleft: 10px;
    -webkit-border-bottom-left-radius: 10px;
    border-bottom-left-radius: 10px;

    font-family: sans-serif;
    max-width: 37%;

    opacity: .8;

    padding-top: 5px;
    position: fixed;
    right: 0px;
    z-index: 11;
}}

div#panel-infos #button-next-PHP,
div#panel-infos #button-prev-PHP {{
    color: gray;
}}

div#panel-infos #button-fontFamily-switch,
div#panel-infos #button-transparency-switch {{
    background-color: #f0f0f0;
    border: inset 2px;
    color: gray;

    -moz-border-radius: 8px;
    -webkit-border-radius: 8px;
    border-radius: 8px;

    margin-top: 2px;
    padding: 1px 2px 2px;
}}
.ie6 div#panel-infos #button-fontFamily-switch,
.ie7 div#panel-infos #button-fontFamily-switch {{ width: 10ex; }}

.ie6 div#panel-infos #button-transparency-switch,
.ie7 div#panel-infos #button-transparency-switch {{
    margin-left: 5px;
    width: 15ex;
}}

div#panel-infos #button-src-switch {{
    background-color: #f0f0f0;
    border: outset 2px;
    color: gray;

    -moz-border-radius: 8px;
    -webkit-border-radius: 8px;
    border-radius: 8px;

    margin-top: 2px;
    padding: 1px 2px 2px;
}}

div#panel-infos div#panel-infos-about {{
    border-top: solid 1px black;
    font-size: smaller;
    padding: 5px 10px;
    position: relative;
    text-align: right;
}}

div#panel-infos div#panel-infos-begin-php {{
    border-top: solid 1px black;
    padding: 5px 10px;
    position: relative;
}}

div#panel-infos div#panel-infos-errors {{
    background-color: red;
    border-top: solid 1px black;
    font-weight: bold;
    padding: 5px 10px;
    position: relative;
}}

div#panel-infos div#panel-infos-infos {{
    border-top: solid 1px black;
    padding: 5px 10px;
    position: relative;
}}

div#panel-infos div#panel-infos-stripped {{
    border-top: solid 1px black;
    padding: 5px 10px;
    position: relative;
}}

div#panel-infos h1 {{
    display: inline;
    font-size: 100%;
    margin: 0px;
    padding: 0px 10px;
}}

div#panel-infos h2 {{
    font-size: 100%;
    font-weight: normal;
    margin: 0px;
    padding: 0px 10px 5px;
}}

div#panel-infos div#panel-infos-warnings {{
    background-color: orange;
    border-top: solid 1px black;
    font-weight: bold;
    padding: 5px 10px;
    position: relative;
}}

div#src {{
    color: transparent;
    left: {3}em;
    white-space: nowrap;
}}

div#stripped {{
    left: {3}em;
    white-space: nowrap;
}}

hr {{
    border-style: none;
    border-bottom: solid 1px gray;
}}

tt {{ font-family: Dina, monospace; }}

tt[title] {{ cursor: help; }}


.error {{
    background-color: red;
    color: black;
    position: relative;
}}

.warning {{
    background-color: orange;
    color: black;
    position: relative;
}}
  </style>
  <script type="text/javascript">/*<![CDATA[*/
var dina = true;  /* si true alors police de caractères Dina ou monospace, sinon directement monospace */
var nb_lines = {5};  /* nombre de lignes */
var num_line_functions = [{6}];  /* liste des numéros de ligne du source PHP débutant une fonction */
var src_show = false;  /* si true alors c'est le source original est affiché, sinon c'est la version stripped et colorée */
var transparency = true;  /* si true alors transparence sur le panneau d'informations, sinon pas */

function deleted_hide() {{
    document.getElementById('deleted').style.display = 'none';
}}

function deleted_show() {{
    if ( !src_show )
        document.getElementById('deleted').style.display = 'block';
}}

function fontFamily_switch() {{
    dina = !dina;
    document.getElementById('button-fontFamily-switch').style.borderStyle = (dina ? 'inset' : 'outset');

    var font = (dina
                ? 'Dina, monospace'
                : 'monospace');
    var a = document.getElementsByTagName('body');

    for (var t in a) {{
        if ( a[t].style != undefined )
            a[t].style.fontFamily = font;
    }}

    var a = document.getElementsByTagName('code');
    for (var t in a) {{
        if ( a[t].style != undefined )
            a[t].style.fontFamily = font;
    }}

    var a = document.getElementsByTagName('tt');
    for (var t in a) {{
        if ( a[t].style != undefined )
            a[t].style.fontFamily = font;
    }}
}}

function next_PHP() {{
    var n = parseInt(window.location.hash.substr(2), 10);

    if ( isNaN(n) )
        n = 1;

    for (var i=0; i<num_line_functions.length; i++) {{
        if ( num_line_functions[i] > n ) {{
            new_n = num_line_functions[i];
            window.location.hash = '#l' + new_n;
            return;
        }}
    }}
    window.location.hash = '#l' + nb_lines;
}}

function prev_PHP() {{
    var n = parseInt(window.location.hash.substr(2), 10);

    if ( isNaN(n) )
        n = 1;

    for (var i=num_line_functions.length-1; i>=0; i--) {{
        if ( num_line_functions[i] < n ) {{
            new_n = num_line_functions[i];
            window.location.hash = '#l' + new_n;
            return;
        }}
    }}
    window.location.hash = '#l1';
}}

function src_switch() {{
    src_show = !src_show;
    document.getElementById('button-src-switch').style.borderStyle = (src_show ? 'inset' : 'outset');

    document.getElementById('src').style.color = (src_show ? 'gray' : 'transparent');
    document.getElementById('deleted').style.display = (src_show ? 'none' : 'block');
    document.getElementById('stripped').style.display = (src_show ? 'none' : 'block');
}}

function start() {{
    document.getElementById('no-javascript').style.display = 'none';

    document.getElementById('button-fontFamily-switch').disabled = undefined;
    document.getElementById('button-fontFamily-switch').style.color = 'black';

    document.getElementById('button-src-switch').disabled = undefined;
    document.getElementById('button-src-switch').style.color = 'black';

    document.getElementById('button-next-PHP').style.color = 'black';
    document.getElementById('button-prev-PHP').style.color = 'black';

    document.getElementById('button-transparency-switch').disabled = undefined;
    document.getElementById('button-transparency-switch').style.color = 'black';
    document.getElementById('panel-infos').style.opacity = .8;
}}

function transparency_switch() {{
    transparency = !transparency;
    document.getElementById('button-transparency-switch').style.borderStyle = (transparency ? 'inset' : 'outset');
    document.getElementById('panel-infos').style.opacity = (transparency ? .8 : 1);
}}
  /*]]>*/</script>
</head>
<!--[if lt IE 7]> <body class="ie ie6" onload="start();"> <![endif]-->
<!--[if IE 7]>    <body class="ie ie7" onload="start();"> <![endif]-->
<!--[if IE 8]>    <body class="ie ie8" onload="start();"> <![endif]-->
<!--[if IE 9]>    <body class="ie ie9" onload="start();"> <![endif]-->
<!--[if gt IE 9]> <body class="ie"     onload="start();"> <![endif]-->
<!--[if !IE]>     --> <body onload="start();"> <!--       <![endif]-->
""".format(src_filename, VERSION.replace('---', '&ndash;'),
           ENCODING_CHARSET[xhtml_encoding], len(str(len(dest_stripped_list))), '#dbe0f2',
           len(dest_stripped_list), ','.join([str(i) for i in begin_PHP])), file=xhtml_f)

        # Regexp 'debug' ou 'assert(' ou '???' (si warn_question_mark)
        RE_WARN_DEBUG = re.compile(r'(.*?)(debug|assert(?=</span><span style="color: #007700">\()|\?\?\?)(.*)' if warn_question_mark
                                   else r'(.*?)(debug|assert(?=</span><span style="color: #007700">\())(.*)',
                                   re.I)

        RE_WARN_ENDWHITE = re.compile(r'(.*?)(((?:&nbsp;)|\s)+)$', re.I)  # regexp "espace" en fin de ligne

        print("""<!--[if lt IE 7]>
  <div class="error" style="padding:1ex">
    Vous utilisez une version d&rsquo;Internet Explorer dépassée.
    Pour obtenir un résultat adéquat veuillez effectuer une mise à jour,
    ou utiliser un vrai navigateur tel que
    <a href="https://www.mozilla.org/fr/firefox/" style="color:blue; text-decoration:underline">Firefox</a>.
  </div>
<![endif]-->
""", file=xhtml_f)
        print('<div id="deleted">', file=xhtml_f)
        i = 0
        dest_deleted_f.seek(0)
        for s in dest_deleted_f:
            i += 1
            s = htmlspecialchars_and_space(s[:-1] if s and (s[-1] == '\n')
                                           else s)

            r = RE_WARN_ENDWHITE.match(s)
            if r:
                _dict_append_value(warnings_dict, i, 'White character in tail of line!')
                s = r.group(1) + '<span class="warning">' + r.group(2) + '</span>'

            print(s + '<br />', file=xhtml_f)
        print('</div>', file=xhtml_f)

        print('<div id="stripped">', file=xhtml_f)
        i = 0
        for s in dest_stripped_list:
            i += 1

            r = RE_WARN_DEBUG.match(s)
            if r:
                _dict_append_value(warnings_dict, i, 'Debug information not stripped!')
                s = r.group(1) + '<span class="warning">' + r.group(2) + '</span>' + r.group(3)

            r = RE_WARN_ENDWHITE.match(s)
            if r:
                _dict_append_value(warnings_dict, i, 'White character in tail of line!')
                if r.group(1) != 'assert</span><span style="color: #007700">(':
                    s = r.group(1) + '<span class="warning">' + r.group(2) + '</span>'
                else:
                    s = r.group(1) + '<span class="warning">' + r.group(2) + '</span>'

            print(s + '<br />', file=xhtml_f)
        print('</div>', file=xhtml_f)

        print('<div id="src">', file=xhtml_f)
        src_f.seek(0)
        for s in src_f:
            print(htmlspecialchars_and_space(s) + '<br />', file=xhtml_f)
        print('</div>', file=xhtml_f)

        print('<div id="linenumber" onmouseover="deleted_hide();" onmouseout="deleted_show();">', file=xhtml_f)
        for i in range(0, len(dest_stripped_list)):
            if i < len(dest_stripped_list):
                i += 1
                print('<a id="l' + str(i) + '"' + _href_attr(i, errors_dict, warnings_dict) + '>' + str(i) + '</a><br />', end=('\n' if not i % 100
                                                                                                                                else ''), file=xhtml_f)
            else:
                print('<br />', end='', file=xhtml_f)
        print('</div>', file=xhtml_f)

        src_size = os.stat(src_filename).st_size  # taille (en octets) du fichier source
        print("""  <div id="panel-infos" xml:lang="en">
    <h1>PHPstripDEBUG</h1>
    <div class="buttons">
      <a{0} class="frame" tabindex="11" title="To first line.">&laquo;</a>
      <a id="button-prev-PHP" href="#" class="frame" tabindex="12" title="To previous '&lt;?php'." onclick="prev_PHP(); return false;">&lt;</a>
      <a id="button-next-PHP" href="#" class="frame" tabindex="13" title="To next '&lt;?php'." onclick="next_PHP(); return false;">&gt;</a>
      <a{1} class="frame" tabindex="14" title="To last line.">&raquo;</a>
      &ensp;
      <button id="button-fontFamily-switch" disabled="disabled" tabindex="15" title="Switch Dina (if available)/monospace font."
              onclick="fontFamily_switch();">Dina font</button>
      <button id="button-transparency-switch" disabled="disabled" tabindex="16" title="Transparency on (if available)/off."
              onclick="transparency_switch();">Transparency</button>
    </div>
    <h2>
      <button id="button-src-switch" disabled="disabled" tabindex="1" title="Switch stripped source/complete source."
              onclick="src_switch();"><tt>{2}</tt></button> {3}&nbsp;b
    </h2>""".format(_href_attr(1, {}, {}), _href_attr(len(dest_stripped_list), {}, {}), src_filename, src_size), file=xhtml_f)

        if errors_dict:
            print("""    <div id="panel-infos-errors">
      Errors:<br />
      <small>""" + '\n'.join(['<a' + _href_attr(k, errors_dict, {}) + '>' + str(k) + '</a>' for k in sorted(errors_dict)]) + """</small>
    </div>""", file=xhtml_f)

        if warnings_dict:
            print("""    <div id="panel-infos-warnings">
      Warnings:<br />
      <small>""" + '\n'.join(['<a' + _href_attr(k, {}, warnings_dict) + '>' + str(k) + '</a>' for k in sorted(warnings_dict.keys())]) + """</small>
    </div>""", file=xhtml_f)

        if dest_deleted_nums:
            print("""      <div id="panel-infos-stripped">
        Stripped:<br />
        <small>""" + '\n'.join(['<a' + _href_attr(i, {}, {}) + '>' + str(i) + '</a>' for i in dest_deleted_nums]) + """</small>
      </div>""", file=xhtml_f)

        if begin_PHP:
            print("""      <div id="panel-infos-begin-php">
        '&lt;?php':<br />
        <small>""" + '\n'.join(['<a' + _href_attr(i, {}, {}) + '>' + str(i) + '</a>' for i in begin_PHP]) + """</small>
      </div>""", file=xhtml_f)

        print('    <div id="panel-infos-infos">', file=xhtml_f)
        if dest_filename is not None:
            dest_size = os.stat(dest_filename).st_size
            percentage = (dest_size*100/src_size if src_size
                          else 100)
            percentage = (str(percentage) if round(percentage) == percentage
                          else '&cong;' + str(round(percentage)))
            print("""      <tt>{0}</tt> {1}&nbsp;b&nbsp;({2}%)<br />""".format(dest_filename,
                                                                               dest_size, percentage),
                  file=xhtml_f)
        print("""      source:&nbsp;{0} &nbsp; destination:&nbsp;{1} &nbsp; XHTML:&nbsp;{2}"""
              .format(_tt_title(src_encoding),
                      _tt_title(dest_encoding, dest_filename is not None),
                      _tt_title(xhtml_encoding, xhtml_filename is not None)),
              file=xhtml_f)
        print("""      <div id="no-javascript">JavaScript is disabled!</div>
    </div>
    <div id="panel-infos-about">
      &copy;&nbsp;<a href="http://www.opimedia.be/">Olivier&nbsp;Pirson</a>
      &mdash;&nbsp;<a href="http://www.opimedia.be/DS/">DragonSoft&nbsp;DS</a>
      &mdash;&nbsp;v.{0}<br />
      <small><small>
        This program comes with <strong>ABSOLUTELY NO WARRANTY</strong>.
        This is <strong>free software</strong>, and you are welcome to <strong>redistribute it</strong> under certain conditions;
        <strong>see the source</strong> for copying conditions.
      </small></small>
    </div>
  </div>
</body>
</html>""".format(VERSION.replace(' ', '&nbsp;').replace('&nbsp;---', ' &mdash;')), file=xhtml_f)

        xhtml_f.close()

    src_f.close()
    dest_deleted_f.close()

    return (ok, len(errors_dict), len(warnings_dict))


def strip_debug(src, dest_f, dest_deleted_f):
    """Renvoie dans le fichier dest_f le fichier PHP src
    tel que les lignes comprises entre un marqueur d'ouverture PHP '<?php'
    et un marqueur de fermeture '?>'
    et qui sont aussi comprises entre '#DEBUG' et '#DEBUG_END'
    sont remplacées par des lignes vides.

    Les lignes ainsi remplacées sont renvoyées dans le fichier dest_deleted_f.

    La fonction renvoie une liste des numéros des lignes '#DEBUG'
    une liste des numéros des lignes contenant '<?php',
    un dictionnaire des lignes erronnées
    et un dictionnaire des lignes avec avertissement

    Pre: src: Iterable de string
         dest_f: fichier texte ouvert en écriture
         dest_deleted_f: fichier texte ouvert en écriture
    Result: (list strictement ordonnée de int,
             list strictement ordonnée de int,
             dictionnaire int: list de string,
             dictionnaire int: list de string)"""
    assert isinstance(src, collections.Iterable), type(src)

    RE_DEBUG = re.compile(r'(\s*)(#DEBUG)(_END)?')  # regexp '#DEBUG' ou '#DEBUG_END'
    RE_PHP = re.compile(r'(.*?)(<\?php|\?>)(.*)')  # regexp d'ouverture '<?php' ou de fermeture '?>' des parties PHP

    in_php = False  # True si position courante entre '<?php' et '?>', False sinon
    in_debug = False  # False si position courante entre '#DEBUG' et '#DEBUG_END', False sinon

    dest_deleted_nums = []  # liste des numéros de lignes '#DEBUG'
    errors_dict = {}  # dictionnaire : numéro de ligne erronée: list de messages d'erreur
    warnings_dict = {}  # dictionnaire : numéro de ligne avec un avertissement: list de messages d'avertissement
    begin_PHP_dict = {}  # dictionnaire : numéro de ligne contentant '<?php': None
    begin_end_PHP_already = False  # True si rencontré au-moins un '<?php'...'?>' correct

    num = 0  # numéro de la ligne
    for s in src:
        num += 1

        if s and (s[-1] == '\n'):
            s = s[:-1]
            end = '\n'  # fin de la ligne
        else:
            end = ''

        while s:  # tant qu'il reste un morceau de la ligne non-analysée
            if in_php:
                r = RE_DEBUG.match(s)
                if r:
                    if r.group(3):  # '(\s*)(#DEBUG)(_END)' : fin d'une partie à supprimée
                        dest_s = ''
                        dest_deleted_s = s
                        if in_debug:
                            in_debug = False
                        else:         # on est déjà hors d'une partie à supprimée
                            _dict_append_value(errors_dict, num, 'Already out #DEBUG...#DEBUG_END section!')
                    else:               # '(\s*)(#DEBUG)' : début d'une partie à supprimée
                        dest_s = ''
                        dest_deleted_s = s
                        if in_debug:  # on est déjà dans une partie à supprimée
                            _dict_append_value(errors_dict, num, 'Already in #DEBUG...#DEBUG_END section!')
                        else:
                            in_debug = True
                        dest_deleted_nums.append(num)
                    s = ''
            if s:
                r = RE_PHP.match(s)
                if r:
                    dest_s = r.group(1) + r.group(2)
                    s = r.group(3)
                    if r.group(2) == '?>':  # '(.*?)(\?>)(.*)' : fin d'une partie PHP
                        if in_php:
                            in_php = False
                            begin_end_PHP_already = True
                        else:       # on est déjà hors d'une partie PHP
                            _dict_append_value(errors_dict, num, 'Already out <?php...?> section!')
                    else:                   # '(.*?)(<\?php)(.*)' : début d'une partie PHP
                        begin_PHP_dict[num] = None
                        if in_php:  # on est déjà dans une partie PHP
                            _dict_append_value(errors_dict, num, 'Already in <?php...?> section!')
                        else:
                            in_php = True
                else:
                    dest_s = s
                    s = ''
                if in_debug:
                    dest_s, dest_deleted_s = '', dest_s
                else:
                    dest_deleted_s = ''

            print(dest_s, end='', file=dest_f)
            print(dest_deleted_s, end='', file=dest_deleted_f)

        print(end=end, file=dest_f)
        print(end=end, file=dest_deleted_f)

    if not begin_end_PHP_already:
        _dict_append_value(warnings_dict, 1, 'No correct <?php...?> in file!')

    return (dest_deleted_nums, sorted(begin_PHP_dict.keys()), errors_dict, warnings_dict)


########
# Main #
########
if __name__ == '__main__':
    if not PHP_available():
        print('! \'{0}\' is unavailable in the PATH!'.format(PHP), file=sys.stderr)
        exit(1)

    if len(sys.argv) <= 1:
        help_msg()

    dest_path = None  # répertoire pour les fichiers destination
    xhtml_path = None  # répertoire pour les fichiers XHTML

    src_encoding = 'latin_1'  # codec du fichier source
    dest_encoding = None  # codec du fichier destination
    xhtml_encoding = None  # codec du fichier XHTML

    really_execute = True  # si True alors exécute vraiment, si False alors ne fait qu'afficher la liste des fichiers
    recursive = False  # si True alors parcourt récursif
    sort_files = False  # si True alors pathnames sera trié
    warn_question_mark = False  # si True alors warning pour les '???'

    pathnames = []  # liste des noms de fichiers

    # Parcourt les paramètres de la ligne de commande
    i = 1
    while i < len(sys.argv):
        s = sys.argv[i]
        if s and (s[0] == '-'):  # option
            if s == '-E':
                i += 1
                if i >= len(sys.argv):
                    print('! -E option require an encoding type!', file=sys.stderr)
                elif sys.argv[i] not in ENCODING_CHARSET:
                    print('! -E option encoding type \'{0}\' don\'t exist!'.format(sys.argv[i]), file=sys.stderr)
                else:
                    src_encoding = sys.argv[i]
            elif s == '-D':
                i += 1
                if i >= len(sys.argv):
                    print('! -D option require an encoding type!', file=sys.stderr)
                elif sys.argv[i] not in ENCODING_CHARSET:
                    print('! -D option encoding type \'{0}\' don\'t exist!'.format(sys.argv[i]), file=sys.stderr)
                else:
                    dest_encoding = sys.argv[i]
            elif s == '-H':
                i += 1
                if i >= len(sys.argv):
                    print('! -H option require an encoding type!', file=sys.stderr)
                elif sys.argv[i] not in ENCODING_CHARSET:
                    print('! -H option encoding type \'{0}\' don\'t exist!'.format(sys.argv[i]), file=sys.stderr)
                else:
                    xhtml_encoding = sys.argv[i]
            elif s == '-d':
                i += 1
                if i >= len(sys.argv):
                    print('! -d option require an directory!', file=sys.stderr)
                else:
                    dest_path = sys.argv[i]
            elif s == '-h':
                i += 1
                if i >= len(sys.argv):
                    print('! -d option require an directory!', file=sys.stderr)
                else:
                    xhtml_path = sys.argv[i]
            elif s == '-n':
                really_execute = False
            elif s == '-q':
                warn_question_mark = True
            elif s == '-r':
                recursive = True
            elif s == '-s':
                sort_files = True
            else:
                help_msg()
        else:                    # fichier(s)
            t = os.path.expanduser(os.path.expandvars(s))
            t = glob.glob(t)
            if not t:
                print('! Argument {0} \'{1}\' don\'t exist!'.format(i, s), file=sys.stderr)
            pathnames.extend(t)
        i += 1

    if not dest_encoding:
        dest_encoding = src_encoding
    if not xhtml_encoding:
        xhtml_encoding = dest_encoding

    xhtml_log_f = open('PHPstripDEBUG_log' + XHTML_EXT, mode='w', encoding=xhtml_encoding, newline=OUTPUT_NEWLINE)
    print("""<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
  <head>
    <title>log &ndash; PHPstripDEBUG v.{0}</title>
      <style type="text/css">
ul {{
    margin: 0px;
    padding-left: 2ex;
}}

li {{
    list-style-position: outside;
    margin: 0px;
    padding: 0px;
}}

span {{ padding: 0px 3px; }}

tt {{ font-family: Dina, monospace; }}

.error {{ background-color: red; }}
.warning {{ background-color: orange; }}
      </style>
  </head>
  <body>
    <div>
      <tt>{1}</tt>
      <hr />""".format(VERSION.replace('---', '&ndash;'), ' '.join(sys.argv)), file=xhtml_log_f)

    nb_files, nb_errors, src_total_size, dest_total_size = interactive_all_files(pathnames, dest_path, xhtml_path, xhtml_log_f=xhtml_log_f,
                                                                                 src_encoding=src_encoding, dest_encoding=dest_encoding,
                                                                                 xhtml_encoding=xhtml_encoding,
                                                                                 warn_question_mark=warn_question_mark,
                                                                                 overwrite=True, really_execute=really_execute,
                                                                                 print_file=True, print_dir=True, sort_files=sort_files,
                                                                                 recursive=recursive)
    percentage = (dest_total_size*100/src_total_size if src_total_size
                  else 100)
    percentage = (str(percentage) if round(percentage) == percentage
                  else '&cong;' + str(round(percentage)))
    print("""      <hr />
      Number of files: {0} &ensp; <span{1}>Number failed: {2}</span><br />
      Total source size: {3}&nbsp;b &ensp; Total destination size: {4}&nbsp;b ({5}%)
    </div>
  </body>
</html>""".format(nb_files, _class_if('error', nb_errors), nb_errors, src_total_size, dest_total_size, percentage), file=xhtml_log_f)
    print("""---
Nb files: {0}   Nb failed: {1}""".format(nb_files, nb_errors))
