/* -*- coding: utf-8 -*- */

/*
  txt2.js

  (c) Olivier Pirson --- http://www.opimedia.be/ --- olivier_pirson_opi@yahoo.fr
  Débuté le 15 octobre 2011
  v.01.00.00 - 15 octobre 2012 (Happy birthday! :-)) : Première version disponible offline.
  v.01.00.01 - 31 octobre 2012 : txt2_replace() : ignore les caractères blancs extérieures dans pattern si de la forme "regexp".
  v.01.00.02 - 30 décembre 2012 : Agrandit les textarea.
  v.01.01.00 - 9 février 2013 :
    - Remplacé la fonction d'initialisation par une fonction anonyme appelée directement.
    - txt2_replace_set_spaces() : remplacé "\s" par "\s+".
    - txt2_trim() remplacée par par une regexp : txt2_replace_set_trim().
    - trim() pour Internet Explorer : modifiée.
    - Sauve dans une variable les .length utilisées dans une boucle
    - Remplacé le :hover sur les boutons Entities et Characters par l'usage d'un clic.
    - Style CSS cursor: pointer sur les boutons.
  v.01.01.01 - 4 mars 2013 :
    - Ajouté conversion entre base dans l'autre sens et adapté txt2_base_to_base().
    - Ajouté liste de sélection de la police de caractères et de sa taille pour les deux zones de saisie.
  v.01.01.02 - 9 avril 2013 :
    - Ajouté des flèches dans le panneau Statistics indiquant ce qui est trié.
    - Ajouté liste de sélection de langages afin d'utiliser le correcteur orthographique dans les deux zones de saisie.
    - Corrigé la construction de la <table> dans txt2_panel_statistics_update() pour qu'elle tourne sous IE.
  v.01.02.00 - 31 décembre 2014 :
    - Adapté liste des caractères à la police de caractères sélectionnée.
    - Ajouté quelques caractères dans la liste des caractères.
    - txt2_add_char() et txt2_add_entity() ajoute maintenant à la position courante plutôt qu'à la fin.
    - Ajouté mode strict dans chaque fonction JavaScript.
    - Changé comparaisons avec undefined.
    - Corrigé bug des polices de caractères sans-serif non disponibles.
    - Corrigé variables statiques de txt2_random_password().
*/

/*
  GPLv3
  ------
  Copyright (C) 2012, 2013, 2014 Olivier Pirson

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

/*
  Dépendances
  ------------
  * Internet Explorer réclame classList.min.js qui définit la méthode classList :
      https://github.com/eligrey/classList.js

  * Fonctions utilisées de la bibliothèque jsPHP (par catégories) : http://jsphp.co/
      Strings :
        addslashes()
        crc32()
        html_entity_decode()
        htmlentities()
        htmlspecialchars()
        md5()
        quotemeta()
        sha1()
        str_shuffle()
        strip_tags()
        stripslashes()
      URLs :
        rawurldecode()
        rawurlencode()
      XML Parser :
        utf8_decode()
        utf8_encode()
*/

/*************
 * Variables *
 *************/
var txt2_cmp_infos = document.getElementById('txt2-cmp-infos');        // résultats des comparaisons
var txt2_regexp_infos = document.getElementById('txt2-regexp-infos');  // l'expression régulière utilisée par un remplacement

var txt2_dest = document.getElementById('txt2-dest');
var txt2_dest_text = document.getElementById('txt2-dest-text');  // champ de saisie contenant le texte transformé
var txt2_dest_infos = document.getElementById('txt2-dest-infos');
var txt2_dest_infos_chars = document.getElementById('txt2-dest-infos-chars');
var txt2_dest_infos_length = document.getElementById('txt2-dest-infos-length');

var txt2_last_action = null;  // mémorise la dernière action pour la répéter
var txt2_last_input = document.getElementById('alphanormalize');  // mémorise le dernier champ input qui est mis en surbrillance

var txt2_panel_statistics_sorted = 1;  // Statistiques non triées si 0, triées par code de caractère si 1, triée par nombre d'occurences si 2
var txt2_panel_statistics_sorted_invert = false;  // Statistiques triées par ordre décroissant

var txt2_src = document.getElementById('txt2-src');
var txt2_src_text = document.getElementById('txt2-src-text');  // champ de saisie contenant le texte de départ
var txt2_src_infos = document.getElementById('txt2-src-infos');
var txt2_src_infos_length = document.getElementById('txt2-src-infos-length');

/*
  Table associative : caractère avec accent, cédille... ou ligature: caractère alphabétique correspondant.
  Par ex. : 'é': 'e'.
*/
var ACCENTALPHA_TO_ALPHA = {
    'À': 'A', 'Á': 'A', 'Â': 'A', 'Ã': 'A', 'Ä': 'A', 'Å': 'A', 'Æ': 'AE', 'Ç': 'C', 'È': 'E', 'É': 'E', 'Ê': 'E', 'Ë': 'E', 'Ì': 'I', 'Í': 'I', 'Î': 'I', 'Ï': 'I', 'Ñ': 'N', 'Ò': 'O', 'Ó': 'O', 'Ô': 'O', 'Õ': 'O', 'Ö': 'O', 'Ø': 'O', 'Ù': 'U', 'Ú': 'U', 'Û': 'U', 'Ü': 'U', 'Ý': 'Y', 'ß': 'sz', 'à': 'a', 'á': 'a', 'â': 'a', 'ã': 'a', 'ä': 'a', 'å': 'a', 'æ': 'ae', 'ç': 'c', 'è': 'e', 'é': 'e', 'ê': 'e', 'ë': 'e', 'ì': 'i', 'í': 'i', 'î': 'i', 'ï': 'i', 'ñ': 'n', 'ò': 'o', 'ó': 'o', 'ô': 'o', 'õ': 'o', 'ö': 'o', 'ø': 'o', 'ù': 'u', 'ú': 'u', 'û': 'u', 'ü': 'u', 'ý': 'y', 'ÿ': 'y', 'Œ': 'OE', 'œ': 'oe', 'Š': 'S', 'š': 's', 'Ÿ': 'Y'
};

/*
  Table associative : lettre grecque: caractère alphabétique romain correspondant.
  Par ex. : caractère de la lettre grecque epsilon: 'e'.

  Adopte la norme ONU/ELOT : cf. http://www.opimedia.be/DS/mementos/grecs.htm
*/
var GREEK_TO_ALPHA = {
    'Α': 'A', 'Β': 'V', 'Γ': 'G', 'Δ': 'D', 'Ε': 'E', 'Ζ': 'Z', 'Η': 'I', 'Θ': 'TH', 'Ι': 'I', 'Κ': 'K', 'Λ': 'L', 'Μ': 'M', 'Ν': 'N', 'Ξ': 'X', 'Ο': 'O', 'Π': 'P', 'Ρ': 'R', 'Σ': 'S', 'Τ': 'T', 'Υ': 'Y', 'Φ': 'F', 'Χ': 'CH', 'Ψ': 'PS', 'Ω': 'O', 'α': 'a', 'β': 'v', 'γ': 'g', 'δ': 'd', 'ε': 'e', 'ζ': 'z', 'η': 'i', 'θ': 'th', 'ι': 'i', 'κ': 'k', 'λ': 'l', 'μ': 'm', 'ν': 'n', 'ξ': 'x', 'ο': 'o', 'π': 'p', 'ρ': 'r', 'ς': 's', 'σ': 's', 'τ': 't', 'υ': 'y', 'φ': 'f', 'χ': 'ch', 'ψ': 'ps', 'ω': 'o', 'ϑ': 'th', 'ϒ': 'y', 'ϖ': 'p'
};

/*
  Table associative : code d'un caractère: nom de l'entitité XHTML correspondante.
  Par ex. : 233: 'eacute'.
  (Cf. http://docs.python.org/py3k/library/html.entities.html#module-html.entities
  "A dictionary that maps Unicode codepoints to HTML entity names.")
*/
var HTML_ENTITIES_CODEPOINT2NAME = {
    34: 'quot', 38: 'amp', 60: 'lt', 62: 'gt', 160: 'nbsp', 161: 'iexcl', 162: 'cent', 163: 'pound', 164: 'curren', 165: 'yen', 166: 'brvbar', 167: 'sect', 168: 'uml', 169: 'copy', 170: 'ordf', 171: 'laquo', 172: 'not', 173: 'shy', 174: 'reg', 175: 'macr', 176: 'deg', 177: 'plusmn', 178: 'sup2', 179: 'sup3', 180: 'acute', 181: 'micro', 182: 'para', 183: 'middot', 184: 'cedil', 185: 'sup1', 186: 'ordm', 187: 'raquo', 188: 'frac14', 189: 'frac12', 190: 'frac34', 191: 'iquest', 192: 'Agrave', 193: 'Aacute', 194: 'Acirc', 195: 'Atilde', 196: 'Auml', 197: 'Aring', 198: 'AElig', 199: 'Ccedil', 200: 'Egrave', 201: 'Eacute', 202: 'Ecirc', 203: 'Euml', 204: 'Igrave', 205: 'Iacute', 206: 'Icirc', 207: 'Iuml', 208: 'ETH', 209: 'Ntilde', 210: 'Ograve', 211: 'Oacute', 212: 'Ocirc', 213: 'Otilde', 214: 'Ouml', 215: 'times', 216: 'Oslash', 217: 'Ugrave', 218: 'Uacute', 219: 'Ucirc', 220: 'Uuml', 221: 'Yacute', 222: 'THORN', 223: 'szlig', 224: 'agrave', 225: 'aacute', 226: 'acirc', 227: 'atilde', 228: 'auml', 229: 'aring', 230: 'aelig', 231: 'ccedil', 232: 'egrave', 233: 'eacute', 234: 'ecirc', 235: 'euml', 236: 'igrave', 237: 'iacute', 238: 'icirc', 239: 'iuml', 240: 'eth', 241: 'ntilde', 242: 'ograve', 243: 'oacute', 244: 'ocirc', 245: 'otilde', 246: 'ouml', 247: 'divide', 248: 'oslash', 249: 'ugrave', 250: 'uacute', 251: 'ucirc', 252: 'uuml', 253: 'yacute', 254: 'thorn', 255: 'yuml', 338: 'OElig', 339: 'oelig', 352: 'Scaron', 353: 'scaron', 376: 'Yuml', 402: 'fnof', 710: 'circ', 732: 'tilde', 913: 'Alpha', 914: 'Beta', 915: 'Gamma', 916: 'Delta', 917: 'Epsilon', 918: 'Zeta', 919: 'Eta', 920: 'Theta', 921: 'Iota', 922: 'Kappa', 923: 'Lambda', 924: 'Mu', 925: 'Nu', 926: 'Xi', 927: 'Omicron', 928: 'Pi', 929: 'Rho', 931: 'Sigma', 932: 'Tau', 933: 'Upsilon', 934: 'Phi', 935: 'Chi', 936: 'Psi', 937: 'Omega', 945: 'alpha', 946: 'beta', 947: 'gamma', 948: 'delta', 949: 'epsilon', 950: 'zeta', 951: 'eta', 952: 'theta', 953: 'iota', 954: 'kappa', 955: 'lambda', 956: 'mu', 957: 'nu', 958: 'xi', 959: 'omicron', 960: 'pi', 961: 'rho', 962: 'sigmaf', 963: 'sigma', 964: 'tau', 965: 'upsilon', 966: 'phi', 967: 'chi', 968: 'psi', 969: 'omega', 977: 'thetasym', 978: 'upsih', 982: 'piv', 8194: 'ensp', 8195: 'emsp', 8201: 'thinsp', 8204: 'zwnj', 8205: 'zwj', 8206: 'lrm', 8207: 'rlm', 8211: 'ndash', 8212: 'mdash', 8216: 'lsquo', 8217: 'rsquo', 8218: 'sbquo', 8220: 'ldquo', 8221: 'rdquo', 8222: 'bdquo', 8224: 'dagger', 8225: 'Dagger', 8226: 'bull', 8230: 'hellip', 8240: 'permil', 8242: 'prime', 8243: 'Prime', 8249: 'lsaquo', 8250: 'rsaquo', 8254: 'oline', 8260: 'frasl', 8364: 'euro', 8465: 'image', 8472: 'weierp', 8476: 'real', 8482: 'trade', 8501: 'alefsym', 8592: 'larr', 8593: 'uarr', 8594: 'rarr', 8595: 'darr', 8596: 'harr', 8629: 'crarr', 8656: 'lArr', 8657: 'uArr', 8658: 'rArr', 8659: 'dArr', 8660: 'hArr', 8704: 'forall', 8706: 'part', 8707: 'exist', 8709: 'empty', 8711: 'nabla', 8712: 'isin', 8713: 'notin', 8715: 'ni', 8719: 'prod', 8721: 'sum', 8722: 'minus', 8727: 'lowast', 8730: 'radic', 8733: 'prop', 8734: 'infin', 8736: 'ang', 8743: 'and', 8744: 'or', 8745: 'cap', 8746: 'cup', 8747: 'int', 8756: 'there4', 8764: 'sim', 8773: 'cong', 8776: 'asymp', 8800: 'ne', 8801: 'equiv', 8804: 'le', 8805: 'ge', 8834: 'sub', 8835: 'sup', 8836: 'nsub', 8838: 'sube', 8839: 'supe', 8853: 'oplus', 8855: 'otimes', 8869: 'perp', 8901: 'sdot', 8968: 'lceil', 8969: 'rceil', 8970: 'lfloor', 8971: 'rfloor', 9001: 'lang', 9002: 'rang', 9674: 'loz', 9824: 'spades', 9827: 'clubs', 9829: 'hearts', 9830: 'diams'
};

/*
  Table associative : nom d'entitité XHTML: code du caractère correspondant.
  Par ex. : 'eacute': 233.
  (Cf. http://docs.python.org/py3k/library/html.entities.html#module-html.entities
  "A dictionary that maps HTML entity names to the Unicode codepoints.")
*/
var HTML_ENTITIES_NAME2CODEPOINT = {
    'AElig': 198, 'Aacute': 193, 'Acirc': 194, 'Agrave': 192, 'Alpha': 913, 'Aring': 197, 'Atilde': 195, 'Auml': 196, 'Beta': 914, 'Ccedil': 199, 'Chi': 935, 'Dagger': 8225, 'Delta': 916, 'ETH': 208, 'Eacute': 201, 'Ecirc': 202, 'Egrave': 200, 'Epsilon': 917, 'Eta': 919, 'Euml': 203, 'Gamma': 915, 'Iacute': 205, 'Icirc': 206, 'Igrave': 204, 'Iota': 921, 'Iuml': 207, 'Kappa': 922, 'Lambda': 923, 'Mu': 924, 'Ntilde': 209, 'Nu': 925, 'OElig': 338, 'Oacute': 211, 'Ocirc': 212, 'Ograve': 210, 'Omega': 937, 'Omicron': 927, 'Oslash': 216, 'Otilde': 213, 'Ouml': 214, 'Phi': 934, 'Pi': 928, 'Prime': 8243, 'Psi': 936, 'Rho': 929, 'Scaron': 352, 'Sigma': 931, 'THORN': 222, 'Tau': 932, 'Theta': 920, 'Uacute': 218, 'Ucirc': 219, 'Ugrave': 217, 'Upsilon': 933, 'Uuml': 220, 'Xi': 926, 'Yacute': 221, 'Yuml': 376, 'Zeta': 918, 'aacute': 225, 'acirc': 226, 'acute': 180, 'aelig': 230, 'agrave': 224, 'alefsym': 8501, 'alpha': 945, 'amp': 38, 'and': 8743, 'ang': 8736, 'aring': 229, 'asymp': 8776, 'atilde': 227, 'auml': 228, 'bdquo': 8222, 'beta': 946, 'brvbar': 166, 'bull': 8226, 'cap': 8745, 'ccedil': 231, 'cedil': 184, 'cent': 162, 'chi': 967, 'circ': 710, 'clubs': 9827, 'cong': 8773, 'copy': 169, 'crarr': 8629, 'cup': 8746, 'curren': 164, 'dArr': 8659, 'dagger': 8224, 'darr': 8595, 'deg': 176, 'delta': 948, 'diams': 9830, 'divide': 247, 'eacute': 233, 'ecirc': 234, 'egrave': 232, 'empty': 8709, 'emsp': 8195, 'ensp': 8194, 'epsilon': 949, 'equiv': 8801, 'eta': 951, 'eth': 240, 'euml': 235, 'euro': 8364, 'exist': 8707, 'fnof': 402, 'forall': 8704, 'frac12': 189, 'frac14': 188, 'frac34': 190, 'frasl': 8260, 'gamma': 947, 'ge': 8805, 'gt': 62, 'hArr': 8660, 'harr': 8596, 'hearts': 9829, 'hellip': 8230, 'iacute': 237, 'icirc': 238, 'iexcl': 161, 'igrave': 236, 'image': 8465, 'infin': 8734, 'int': 8747, 'iota': 953, 'iquest': 191, 'isin': 8712, 'iuml': 239, 'kappa': 954, 'lArr': 8656, 'lambda': 955, 'lang': 9001, 'laquo': 171, 'larr': 8592, 'lceil': 8968, 'ldquo': 8220, 'le': 8804, 'lfloor': 8970, 'lowast': 8727, 'loz': 9674, 'lrm': 8206, 'lsaquo': 8249, 'lsquo': 8216, 'lt': 60, 'macr': 175, 'mdash': 8212, 'micro': 181, 'middot': 183, 'minus': 8722, 'mu': 956, 'nabla': 8711, 'nbsp': 160, 'ndash': 8211, 'ne': 8800, 'ni': 8715, 'not': 172, 'notin': 8713, 'nsub': 8836, 'ntilde': 241, 'nu': 957, 'oacute': 243, 'ocirc': 244, 'oelig': 339, 'ograve': 242, 'oline': 8254, 'omega': 969, 'omicron': 959, 'oplus': 8853, 'or': 8744, 'ordf': 170, 'ordm': 186, 'oslash': 248, 'otilde': 245, 'otimes': 8855, 'ouml': 246, 'para': 182, 'part': 8706, 'permil': 8240, 'perp': 8869, 'phi': 966, 'pi': 960, 'piv': 982, 'plusmn': 177, 'pound': 163, 'prime': 8242, 'prod': 8719, 'prop': 8733, 'psi': 968, 'quot': 34, 'rArr': 8658, 'radic': 8730, 'rang': 9002, 'raquo': 187, 'rarr': 8594, 'rceil': 8969, 'rdquo': 8221, 'real': 8476, 'reg': 174, 'rfloor': 8971, 'rho': 961, 'rlm': 8207, 'rsaquo': 8250, 'rsquo': 8217, 'sbquo': 8218, 'scaron': 353, 'sdot': 8901, 'sect': 167, 'shy': 173, 'sigma': 963, 'sigmaf': 962, 'sim': 8764, 'spades': 9824, 'sub': 8834, 'sube': 8838, 'sum': 8721, 'sup': 8835, 'sup1': 185, 'sup2': 178, 'sup3': 179, 'supe': 8839, 'szlig': 223, 'tau': 964, 'there4': 8756, 'theta': 952, 'thetasym': 977, 'thinsp': 8201, 'thorn': 254, 'tilde': 732, 'times': 215, 'trade': 8482, 'uArr': 8657, 'uacute': 250, 'uarr': 8593, 'ucirc': 251, 'ugrave': 249, 'uml': 168, 'upsih': 978, 'upsilon': 965, 'uuml': 252, 'weierp': 8472, 'xi': 958, 'yacute': 253, 'yen': 165, 'yuml': 255, 'zeta': 950, 'zwj': 8205, 'zwnj': 8204
};



/*************
 * Fonctions *
 *************/
/*
  Ajoute la fonction trim() au type String si elle n'existe pas (pour Internet Explorer)
*/
if (typeof String.prototype.trim !== 'function') {
    String.prototype.trim = function(s) {
        'use strict';

        return s.replace(/(^\s+|\s+$)/g, '');
    };
}



/*
  Renvoie une copie de s dont
    les caractères "accentués" ont été convertis en enlevant leur "accent"
      (les caractères convertis sont ceux de la table associative ACCENTALPHA_TO_ALPHA)
    et les caractères de l'alphabet grec ont été convertis en caractères alphabétiques
      (les caractères convertis sont ceux de la table associative GREEK_TO_ALPHA).
  Ensuite les caractères non-alphanumériques sont remplacés par replacement.
  Si plusieurs caractères sont remplacés par replacement,
    seulement max_nb consécutifs sont utilisés, les suivants sont ignorés.

  Pre: s: String
       replacement: String
       max_nb: Number entier >= 0
*/
function alphanormalize(s, replacement, max_nb) {
    'use strict';

    var c;
    var i;

    var a = [];

    var nb_consecutive = 0;  // comptabilise le nombre de caractères consécutifs qui ont été remplacés

    var s_LENGTH = s.length;

    for (i = 0; i < s_LENGTH; i++) {
        c = s.charAt(i);

        if ((('0' <= c) && (c <= '9')) || (('A' <= c) && (c <= 'Z')) || (('a' <= c) && (c <= 'z'))) {  // caractère alphanumérique
            a.push(c);
            nb_consecutive = 0;
        }
        else if (c in ACCENTALPHA_TO_ALPHA) {  // caractère "accentué" -> 1 ou 2 caractères alphabétiques
            a.push(ACCENTALPHA_TO_ALPHA[c]);
            nb_consecutive = 0;
        }
        else if (c in GREEK_TO_ALPHA) {        // lettre grecque -> 1 ou 2 caractères alphabétiques
            a.push(GREEK_TO_ALPHA[c]);
            nb_consecutive = 0;
        }
        else if (nb_consecutive < max_nb) {   // autre caractère -> replacement, si pas déjà précédé de suffisament de remplacement
            nb_consecutive++;
            a.push(replacement);
        }
    }

    return a.join('');
}


/*
  Renvoie l'entier n converti en hexadécimal
  dans une String compléter de '0' pour obtenir une longueur multiple de 4

  Pre: n: Number entier >= 0

  Result: String
*/
function n_to_hexunicode(n) {
    'use strict';

    var hex = n.toString(16);

    while (hex.length%4 > 0) {
        hex = '0' + hex;
    }

    return hex;
}


/*
  Ajoute le caractère Unicode de numéro n dans txt2_src_text.value

  Pre: n: Number
*/
function txt2_add_char(n) {
    'use strict';

    if (txt2_src_text.selectionStart === undefined) {  // pour vieux IE ajouter à la fin
        txt2_src_text.value += String.fromCharCode(n);
    }
    else {                                             // ajouter à la position courante
        var i = txt2_src_text.selectionStart;

        txt2_src_text.value = txt2_src_text.value.substr(0, i) + String.fromCharCode(n) + txt2_src_text.value.substr(i);

        txt2_src_text.selectionStart = i + 1;
    }

    if (txt2_last_action !== null) {
        txt2_last_action();
    }

    txt2_post();
}


/*
  Ajoute '&' + n + ';' dans txt2_src_text.value

  Pre: s: String
*/
function txt2_add_entity(s) {
    'use strict';

    if (txt2_src_text.selectionStart === undefined) {  // pour vieux IE ajouter à la fin
        txt2_src_text.value += '&' + s + ';';
    }
    else {                                             // ajouter à la position courante
        var i = txt2_src_text.selectionStart;
        var added = '&' + s + ';';

        txt2_src_text.value = txt2_src_text.value.substr(0, i) + added + txt2_src_text.value.substr(i);

        txt2_src_text.selectionStart = i + added.length;
    }

    if (txt2_last_action !== null) {
        txt2_last_action();
    }

    txt2_post();
}


/*
  Exécuté après chaque changement du texte destination
*/
function txt2_change_dest() {
    'use strict';

    txt2_post();
}


/*
  Exécuté après chaque changement du texte source
*/
function txt2_change_src() {
    'use strict';

    txt2_src_infos_length.innerHTML = txt2_src_text.value.length;
}


/*
  Efface les deux zones de textes
*/
function txt2_clear() {
    'use strict';

    txt2_src_text.value = '';
    txt2_src_infos_length.innerHTML = 0;

    if (txt2_last_action !== null) {
        txt2_last_action();
    }

    txt2_post();
}


/*
  Change la police de caractères des deux zones de saisies,
  de la sélection courante
  et la liste des caractères.

  Pre: size: String
*/
function txt2_font(font) {
    'use strict';

    document.getElementById('choice-font-select').style.fontFamily = font;
    document.getElementById('txt2-src-text').style.fontFamily = font;
    document.getElementById('txt2-dest-text').style.fontFamily = font;
    document.getElementById('list-characters-list').style.fontFamily = font;
}


/*
  Change la taille de la police de caractères des deux zones de saisies
  et de la sélection courante

  Pre: size: String
*/
function txt2_font_size(size) {
    'use strict';

    document.getElementById('choice-font-size-select').style.fontSize = size;
    document.getElementById('txt2-src-text').style.fontSize = size;
    document.getElementById('txt2-dest-text').style.fontSize = size;
}


/*
  Exécuté à chaque pression de touche dans la zone de saisie du texte destination
*/
function txt2_key_dest() {
    'use strict';

    txt2_post();
}


/*
  Exécuté à chaque pression de touche dans la zone de saisie du texte source
*/
function txt2_key_src() {
    'use strict';

    if (txt2_last_action !== null) {
        txt2_last_action();
    }
    else {
        txt2_src_infos_length.innerHTML = txt2_src_text.value.length;
    }
}


/*
  Change la langue des <textarea> txt2-src-text et txt2-dest-text

  Pre: lang: String
*/
function txt2_lang(lang) {
    'use strict';

    document.getElementById('txt2-src-text').lang = lang;
    document.getElementById('txt2-dest-text').lang = lang;
}


/*
  Exécuté après chaque fonction de transformation ou chaque modification du texte de destination
*/
function txt2_post() {
    'use strict';

    txt2_dest_infos_length.innerHTML = txt2_dest_text.value.length;

    var nb_ascii = 0;
    var nb_latin1 = 0;

    var nb_white = 0;
    var nb_upper = 0;
    var nb_lower = 0;
    var nb_numeric = 0;
    var nb_special = 0;
    var nb_other = 0;

    var stats = {};

    var c, i;

    var txt2_dest_text_value_LENGTH = txt2_dest_text.value.length;

    for (i = 0; i < txt2_dest_text_value_LENGTH; i++) {
        c = txt2_dest_text.value.charCodeAt(i);

        if (document.getElementById('panel-statistics-content').style.display === 'block') {
            if (c in stats) {
                stats[c]++;
            }
            else  {
                stats[c] = 1;
            }
        }


        if (c <= 127) {
            nb_ascii++;
            nb_latin1++;
        }
        else if (c <= 255) {
            nb_latin1++;
        }


        if ((48 <= c) && (c <= 57)) {        // '0' <= c <= '9'
            nb_numeric++;
        }
        else if ((65 <= c) && (c <= 90)) {   // 'A' <= c <= 'Z'
            nb_upper++;
        }
        else if ((97 <= c) && (c <= 122)) {  // 'a' <= c <= 'z'
            nb_lower++;
        }
        else {
            c = String.fromCharCode(c);

            if (/\s/.test(c)) {
                nb_white++;
            }
            else if (/[!"#$%&'()*+,-.\/:;<=>?@\[\\\]_{|}]/.test(c)) {
                nb_special++;
            }
            else {
                nb_other++;
            }
        }
    }

    txt2_dest_infos_chars.innerHTML = '<span title="Number of ASCII&rsquo;s characters:\ncode &lt; 128.">ASCII:&nbsp;' + nb_ascii + '</span>&ensp;<span title="Number of Latin-1 (ISO 8859-1)&rsquo;s characters:\ncode &lt; 256.">Latin-1:&nbsp;' + nb_latin1 + '</span>&nbsp;|&nbsp;<span title="Number of these characters:\n\\t\\n\\v\\f\\r \\u00a0\\u2000\\u2001\\u2002\\u2003\\u2004\\u2005\\u2006\\u2007\\u2008\\u2009\\u200a\\u200b\\u2028\\u2029\\u3000">white:&nbsp;' + nb_white.toString() + '</span>&ensp;<span title="Number of these characters:\nABCDEFGHIJKLMNOPQRSTUVWXYZ">UPPER:&nbsp;' + nb_upper.toString() + '</span>&ensp;<span title="Number of these characters:\nabcdefghijklmnopqrstuvwxyz">lower:&nbsp;' + nb_lower.toString() + '</span>&ensp;<span title="Number of these characters:\n0123456789">numeric:&nbsp;' + nb_numeric.toString() + '</span>&ensp;<span title="Number of these characters:\n!&quot;#$%&amp;\'()*+,-./:;&lt;=&gt;?@[\\]_{|}">special:&nbsp;' + nb_special.toString() + '</span>&ensp;<span title="Number of other characters.">other:&nbsp;' + nb_other.toString() + '</span>';

    txt2_cmp_infos.innerHTML = 'Strings comparison: Source ' + (txt2_src_text.value < txt2_dest_text.value ? '&lt;' : (txt2_src_text.value > txt2_dest_text.value ? '&gt;' : '=')) + ' destination ';


    var dest = txt2_dest_text.value.toLowerCase();
    var src = txt2_src_text.value.toLowerCase();

    txt2_cmp_infos.innerHTML += ' &emsp; Insensitive comparison: ' + (src < dest ? '&lt;' : (src > dest ? '&gt;' : '='));

    dest = alphanormalize(txt2_dest_text.value,
                          document.getElementById('alphanormalize_replacement').value,
                          document.getElementById('alphanormalize_max_nb').value);
    src = alphanormalize(txt2_src_text.value,
                         document.getElementById('alphanormalize_replacement').value,
                         document.getElementById('alphanormalize_max_nb').value);
    txt2_cmp_infos.innerHTML += ' &emsp; Alphanormalized comparison: ' + (src < dest ? '&lt;' : (src > dest ? '&gt;' : '='));


    if (document.getElementById('panel-statistics-content').style.display === 'block') {
        txt2_panel_statistics_update(stats);
    }
}


/*
  Exécuté avant chaque fonction de transformation
*/
function txt2_pre() {
    'use strict';

    txt2_last_action = null;

    txt2_src_infos_length.innerHTML = txt2_src_text.value.length;
    txt2_regexp_infos.innerHTML = '';
}


/*
  Met en évidence (classe CSS selected) l'élement du DOM item
  et enlève la surbrillance sur l'élement du DOM txt2_last_input

  Pre: item: élément du DOM
             ou String (id d'un élément du DOM)
*/
function txt2_select(item) {
    'use strict';

    if (typeof item === 'string') {
        item = document.getElementById(item);
    }

    if (txt2_last_input) {
        txt2_last_input.classList.remove('selected');
    }

    item.classList.add('selected');
    txt2_last_input = item;
}



/*******************************
 * Fonctions de transformation *
 *******************************/
/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont n'est conservé que les premiers caractères de chaque "mot"
*/
function txt2_acronyme() {
    'use strict';

    txt2_pre();

    var c, i;

    var a = [];
    var prev_white = true;

    var txt2_src_text_value_LENGTH = txt2_src_text.value.length;

    for (i = 0; i < txt2_src_text_value_LENGTH; i++) {
        c = txt2_src_text.value.charAt(i);

        if (prev_white) {
            a.push(c);
        }

        prev_white = /\s/.test(c);
    }

    txt2_dest_text.value = a.join('');

    txt2_post();

    txt2_last_action = txt2_acronyme;
}


/*
  Place dans txt2_dest_text.value
  la valeur de alphanormalize(txt2_src_text.value, replacement, max_nb)

  Pre: replacement: String
       max_nb: String ou (Number entier >= 0)

  Bord: fixe la valeur document.getElementById('alphanormalize_max_nb').value à length
          avec 0 pour minimun et 99 pour maximum
*/
function txt2_alphanormalize(replacement, max_nb) {
    'use strict';

    txt2_pre();

    if (isNaN(max_nb)) {
        max_nb = parseInt(max_nb, 10);
    }
    max_nb = (isNaN(max_nb) ? 0 : Math.min(99, Math.max(0, max_nb)));

    if (document.getElementById('alphanormalize_max_nb').value !== max_nb.toString()) {
        document.getElementById('alphanormalize_max_nb').value = max_nb.toString();
    }

    txt2_dest_text.value = alphanormalize(txt2_src_text.value, replacement, max_nb);

    txt2_post();

    txt2_last_action = function () { return txt2_alphanormalize(replacement, max_nb); };
}


/*
  Découpe txt2_dest_text.value sur les caractères blancs
  et renvoie la liste des valeurs réprésentant un entier en base src_base convertie en base dest_base séparées par une espace
  (chaque élément donnant un NaN lors de sa conversion est remplacé par un '?')

  Pre: left_to_right: booléen

  Bord: fixe la valeur document.getElementById('base_to_base_left_base').value à une valeur comprise entre 2 et 36
        fixe la valeur document.getElementById('base_to_base_right_base').value à une valeur comprise entre 2 et 36
*/
function txt2_base_to_base(left_to_right) {
    'use strict';

    txt2_pre();

    var src_base;
    var dest_base;

    if (left_to_right) {
        src_base = document.getElementById('base_to_base_left_base').value;
        dest_base = document.getElementById('base_to_base_right_base').value;
    }
    else {
        src_base = document.getElementById('base_to_base_right_base').value;
        dest_base = document.getElementById('base_to_base_left_base').value;
    }

    if (isNaN(src_base)) {
        src_base = parseInt(src_base, 10);
    }
    src_base = Math.max(2, src_base);
    src_base = Math.min(36, src_base);

    if (isNaN(dest_base)) {
        dest_base = parseInt(dest_base, 10);
    }
    dest_base = Math.max(2, dest_base);
    dest_base = Math.min(36, dest_base);

    if (left_to_right) {
        if (document.getElementById('base_to_base_left_base').value !== src_base.toString()) {
            document.getElementById('base_to_base_left_base').value = src_base.toString();
        }
        if (document.getElementById('base_to_base_right_base').value !== dest_base.toString()) {
            document.getElementById('base_to_base_right_base').value = dest_base.toString();
        }
    }
    else {
        if (document.getElementById('base_to_base_right_base').value !== src_base.toString()) {
            document.getElementById('base_to_base_right_base').value = src_base.toString();
        }
        if (document.getElementById('base_to_base_left_base').value !== dest_base.toString()) {
            document.getElementById('base_to_base_left_base').value = dest_base.toString();
        }
    }

    var src = txt2_src_text.value.split(/\s+/);
    var dest = [];

    var i, n;

    for (i in src) {
        n = parseInt(src[i], src_base);
        dest.push(isNaN(n) ? '?' : n.toString(dest_base));
    }

    txt2_dest_text.value = dest.join(' ');

    txt2_post();

    txt2_last_action = function () { return txt2_base_to_base(left_to_right); };
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont,
    les caractères "accentués" ont été convertis en enlevant leur "accent"
      (les caractères convertis sont ceux de la table associative ACCENTALPHA_TO_ALPHA)
    et les caractères de l'alphabet grec ont été convertis en caractères alphabétiques
      (les caractères convertis sont ceux de la table associative GREEK_TO_ALPHA).
  Ensuite les caractères non-alphanumériques sont supprimés
  et chaque première lettre est convertie en majuscule.
*/
function txt2_camelcase() {
    'use strict';

    txt2_pre();

    var s = alphanormalize(txt2_src_text.value, ' ', 1);

    var c, i;

    var a = [];
    var prev_white = true;

    var s_LENGTH = s.length;

    for (i = 0; i < s_LENGTH; i++) {
        c = s.charAt(i);

        if (prev_white) {
            c = c.toUpperCase();
        }

        prev_white = (c === ' ');

        if (!prev_white) {
            a.push(c);
        }
    }

    txt2_dest_text.value = a.join('');

    txt2_post();

    txt2_last_action = txt2_camelcase;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont la première lettre de chaque "mot" est convertie en majuscule
  (en fait capitalise chaque caractère non blanc suivant un caractère blanc ou débutant le texte)
*/
function txt2_capitalize() {
    'use strict';

    txt2_pre();

    var c, i;

    var a = [];
    var prev_white = true;

    var txt2_src_text_value_LENGTH = txt2_src_text.value.length;

    for (i = 0; i < txt2_src_text_value_LENGTH; i++) {
        c = txt2_src_text.value.charAt(i);

        if (prev_white) {
            c = c.toUpperCase();
        }

        a.push(c);

        prev_white = /\s/.test(c);
    }

    txt2_dest_text.value = a.join('');

    txt2_post();

    txt2_last_action = txt2_capitalize;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont les nombres naturels ont été remplacés par leur caractère Unicode correspondant
  (si une espace suit immédiatement un nombre elle est supprimée)
*/
function txt2_chr() {
    'use strict';

    txt2_pre();

    var a = [];  // tableau pour récolter le résultat
    var n = [];  // tableau temporaire pour récolter les nombres à convertir

    var i;
    var c;

    var txt2_src_text_value_LENGTH = txt2_src_text.value.length;

    for (i = 0; i < txt2_src_text_value_LENGTH; i++) {
        c = txt2_src_text.value.charAt(i);

        if (('0' <= c) && (c <= '9')) {  // chiffre à mettre de côté
            n.push(c);
        }
        else {                           // autre caractère à simplement ajouter, après un éventuel nombre converti
            if (n.length) {
                a.push(String.fromCharCode(parseInt(n.join(''), 10)));
                n = [];
                if (c === ' ') {  // supprimer la première espace suivant le nombre
                    c = '';
                }
            }
            a.push(c);
        }
    }

    if (n.length) {  // reste un nombre à convertir
        a.push(String.fromCharCode(parseInt(n.join(''), 10)));
    }

    txt2_dest_text.value = a.join('');

    txt2_post();

    txt2_last_action = txt2_chr;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value
*/
function txt2_copy() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = txt2_src_text.value;

    txt2_post();

    txt2_last_action = txt2_copy;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont les entities HTML &...; ou &#...; ont été converties en leur caractère
*/
function txt2_html_entity_decode() {
    'use strict';

    txt2_pre();

    var i;
    var s;

    var a = txt2_src_text.value.split(/(&[A-Za-z0-9]{2,8};|&#\d{1,20};)/);  // découpe sur les &...; ou &#...;

    var a_LENGTH = a.length;

    for (i = 0; i < a_LENGTH; i++) {
        s = a[i];

        if ((s.length >= 4) && (s.charAt(0) === '&') && (s.charAt(s.length - 1) === ';')) {
            if (/&[A-Za-z0-9]{2,8};/.test(s)) {  // &...;
                s = s.substring(1, s.length - 1);
                if (s in HTML_ENTITIES_NAME2CODEPOINT) {
                    a[i] = String.fromCharCode(HTML_ENTITIES_NAME2CODEPOINT[s]);
                }
            }
            else if (/&#\d{1,20};/.test(s)) {    // &#...;
                a[i] = String.fromCharCode(s.substring(2, s.length - 1));
            }
        }
    }

    txt2_dest_text.value = a.join('');

    txt2_post();

    txt2_last_action = txt2_html_entity_decode;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont les caractères pour lesquels il existe une entity HTML &...; ont été convertis
*/
function txt2_htmlentities() {
    'use strict';

    txt2_pre();

    var a = [];

    var i;
    var c;

    var txt2_src_text_value_LENGTH = txt2_src_text.value.length;

    for (i = 0; i < txt2_src_text_value_LENGTH; i++) {
        c = txt2_src_text.value.charCodeAt(i);
        if (c in HTML_ENTITIES_CODEPOINT2NAME) {
            a.push('&');
            a.push(HTML_ENTITIES_CODEPOINT2NAME[c]);
            a.push(';');
        }
        else {
            a.push(txt2_src_text.value.charAt(i));
        }
    }

    txt2_dest_text.value = a.join('');

    txt2_post();

    txt2_last_action = txt2_htmlentities;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie en minuscules
*/
function txt2_lower() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = txt2_src_text.value.toLowerCase();

    txt2_post();

    txt2_last_action = txt2_lower;
}


/*
  Place dans txt2_dest_text.value
  la liste des codes Unicode des caractères de txt2_src_text.value séparés par une espace
*/
function txt2_ord() {
    'use strict';

    txt2_pre();

    var a = [];

    var i;

    var txt2_src_text_value_LENGTH = txt2_src_text.value.length;

    for (i = 0; i < txt2_src_text_value_LENGTH; i++) {
        a.push(txt2_src_text.value.charCodeAt(i));
    }

    txt2_dest_text.value = a.join(' ');

    txt2_post();

    txt2_last_action = txt2_ord;
}


/*
  Ouvre/ferme le panneau des Characters
  et actionne/libère le bouton
  (et ferme celui des Entities)
*/
function txt2_panel_list_characters_open_close() {
    'use strict';

    document.getElementById('list-entities-list').style.display = 'none';
    document.getElementById('list-entities-button').classList.remove('selected');

    var panel = document.getElementById('list-characters-list');

    if (panel.style.display === 'block') {  // close
        panel.style.display = 'none';
        document.getElementById('list-characters-button').classList.remove('selected');
    }
    else {                                  // open
        document.getElementById('list-characters-button').classList.add('selected');
        panel.style.display = 'block';
    }
}


/*
  Ouvre/ferme le panneau des Entities
  et actionne/libère le bouton
  (et ferme celui des Characters)
*/
function txt2_panel_list_entities_open_close() {
    'use strict';

    document.getElementById('list-characters-list').style.display = 'none';
    document.getElementById('list-characters-button').classList.remove('selected');

    var panel = document.getElementById('list-entities-list');

    if (panel.style.display === 'block') {  // close
        panel.style.display = 'none';
        document.getElementById('list-entities-button').classList.remove('selected');
    }
    else {                                  // open
        document.getElementById('list-entities-button').classList.add('selected');
        panel.style.display = 'block';
    }
}


/*
  Ouvre/ferme le tableau des résultats statistiques,
  le met à jour
  et actionne/libère le bouton
*/
function txt2_panel_statistics_open_close() {
    'use strict';

    var panel = document.getElementById('panel-statistics-content');

    if (panel.style.display === 'block') {  // close
        panel.style.display = 'none';
        document.getElementById('panel-statistics-results').innerHTML = '';
        document.getElementById('statistics').classList.remove('selected');
    }
    else {                                  // open
        document.getElementById('statistics').classList.add('selected');
        txt2_panel_statistics_update_buttons();
        txt2_panel_statistics_update();
        panel.style.display = 'block';
    }
}


/*
  Change le tri du tableau de résultats statistiques
  puis le met à jour

  Pre: n: Int 1 ou 2
*/
function txt2_panel_statistics_sort(n) {
    'use strict';

    if (txt2_panel_statistics_sorted === n) {
        if (txt2_panel_statistics_sorted_invert) {
            txt2_panel_statistics_sorted = 0;
            txt2_panel_statistics_sorted_invert = false;
        }
        else {
            txt2_panel_statistics_sorted_invert = true;
        }
    }
    else {
        txt2_panel_statistics_sorted = n;
        txt2_panel_statistics_sorted_invert = false;
    }

    txt2_panel_statistics_update_buttons();

    txt2_panel_statistics_update();
}


/*
  Comptabilise les occurences de chaque caractère de txt2_dest_text,
  rempli le tableau HTML des résultats.

  Si stats n'est pas indéfinie
  alors utilise ces statistiques et ne les recalcule pas

  Pre: stats: undefined ou table associative
*/
function txt2_panel_statistics_update(stats) {
    'use strict';

    txt2_panel_statistics_update_buttons();

    var n;

    if (stats === undefined) {
        stats = {};

        var i;

        var txt2_dest_text_value_LENGTH = txt2_dest_text.value.length;

        for (i = 0; i < txt2_dest_text_value_LENGTH; i++) {
            n = txt2_dest_text.value.charCodeAt(i);

            if (n in stats) {
                stats[n]++;
            }
            else {
                stats[n] = 1;
            }
        }
    }

    var keys = [];

    for (n in stats) {
        keys.push(parseInt(n, 10));
    }

    if (txt2_panel_statistics_sorted) {
        if (txt2_panel_statistics_sorted === 1) {
            if (txt2_panel_statistics_sorted_invert) {
                keys.sort(function (a, b) { return b - a; });
            }
            else {
                keys.sort(function (a, b) { return a - b; });
            }
        }
        else {
            if (txt2_panel_statistics_sorted_invert) {
                keys.sort(function (a, b) { return stats[b] - stats[a]; });
            }
            else {
                keys.sort(function (a, b) { return stats[a] - stats[b]; });
            }
        }
    }


    var tbody = document.getElementById('panel-statistics-results');

    while (tbody.children.length > 0) {  /* vide la <table> */
        tbody.deleteRow(-1);
    }

    var row;
    var cell;

    for (n in keys) {
        n = keys[n];

        row = tbody.insertRow(-1);

        row.insertCell(-1).innerHTML = '&amp;#' + n.toString() + ';';
        row.insertCell(-1).innerHTML = '\\u' + n_to_hexunicode(n);

        cell = row.insertCell(-1);
        cell.setAttribute('align', 'left');
        cell.innerHTML = String.fromCharCode(n);

        row.insertCell(-1).innerHTML = stats[n].toString();
    }
}


/*
  Actualise les boutons de tri
*/
function txt2_panel_statistics_update_buttons() {
    'use strict';

    var c = (txt2_panel_statistics_sorted_invert ? '&uarr;' : '&darr;');
    var item;

    document.getElementById('statistics-unicode-button').classList.remove('selected');
    document.getElementById('statistics-character-button').classList.remove('selected');
    document.getElementById('statistics-number-button').classList.remove('selected');

    if (txt2_panel_statistics_sorted) {
        if (txt2_panel_statistics_sorted === 1) {
            document.getElementById('statistics-unicode-button').classList.add('selected');
            document.getElementById('statistics-character-button').classList.add('selected');

            document.getElementById('statistics-number-arrow-l').style.visibility = 'hidden';
            document.getElementById('statistics-number-arrow-r').style.visibility = 'hidden';

            item = document.getElementById('statistics-unicode-arrow-l');
            item.innerHTML = c;
            item.style.visibility = 'visible';

            item = document.getElementById('statistics-unicode-arrow-r');
            item.innerHTML = c;
            item.style.visibility = 'visible';

            item = document.getElementById('statistics-character-arrow-l');
            item.innerHTML = c;
            item.style.visibility = 'visible';

            item = document.getElementById('statistics-character-arrow-r');
            item.innerHTML = c;
            item.style.visibility = 'visible';
        }
        else {
            document.getElementById('statistics-number-button').classList.add('selected');

            document.getElementById('statistics-unicode-arrow-l').style.visibility = 'hidden';
            document.getElementById('statistics-unicode-arrow-r').style.visibility = 'hidden';
            document.getElementById('statistics-character-arrow-l').style.visibility = 'hidden';
            document.getElementById('statistics-character-arrow-r').style.visibility = 'hidden';

            item = document.getElementById('statistics-number-arrow-l');
            item.innerHTML = c;
            item.style.visibility = 'visible';

            item = document.getElementById('statistics-number-arrow-r');
            item.innerHTML = c;
            item.style.visibility = 'visible';
        }
    }
    else {
        document.getElementById('statistics-unicode-arrow-l').style.visibility = 'hidden';
        document.getElementById('statistics-unicode-arrow-r').style.visibility = 'hidden';
        document.getElementById('statistics-character-arrow-l').style.visibility = 'hidden';
        document.getElementById('statistics-character-arrow-r').style.visibility = 'hidden';
        document.getElementById('statistics-number-arrow-l').style.visibility = 'hidden';
        document.getElementById('statistics-number-arrow-r').style.visibility = 'hidden';
    }

}


/*
  Place dans txt2_dest_text.value
  un mot de passe de longueur length généré aléatoirement
  (choisi parmi les caractères alphabétiques, numériques et/ou spéciaux)

  Pre: length: String ou (Number entier >= 0)
       alphabetic: booléen
       numeric: booléen
       special: booléen

  Bord: fixe la valeur document.getElementById('random_password_length').value à length
          avec 1 pour minimun et 99 pour maximum
*/
function txt2_random_password(length, alphabetic, numeric, special) {
    'use strict';

    txt2_pre();

    if (isNaN(length)) {
        length = parseInt(length, 10);
    }
    length = (isNaN(length) ? 1 : Math.min(99, Math.max(1, length)));

    if (prev_alphabetic === undefined) {
        var prev_alphabetic;
        var prev_numeric;
        var prev_special;
        var chars;
    }

    if ((prev_alphabetic !== alphabetic) || (prev_numeric !== numeric) || (prev_special !== special) || (chars === undefined)) {
        prev_alphabetic = alphabetic;
        prev_numeric = numeric;
        prev_special = special;

        chars = '';
        if (alphabetic) {
            chars += 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
        }
        if (numeric) {
            chars += '0123456789';
        }
        if (special) {
            chars += '!"#$%&\'()*+,-./:;<=>?@[\\]_{|}';
        }

        if (chars !== '') {
            chars += chars;
            while (chars.length < length*3) {
                chars += chars;
            }
        }
    }

    var i;
    var password;

    for (i = 0; i < 100; i++) {
        chars = str_shuffle(chars);  // jsPHP str_shuffle()
        password = chars.substring(0, length);

        if ((!alphabetic || /[A-Za-z]/.test(password)) && (!numeric || /[0-9]/.test(password)) && (!special || /[^A-Za-z0-9]/.test(password))) {
            break;
        }
    }

    if (document.getElementById('random_password_length').value !== password.length.toString()) {
        document.getElementById('random_password_length').value = password.length.toString();
    }

    txt2_dest_text.value = password;

    txt2_post();

    txt2_last_action = function () { return txt2_random_password(length, alphabetic, numeric, special); };
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value dont chaque sous-chaîne src a été remplacé par la sous-chaîne dest.

  Si pattern est de la forme "regexp" (entouré de deux ", les caractères blancs extérieures aux deux " étant ignorés)
  alors c'est l'expression régulière JavaScript RegExp(rexexp, regexp_flags) qui est utilisée

  Pre: pattern: String
       regexp_flags: String
       replacement: String

  Bord: dans le cas d'une regexp, sa valeur est placée dans txt2_regexp_infos
*/
function txt2_replace(pattern, regexp_flags, replacement) {
    'use strict';

    txt2_pre();

    var pattern_regexps = /^\s*"(.*)"\s*$/.exec(pattern);

    if (pattern_regexps !== null) {  // "regexp"
        try {
            var re = new RegExp(pattern_regexps[1], regexp_flags);

            txt2_regexp_infos.innerHTML = 'Regexp to replace: <tt>' + re.toString() + '</tt>';
            txt2_dest_text.value = txt2_src_text.value.replace(re, replacement);
        }
        catch (err) {
            txt2_regexp_infos.innerHTML = 'Regexp to replace: <span style="color:red">FAILED</span> <small>(' + err + ')</small>';
        }
    }
    else {                             // substring
        txt2_dest_text.value = txt2_src_text.value.split(pattern).join(replacement);
    }

    txt2_post();

    txt2_last_action = function () { return txt2_replace(pattern, regexp_flags, replacement); };
}


/*
  Fixe les paramètres pour que txt2_replace() remplace les adresses emails correctes
  puis appelle la fonction
*/
function txt2_replace_set_email() {
    'use strict';

    document.getElementById('replace_pattern').value = '"(?:[a-z0-9!#$%&\'*+/=?^_`{|}~-]+(?:\\.[a-z0-9!#$%&\'*+/=?^_`{|}~-]+)*|"(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21\\x23-\\x5b\\x5d-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21-\\x5a\\x53-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])+)\\])"';
    document.getElementById('replace_regexp_flags').value = 'gim';

    txt2_replace(document.getElementById('replace_pattern').value, document.getElementById('replace_regexp_flags').value, document.getElementById('replace_replacement').value);
}


/*
  Fixe les paramètres pour que txt2_replace() remplace les caractères blancs
  puis appelle la fonction
*/
function txt2_replace_set_spaces() {
    'use strict';

    document.getElementById('replace_pattern').value = '"\\s+"';
    document.getElementById('replace_regexp_flags').value = 'g';

    txt2_replace(document.getElementById('replace_pattern').value, document.getElementById('replace_regexp_flags').value, document.getElementById('replace_replacement').value);
}


/*
  Fixe les paramètres pour que txt2_replace() remplace les caractères blancs extérieurs
  puis appelle la fonction
*/
function txt2_replace_set_trim() {
    'use strict';

    document.getElementById('replace_pattern').value = '"(^\\s+|\\s+$)"';
    document.getElementById('replace_regexp_flags').value = 'g';

    txt2_replace(document.getElementById('replace_pattern').value, document.getElementById('replace_regexp_flags').value, document.getElementById('replace_replacement').value);
}


/*
  Fixe les paramètres pour que txt2_replace() remplace les URI correctes
  puis appelle la fonction
*/
function txt2_replace_set_uri() {
    'use strict';

    document.getElementById('replace_pattern').value = '"\\b((?:[a-z][\\w-]+:(?:/{1,3}|[a-z0-9%])|www\\d{0,3}[.]|[a-z0-9.\\-]+[.][a-z]{2,4}/)(?:[^\\s()<>]+|\\(([^\\s()<>]+|(\\([^\\s()<>]+\\)))*\\))+(?:\\(([^\\s()<>]+|(\\([^\\s()<>]+\\)))*\\)|[^\\s`!()\\[\\]{};:\'".,<>?«»“”‘’]))"';
    document.getElementById('replace_regexp_flags').value = 'gim';

    txt2_replace(document.getElementById('replace_pattern').value, document.getElementById('replace_regexp_flags').value, document.getElementById('replace_replacement').value);
}


/*
  Fixe les paramètres pour que txt2_replace() remplace les URI correctes avec un protocole http ou https
  puis appelle la fonction
*/
function txt2_replace_set_uri_http() {
    'use strict';

    document.getElementById('replace_pattern').value = '"\\b((?:https?://|www\\d{0,3}[.]|[a-z0-9.\\-]+[.][a-z]{2,4}/)(?:[^\\s()<>]+|\\(([^\\s()<>]+|(\\([^\\s()<>]+\\)))*\\))+(?:\\(([^\\s()<>]+|(\\([^\\s()<>]+\\)))*\\)|[^\\s`!()\\[\\]{};:\'".,<>?«»“”‘’]))"';
    document.getElementById('replace_regexp_flags').value = 'gim';

    txt2_replace(document.getElementById('replace_pattern').value, document.getElementById('replace_regexp_flags').value, document.getElementById('replace_replacement').value);
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value inversée
*/
function txt2_reverse() {
    'use strict';

    txt2_pre();

    var a = txt2_src_text.value.split('\n');

    a.reverse();
    txt2_dest_text.value = a.join('\n');

    txt2_post();

    txt2_last_action = txt2_reverse;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value mélangée
*/
function txt2_shuffle() {
    'use strict';

    txt2_pre();

    var a = txt2_src_text.value.split('\n');

    var j, k, l;
    var tmp;

    var i = -1;
    var a_LENGTH = a.length;

    while (++i < a_LENGTH) {
        j = Math.floor(Math.random()*a_LENGTH);
        k = Math.floor(Math.random()*a_LENGTH);
        l = Math.floor(Math.random()*a_LENGTH);

        tmp = a[i];
        a[i] = a[j];
        a[j] = a[k];
        a[k] = a[l];
        a[l] = tmp;
    }
    while (--i >= 0) {
        j = Math.floor(Math.random()*a_LENGTH);
        k = Math.floor(Math.random()*a_LENGTH);
        l = Math.floor(Math.random()*a_LENGTH);

        tmp = a[i];
        a[i] = a[j];
        a[j] = a[k];
        a[k] = a[l];
        a[l] = tmp;
    }

    txt2_dest_text.value = a.join('\n');

    txt2_post();

    txt2_last_action = txt2_shuffle;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value triée (insensible à la casse)
*/
function txt2_sort_insensitive() {
    'use strict';

    txt2_pre();

    var a = txt2_src_text.value.split('\n');

    a.sort(
        function (a, b) {
            a = a.toUpperCase();
            b = b.toUpperCase();

            return (a > b ? 1 : (a < b ? -1 : 0));
        });
    txt2_dest_text.value = a.join('\n');

    txt2_post();

    txt2_last_action = txt2_sort_insensitive;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value triée (numériquement)
*/
function txt2_sort_numeric() {
    'use strict';

    txt2_pre();

    var a = txt2_src_text.value.split('\n');

    a.sort(
        function (a, b) {
            a = parseFloat(a);
            b = parseFloat(b);

            return (a > b ? 1 : (a < b ? -1 : 0));
        });
    txt2_dest_text.value = a.join('\n');

    txt2_post();

    txt2_last_action = txt2_sort_numeric;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value triée (sensible à la casse)
*/
function txt2_sort_sensitive() {
    'use strict';

    txt2_pre();

    var a = txt2_src_text.value.split('\n');

    a.sort();
    txt2_dest_text.value = a.join('\n');

    txt2_post();

    txt2_last_action = txt2_sort_sensitive;
}


/*
  Echange txt2_src_text.value et txt2_dest_text.value
*/
function txt2_swap() {
    'use strict';

    var tmp = txt2_last_action;

    txt2_pre();
    txt2_last_action = tmp;

    /*
      [txt2_dest_text.value, txt2_src_text.value] = [txt2_src_text.value, txt2_dest_text.value];
      ne fonctionne pas sous IE, Chrome...
    */

    tmp = txt2_dest_text.value;

    txt2_dest_text.value = txt2_src_text.value;
    txt2_src_text.value = tmp;

    txt2_post();
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie en majuscules
*/
function txt2_upper() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = txt2_src_text.value.toUpperCase();

    txt2_post();

    txt2_last_action = txt2_upper;
}



/*************************************************************
 * Fonctions de transformation équivalent à une fonction PHP *
 *************************************************************/
/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP addslashes()
*/
function txt2_php_addslashes() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = addslashes(txt2_src_text.value);  // jsPHP addslashes()

    txt2_post();

    txt2_last_action = txt2_php_addslashes;
}


/*
  Place dans txt2_dest_text.value
  la somme de contrôle valeur CRC-32 de txt2_src_text.value
*/
function txt2_php_crc32() {
    'use strict';

    txt2_pre();

    var n = crc32(txt2_src_text.value);  // jsPHP crc32()

    if (n < 0) {
        n = 4294967296 + n;
    }

    txt2_dest_text.value = n.toString();

    txt2_post();

    txt2_last_action = txt2_php_crc32;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP html_entity_decode()
*/
function txt2_php_html_entity_decode() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = html_entity_decode(txt2_src_text.value);  // jsPHP html_entity_decode()

    txt2_post();

    txt2_last_action = txt2_php_html_entity_decode;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP htmlentities()
*/
function txt2_php_htmlentities() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = htmlentities(txt2_src_text.value);  // jsPHP htmlentities()

    txt2_post();

    txt2_last_action = txt2_php_htmlentities;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP htmlspecialchars()
*/
function txt2_php_htmlspecialchars() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = htmlspecialchars(txt2_src_text.value);  // jsPHP htmlspecialchars()

    txt2_post();

    txt2_last_action = txt2_php_htmlspecialchars;
}


/*
  Place dans txt2_dest_text.value
  la valeur de txt2_src_text.value cryptée en MD5
*/
function txt2_php_md5() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = md5(txt2_src_text.value);  // jsPHP md5()

    txt2_post();

    txt2_last_action = txt2_php_md5;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP quotemeta()
*/
function txt2_php_quotemeta() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = quotemeta(txt2_src_text.value);  // jsPHP quotemeta()

    txt2_post();

    txt2_last_action = txt2_php_quotemeta;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP rawurldecode()
*/
function txt2_php_rawurldecode() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = rawurldecode(txt2_src_text.value);  // jsPHP rawurldecode()

    txt2_post();

    txt2_last_action = txt2_php_rawurldecode;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP rawurlencode()
*/
function txt2_php_rawurlencode() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = rawurlencode(txt2_src_text.value);  // jsPHP rawurlencode()

    txt2_post();

    txt2_last_action = txt2_php_rawurlencode;
}


/*
  Place dans txt2_dest_text.value
  la valeur de txt2_src_text.value cryptée en SHA-1
*/
function txt2_php_sha1() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = sha1(txt2_src_text.value);  // jsPHP sha1()

    txt2_post();

    txt2_last_action = txt2_php_sha1;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP stripslashes()
*/
function txt2_php_stripslashes() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = stripslashes(txt2_src_text.value);  // jsPHP stripslashes()

    txt2_post();

    txt2_last_action = txt2_php_stripslashes;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP strip_tags()
*/
function txt2_php_strip_tags() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = strip_tags(txt2_src_text.value);  // jsPHP strip_tags()

    txt2_post();

    txt2_last_action = txt2_php_strip_tags;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP utf8_decode()
*/
function txt2_php_utf8_decode() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = utf8_decode(txt2_src_text.value);  // jsPHP utf8_decode()

    txt2_post();

    txt2_last_action = txt2_php_utf8_decode;
}


/*
  Place dans txt2_dest_text.value
  une copie de txt2_src_text.value convertie par l'équivalent de la fonction PHP utf8_encode()
*/
function txt2_php_utf8_encode() {
    'use strict';

    txt2_pre();

    txt2_dest_text.value = utf8_encode(txt2_src_text.value);  // jsPHP utf8_encode()

    txt2_post();

    txt2_last_action = txt2_php_utf8_encode;
}



/********
 * Main *
 ********/
(function () {  // initialisation
    'use strict';

    var lang = navigator.language || navigator.userLanguage;

    document.getElementById('choice-lang-select-navigator-language').innerHTML = 'Navigator: ' + lang;
    txt2_lang(lang);


    var a = [];
    var prev = 1;

    var i;

    var chars = [];

    for (i = 0; i < 256; i++) {
        chars.push(i);
    }
    for (i in HTML_ENTITIES_CODEPOINT2NAME) {
        i = parseInt(i, 10);
        if ((i >= 256) && ((i < 8448) || (i > 8523))) {
            chars.push(i);
        }
    }
    for (i = 8448; i <= 8523; i++) {
        chars.push(i);
    }
    for (i = 8544; i <= 8575; i++) {
        chars.push(i);
    }
    for (i = 9985; i <= 10008; i++) {
        chars.push(i);
    }

    chars.sort(function (a, b) { return a - b; });

    for (i in chars) {
        i = chars[i];
        if (Math.floor(i/64) !== Math.floor(prev/64)) {
            a.push('<br>');
        }
        if (i in HTML_ENTITIES_CODEPOINT2NAME) {
            a.push('<span title="&amp;#' + i + ';\n\\u' + n_to_hexunicode(i) + '\n&amp;' + HTML_ENTITIES_CODEPOINT2NAME[i] + ';" onclick="txt2_add_char(' + i + ');">&#' + i + ';</span>');
        }
        else {
            a.push('<span class="no-entity" title="&amp;#' + i + ';\n\\u' + n_to_hexunicode(i) + '" onclick="txt2_add_char(' + i + ');">&#' + i + ';</span>');
        }

        prev = i;
    }

    document.getElementById('list-characters-list').innerHTML = a.join('');

    a = [];
    prev = 1;
    for (i in HTML_ENTITIES_CODEPOINT2NAME) {
        i = parseInt(i, 10);
        if (Math.floor(i/64) !== Math.floor(prev/64)) {
            a.push('<br>');
        }
        a.push('<span title="&' + HTML_ENTITIES_CODEPOINT2NAME[i] + ';\n&amp;#' + i + ';\n\\u' + n_to_hexunicode(i) + '" onclick="txt2_add_entity(\'' + HTML_ENTITIES_CODEPOINT2NAME[i] + '\');">&amp;' + HTML_ENTITIES_CODEPOINT2NAME[i] + ';</span>');

        prev = i;
    }

    document.getElementById('list-entities-list').innerHTML = a.join('');
}());



txt2_pre();

txt2_last_action = function () {
    'use strict';

    return txt2_alphanormalize(document.getElementById('alphanormalize_replacement').value,
                               document.getElementById('alphanormalize_max_nb').value);
};

txt2_last_action();

document.getElementById('txt2-src-text').focus();

txt2_post();

txt2_font(document.getElementById('choice-font-select').value);
