/* MandelGMP
 *   Calcule l'ensemble de Mandelbrot
 *   et sauve l'image dans "MandelGMP.ppm"
 * (ncessite la bibliothque GMP : http://gmplib.org/ )
 * (c) Olivier Pirson --- DragonSoft
 * http://www.opimedia.be/DS/
 * Dbut le 14 aot 2009
 *         --- 2 janvier 2012 : nouveau site web
 ********************************************************/
#define VERSION "v.01.00 --- 2012 January 2"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include <gmp.h>



/*********************
 * Macros constantes *
 *********************/
/* Largeur de l'image */
#define WIDTH 500

/* Hauteur de l'image */
#define HEIGHT 500



/************
 * Fonction *
 ************/
/* Calcule une image de taille WIDTH x HEIGHT de l'ensemble de Mandelbrot
   et la sauve dans le fichier "Mandel.ppm"
   pre: max_ite: nombre maximum d'itrations (pour chaque point),
        black: noir&blanc, sinon en niveau gris */
void mandelbrot_set(unsigned int max_ite, bool black) {
  FILE
    *file = fopen("MandelGMP.ppm", "wb");  /* fichier binaire ouvert en criture */

  mpf_t
    /* Coin suprieur droit */
    z0_real,
    z0_imag,

    /* Coin infrieur gauche */
    z1_real,
    z1_imag,

    /* Coefficients pour calculer les coordonnes correspondant aux pixels */
    COEF_C_real,
    COEF_C_imag,

    C_real,
    C_imag,

    z_real,
    z_imag,

    z_real2,
    z_imag2,

    z_abs2;

  unsigned int
    y;


  fprintf(file, "P6\n%u %u\n255\n", WIDTH, HEIGHT);  /* en-tte de l'image au format PPM */

  mpf_init_set_si(z0_real, -2);
  mpf_init_set_si(z0_imag, 2);

  mpf_init_set_si(z1_real, 2);
  mpf_init_set_si(z1_imag, -2);

  mpf_init(COEF_C_real);
  mpf_init(COEF_C_imag);
  mpf_init(C_real);
  mpf_init(C_imag);

  mpf_init(z_real);
  mpf_init(z_imag);
  mpf_init(z_real2);
  mpf_init(z_imag2);
  mpf_init(z_abs2);

  /* COEF_C_real = (z1_real - z0_real)/(WIDTH - 1) */
  mpf_sub(COEF_C_real, z1_real, z0_real);
  mpf_div_ui(COEF_C_real, COEF_C_real, WIDTH - 1);

  /* COEF_C_imag = (z0_imag - z1_imag)/(HEIGHT - 1) */
  mpf_sub(COEF_C_imag, z0_imag, z1_imag);
  mpf_div_ui(COEF_C_imag, COEF_C_imag, HEIGHT - 1);

  /* Calcule chaque pixel de l'image, ligne par ligne */
  for(y=0; y<HEIGHT; y++) {
    unsigned int
      x;

    /* C_imag = z0_imag - COEF_C_imag*y */
    mpf_mul_ui(C_imag, COEF_C_imag, y);
    mpf_sub(C_imag, z0_imag, C_imag);

    for(x=0; x<WIDTH; x++) {
      /* C_real = z0_real + COEF_C_real*x */
      mpf_mul_ui(C_real, COEF_C_real, x);
      mpf_add(C_real, z0_real, C_real);

      /* On va itrer la fonction pour le point complexe C_real + i.C_imag */
      mpf_set(z_real, C_real);
      mpf_set(z_imag, C_imag);

      {
        unsigned int
          nb_ite;  /* nombre d'itrations */

        for (nb_ite=0; nb_ite<max_ite; nb_ite++)  {
          mpf_mul(z_real2, z_real, z_real);  /* z_real2 = z_real*z_real */
          mpf_mul(z_imag2, z_imag, z_imag);  /* z_imag2 = z_imag*z_imag */

          mpf_add(z_abs2, z_real2, z_imag2);  /* carr du module de z = z_real2 + z_imag2 */

          if (mpf_cmp_ui(z_abs2, 4) > 0)
            break;

          /* z_imag = z_imag*z_real*2 + C_imag */
          mpf_mul(z_imag, z_imag, z_real);
          mpf_mul_ui(z_imag, z_imag, 2);
          mpf_add(z_imag, z_imag, C_imag);

          /* z_real = z_real2 - z_imag2 + C_real */
          mpf_sub(z_real, z_real2, z_imag2);
          mpf_add(z_real, z_real, C_real);
        }
        {
          const unsigned int
            c = (mpf_cmp_ui(z_abs2, 4) <= 0
                 ? 0
                 : (black
                    ? 255
                    : 255 - nb_ite*254/max_ite));
          fputc(c, file);  /* rouge */
          fputc(c, file);  /* vert */
          fputc(c, file);  /* bleu */
        }
      }
    }
  }

  fclose(file);

  /* Libre la mmoire utilise par GMP */
  mpf_clear(z0_real);
  mpf_clear(z0_imag);
  mpf_clear(z1_real);
  mpf_clear(z1_imag);

  mpf_clear(COEF_C_real);
  mpf_clear(COEF_C_imag);
  mpf_clear(C_real);
  mpf_clear(C_imag);

  mpf_clear(z_real);
  mpf_clear(z_imag);
  mpf_clear(z_real2);
  mpf_clear(z_imag2);
  mpf_clear(z_abs2);
}



/********
 * Main *
 ********/
int main(int argc, const char *argv[]) {
  fprintf(stderr, "MandelGMP (GMP version %s) --- " VERSION "\n"
          "--- Begin calculation... ", gmp_version);
  fflush(stderr);

  mandelbrot_set(50, false);

  fputs("end ---\n", stderr);
  return EXIT_SUCCESS;
}
