#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\file
# Script crant une table HTML des naturels
#   avec les valeurs de diffrentes fonctions arithmtiques
#
# Cf. \htmlonly <a href="http://www.opimedia.be/naturels/" target="_blank">
#   <tt>http://www.opimedia.be/naturels/</tt></a>\endhtmlonly

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut fin 2006
# v.01.00 --- 12 mai 2007
#         --- 14 november 2007
#         --- 2 mars 2008
# v.01.01 --- 18 juin 2008 : correction &piv;(0) = 0
# v.01.02 --- 13 novembre 2008 : //  la place de / et tests des types
# v.01.03 --- 29 septembre 2009 : nouveau site web, changement print()
# v.01.04 --- 14 dcember 2009 : from __future__ import division
# v.01.05 --- 16 mars 2010 : nouveau site web et changement des % en .format()
#         --- 12 avril 2010 : cfr. -> cf.
###############################################################################
from __future__ import division
from __future__ import print_function

## Version
VERSION = 'v.01.05 --- 2010 April 12'

import math, numbers, sys, time

import DSPython
import DSPython.factors as factors
import DSPython.natural as natural
import DSPython.nbsystem as nbsystem

try:
    if not 'profile' in dir():
        import psyco
        psyco.full()
except ImportError:
    pass



# ############
# Constantes #
##############
## Titre pour n
N = 'n&thinsp;'


## Caractre infini (en gris)
INFIN = '<font color="gray">&infin;</font>'

## Caractre N, ensemble des naturels
NATURALS = '&#8469;'

## Caractre ~ (en petit)
SIM = '<font size="-2">&sim;</font>'

## Caractre x (en gris)
X = '<font color="gray" face="Arial">x</font>'


## Pour aligner une cellule  gauche
LEFT = 1

## Pour aligner une cellule  droite
RIGHT = 0



# ###########
# Fonctions #
#############
## Renvoie sous forme de string les diviseurs du naturel correspondant aux primaries p
def divisors_str(p):
    """Renvoie sous forme de string les diviseurs du naturel correspondant aux primaries p

    Pre: p: 0 ou primaries"""
    assert (p == 0) or factors.primaries_is(p), p

    if p != 0:
        ld = natural.divisors(factors.primaries_to_n(p))
        l = []
        for d in ld:
            l.append(n_to_href(d))
        return ','.join(l)
    else:
        return NATURALS


## Renvoie sous forme de string les facteurs premiers du naturel correspondant aux primaries p
def factors_str(p):
    """Renvoie sous forme de string les facteurs premiers du naturel correspondant aux primaries p

    Pre: p: 0 ou primaries"""
    assert (p == 0) or factors.primaries_is(p), p

    if p != 0:
        f_l = []
        for f_n in p:
            if f_n[1] > 1:
                f_l.append('{0}<sup>{1}</sup>'.format(n_to_href(f_n[0]), n_to_href(f_n[1])))
            else:
                f_l.append(n_to_href(f_n[0]))
        f_str = '.'.join(f_l)
        return(f_str if f_str != ''
               else '&nbsp;')
    else:
        return X


##\brief Affiche le message d'aide sur la sortie des erreurs
## et termine le programme par un code d'erreur 1
def help_msg():
    """Affiche le message d'aide sur la sortie des erreurs
    et termine le programme par un code d'erreur 1"""
    print("""naturalstable [-nothref] [to [from]]

  HTML table of naturals and arithmetic functions on standard output
  (c) Olivier Pirson --- DragonSoft --- {0}
             {1}
  Options: to:   last natural [50]
           from: first natural [0]
           -nothref: HTML links
           --help:   print this message on error output and exit


  Infos:
  {2}
  Python {3}""".format(DSPython.DS_web, VERSION, DSPython.VERSION, sys.version),
          file=sys.stderr)
    sys.exit(1)


## Renvoie n (dans un notation rduite si n est trop grand)
def n_to_big(n):
    """Renvoie n (dans un notation rduite si n est trop grand)
    pre: n: quelconque"""
    if isinstance(n, numbers.Number) and (n >= 10**9):
        s =  '{0:g}'.format(float(n))
        k = s.find('e')
        e = s[k + 2:].lstrip('0')
        s = s[:k]
        k = s.find('.')
        if k >= 0:
            s = s[:k]
        return SIM + s + '.10<sup>' + e + '</sup>'
    else:
        return n


## Renvoie True si p est les primaries d'un nombre premier, False sinon
def prime_is(p):
    """Renvoie True si p est les primaries d'un nombre premier, False sinon

    Pre: p: 0 ou primaries"""
    assert (p == 0) or factors.primaries_is(p), p

    return (p != 0) and factors.primaries_prime_is(p)


## Envoie sur la sortie standard la fin du document HTML
def print_html_footer():
    """Envoie sur la sortie standard la fin du document HTML"""
    print("""</body>
</html>""")


## Envoie sur la sortie standard l'en-tte du document HTML
def print_html_header(n_from=2, n_to=50):
    """Envoie sur la sortie standard l'en-tte du document HTML

    Pre: n_from: naturel <= n_to
         n_to: naturel"""
    assert DSPython.natural_is(n_from), n_from
    assert DSPython.natural_is(n_to), n_to
    assert n_from <= n_to, (n_from, n_to)

    print("""<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">
<html lang="fr">
<head>
  <meta http-equiv="content-type" content="text/html; charset=ISO-8859-1">
  <meta name="author" content="Olivier Pirson">
  <meta name="date-revision-yyyymmdd" content="{0}">
  <meta name="description" content="table des naturels (de {1}  {2}) avec leur dcomposition en facteurs premiers...
    (gnre par DSPython/naturalstable.py
    --- {3} --- DragonSoft --- {4})">
  <meta name="keywords" content="naturels, facteurs premiers, sigma, totient">
  <meta name="keywords" lang="en" content="naturals, prime factors, sigma, totient">

  <title>Table des naturels (de {5}  {6})
    avec leur dcomposition en facteurs premiers et fonctions associes</title>
  <style type="text/css"><!--a {{color:inherit;text-decoration:none}}--></style>
</head>

<body>""".format(time.strftime('%Y%m%d'), n_from, n_to, VERSION, DSPython.DS_web, n_from, n_to))


## Envoie sur la sortie standard la table HTML
def print_html_table(n_from=0, n_to=50, href=True):
    """Envoie sur la sortie standard la table HTML.
    Si href alors des liens HTML sont crs pour les naturels considrs

    Pre: n_from: naturel <= n_to
         n_to: naturel
         href: valeur boolenne"""
    assert DSPython.natural_is(n_from), n_from
    assert DSPython.natural_is(n_to), n_to
    assert n_from <= n_to, (n_from, n_to)

    global n_to_href
    if href:
        def n_to_href(n, name=False):
            """Renvoie n sous forme de string en ajoutant le lien HTML
              si n est un naturel accessible.
            Si name alors ajoute l'ancre et le lien HTML"""
            if name:
                return '<a name="{0}" href="#{1}">{2}</a>'.format(n, n, n_to_big(n))
            elif isinstance(n, numbers.Number) and (n_from <= n <= n_to) and (int(n) == n):
                return '<a href="#{0}">{1}</a>'.format(n, n_to_big(n))
            else:
                return str(n_to_big(n))
    else:
        def n_to_href(n, name=False):
            """Renvoie n sous forme de string"""
            return str(n_to_big(n))

    # Initialise quelques valeurs pour pouvoir commencer  n_from
    ## Somme des fonctions de Liouville (fonction L)
    global _liouville_sum
    ## Fonction M de Mertens
    global _mertens
    ## Fonction pi
    global _pi

    _pi = 0
    _mertens = 0
    _liouville_sum = 0
    for n in range(1, n_from):
        p = factors.primaries(n)
        if prime_is(p):
            _pi +=1
        _mertens += factors.mobius(p)
        _liouville_sum += (-1)**len(factors.primaries_to_primes(p))

    print_html_table_header(n_from=n_from, n_to=n_to)
    print_html_title_line()
    print('<tr><td></td></tr>')

    # Lignes de la table
    for n in range(n_from, n_to + 1):
        p = (0 if n==0
             else factors.primaries(n))
        if prime_is(p):
            _pi += 1
        if n > 0:
            _liouville_sum += (-1)**len(factors.primaries_to_primes(p))
            _mertens += factors.mobius(p)
        line = '<tr align="right">'
        # Colonnes de la table
        for i in range(len(L)):
            col = L[i]
            if col[0] == '':  # colonne vide
                line += '<td></td>'
                continue
            if (len(col) >= 3) and (col[2] == LEFT):  # colonne aligne  gauche
                begin = '<td align="left">'
            else:                                     # colonne normale
                begin = '<td>'
            end = '</td>'
            if len(col) >= 5:  # pour smaller
                begin += col[3]
                end = col[4] + end

            v = col[1](p)  # valeur telle que renvoye par la fonction de L
            s = n_to_href(v, name=col[0]==N)  # valeur dans un string avec lien HTML s'il le faut

            if ((col[0] != N) and (v == n)) or ((col[0] == N) and prime_is(p)):  # en gras
                begin += '<b>'
                end = '</b>' + end
            elif v == 0:                                                         # en gris
                begin += '<font color="gray">'
                end = '</font>' + end
            line += begin + s + end
        print(line + '</tr>')
    print_html_table_footer()


## Envoie sur la sortie standard la fin de la table HTML
def print_html_table_footer():
    """Envoie sur la sortie standard la fin de la table HTML"""
    print('</table>')


## Envoie sur la sortie standard le dbut de la table HTML
def print_html_table_header(n_from=2, n_to=50):
    """Envoie sur la sortie standard le dbut de la table HTML

    Pre: n_from: naturel <= n_to
         n_to: naturel"""
    assert DSPython.natural_is(n_from), n_from
    assert DSPython.natural_is(n_to), n_to
    assert n_from <= n_to, (n_from, n_to)

    print('<table border="1" cellpadding="0" cellspacing="0" summary="table {0}  {1}">'
          .format(n_from, n_to))


## Envoie sur la sortie standard la ligne de titre de la table HTML
def print_html_title_line():
    """Envoie sur la sortie standard la ligne de titre de la table HTML"""
    line = '<tr align="right">'
    for i in range(len(L)):
        col = L[i]
        s = col[0]
        if s == '':  # colonne vide
            line += '<td></td>'
        else:
            line += '<td{0}>{1}</td>'.format(' align="left"' if (len(col) >= 3) and (col[2] == LEFT)
                                             else '', s)
    print(line + '</tr>')


##\brief Fonction sigma<sub>-1</sub>,
## la somme des inverses des diviseurs du naturel correspondant aux primaries p
def sigma_1(p):
    """Fonction sigma_(-1),
    la somme des inverses des diviseurs du naturel correspondant aux primaries p

    Pre: p: 0 ou primaries"""
    assert (p == 0) or factors.primaries_is(p), p

    if p != 0:
        n = factors.primaries_to_n(p)
        r = factors.divisors_sum(p) / n
        if math.floor(r) == r:
            r = int(r)
        else:
            r = ('{0:.2f}'.format(r)).rstrip('0')
            if float(r)*n != factors.divisors_sum(p):
                r = SIM + r
        return r
    else:
        return INFIN



# ######\cond MAIN
# Main #
########
## Premier naturel de la table
n_from = 0

## Dernier naturel de la table
n_to = 50


## Si True alors ajoute des liens HTML
href = True


## Avant un lment pour rduire sa taille
begin_smaller = ''

## Aprs un lment pour rduire sa taille
end_smaller = ''


## Liste des paramtres numriques
params = []
for i in range(1, len(sys.argv)):  # parcours les paramtres
    if sys.argv[i] == '-nothref':
        href = False
    elif sys.argv[i].lower() == '--help':
        help_msg()
    else:
        try:
            params.append(int(sys.argv[i]))
        except:
            help_msg()

if len(params) >= 1:
    n_to = params[0]
    if len(params) >= 2:
        n_from = params[1]

if n_to >= 100:
    begin_smaller = '<font size="-2">'
    end_smaller = '</font>'


## Tuple des fonctions de la table
## ('titre', fonction  partir des primaries, align, 'dbut smaller', 'fin smaller')
L = ((N, lambda p: (0 if p==0
                    else factors.primaries_to_n(p))),
     ('', ),
     ('facteurs', factors_str, LEFT),
     ('&pi;(n)', lambda p: _pi),
     ('&omega;(n)', lambda p: (INFIN if p==0
                               else len(p))),
     ('&Omega;(n)', lambda p: (INFIN if p==0
                               else len(factors.primaries_to_primes(p)))),
     ('&sigma;<sub>-1</sub>(n)', sigma_1),
     ('&nu;(n)', lambda p: (INFIN if p==0
                            else factors.divisors_nb(p))),
     ('&sigma;(n)', lambda p: (INFIN if p==0
                               else factors.divisors_sum(p))),
     ('&sigma;<sub>i</sub>(n)', lambda p: (INFIN if p==0
                                           else factors.divisors_sum_odd(p))),
     ('&sigma;<sub>p</sub>(n)', lambda p: (INFIN if p==0
                                           else factors.divisors_sum_even(p))),
     ('s(n)', lambda p: (INFIN if p==0
                         else factors.properdivisors_sum(p))),
     ('&piv;(n)', lambda p: (0 if p==0
                             else factors.divisors_prod(p))),
     ('GR(n)', lambda p: (1 if p==0
                          else factors.nb_in_integers_4sqr(p))),
     ('&phi;(n)', lambda p: (1 if p==0
                             else factors.totient(p))),
     ('&lambda;(n)', lambda p: (X if p==0
                                else (-1)**len(factors.primaries_to_primes(p)))),
     ('L(n)', lambda p: _liouville_sum),
     ('&mu;(n)', lambda p: (X if p==0
                            else factors.mobius(p))),
     ('M(n)', lambda p: _mertens),
     ('<span style="font-size:smaller;text-decoration:overline"><font>'
      + 'n</font></span><sup><sup>F</sup></sup>',
      lambda p: ('0' if p==0
                 else natural.bin(nbsystem.fibonacci(factors.primaries_to_n(p)))),
      RIGHT, begin_smaller, end_smaller),
     ('<span style="font-size:smaller;text-decoration:overline"><font>'
      + 'n</font></span><sup><sup>2</sup></sup>',
      lambda p: ('0' if p==0
                 else natural.bin(factors.primaries_to_n(p))),
      RIGHT, begin_smaller, end_smaller),
     ('<span style="font-size:smaller;text-decoration:overline"><font>'
      + 'n</font></span><sup><sup>3</sup></sup>',
      lambda p: ('0' if p==0
                 else nbsystem.to_str(factors.primaries_to_n(p), b=3)),
      RIGHT, begin_smaller, end_smaller),
     ('diviseurs', divisors_str, LEFT, begin_smaller, end_smaller))


print_html_header(n_from=n_from, n_to=n_to)
print_html_table(n_from=n_from, n_to=n_to, href=href)
print_html_footer()
##\endcond MAIN
