#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.urmCutland
# URM (Unlimited Register Machine) de <span style="font-variant:small-caps">Cutland</span>
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Theo" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Theo</tt></a>\endhtmlonly

##\file
# URM (Unlimited Register Machine) de <span style="font-variant:small-caps">Cutland</span>
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Theo" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Theo</tt></a>\endhtmlonly

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 6 janvier 2008
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'urmCutland --- 2010 April 12'

import DSPython
import DSPython.factors as factors



# ######### ??? changer les exceptions pour chaque raise
# Classes #
###########
## Une instruction pour la machine virtuelle UrmCutland
class UrmCutlandInstruction:
    """Une instruction pour la machine virtuelle UrmCutland"""

    ## self == value ?
    def __eq__(self, value):
        """Renvoie True si self reprsente la mme instruction que value
        (cd si mme type d'instruction et mmes arguments),
        False sinon

        Result: bool

        O() = 1"""
        return (isinstance(value, UrmCutlandInstruction)
                and (self._inst == value._inst) and (self._a == value._a)
                and ((self._inst != T) or (self._b == value._b))
                and ((self._inst != J) or ((self._b == value._b) and (self._c == value._c))))


    ## Initialise l'instruction.
    def __init__(self, inst, a=1, b=1, c=1):
        """Si inst == Z alors initialise l'instruction  Z(a),
        si inst == S alors initialise l'instruction  S(a),
        si inst == T alors initialise l'instruction  T(a,b),
        si inst == J alors initialise l'instruction  J(a,b,c)

        Si inst est un string
        alors tous les espaces ' ' et tabulations '\t' sont ignors
        et si inst est 'Z(a)'     alors initialise l'instruction  Z(a),
           si inst est 'S(a)'     alors initialise l'instruction  S(a),
           si inst est 'T(a,b)'   alors initialise l'instruction  T(a,b),
           si inst est 'J(a,b,c)' alors initialise l'instruction  J(a,b,c)
        Si la syntaxe de inst est incorrecte alors une exception est leve

        Pre: inst: Z, S, T, J
                   ou string de la forme 'Z(a)', 'S(a)', 'T(a,b)'
                                         ou 'J(a,b,c)'
                      avec ventuellement des ' ' et '\t',
                           a, b et c reprsentants des naturels >= 1
                                     crits en base 10
             Pre: a: naturel >= 1
                  b: naturel >= 1
                  c: naturel >= 1

        O() = 1"""
        if isinstance(inst, int):  # instruction par son code
            assert Z <= inst <= J, inst
            assert DSPython.natural_is(a), (type(a), a)
            assert a >= 1, a
            assert DSPython.natural_is(b), (type(b), b)
            assert b >= 1, b
            assert DSPython.natural_is(c), (type(c), c)
            assert c >= 1, c

            self._inst = inst    # code de l'instruction
            self._a = a          # 1er argument
            if inst >= T:
                self._b = b      # 2e argument
                if inst == J:
                    self._c = c  # 3e argument
        else:                      # string
            assert isinstance(inst, str), (type(inst), inst)

            s = inst.replace(' ', '')
            s = s.replace('\t', '')
            if (len(s) < 4) or (s[1] != '(') or (s[-1] != ')'):
                raise "syntax error in instruction '{0}'".format(inst)

            args = s[2:-1].split(',')
            try:
                args = [int(t) for t in args]
            except:
                raise "bad argument in instruction '{0}'".format(inst)
            if args[0] <= 0:
                raise "bad 1st argument in instruction '{0}'".format(inst)
            self._a = args[0]

            if s[0] == 'Z':    # Z(a)
                self._inst = Z
                if len(args) != 1:
                    raise "Z take 1 argument ({0} given): '{1}'".format(len(args), inst)
            elif s[0] == 'S':  # S(a)
                self._inst = S
                if len(args) != 1:
                    raise "S take 1 argument ({0} given): '{1}'".format(len(args), inst)
            elif s[0] == 'T':  # T(a,b)
                self._inst = T
                if len(args) != 2:
                    raise "T take 2 arguments ({0} given): '{1}'".format(len(args), inst)
                if args[1] <= 0:
                    raise "bad 2nd argument in instruction '{0}'".format(inst)
                self._b = args[1]
            elif s[0] == 'J':  # J(a,b,c)
                self._inst = J
                if len(args) != 3:
                    raise "J take 3 arguments ({0} given): '{1}'".format(len(args), inst)
                if args[1] <= 0:
                    raise "bad 2nd argument in instruction '{0}'".format(inst)
                self._b = args[1]
                if args[2] <= 0:
                    raise "bad 3rd argument in instruction '{0}'".format(inst)
                self._c = args[2]
            else:              # erreur
                raise "bad instruction '{0}'".format(inst)


    ## self != value ?
    def __ne__(self, value):
        """Renvoie True si self reprsente une instruction diffrente de value
        (cd si type d'instruction diffrent
        ou au moins un argument diffrent),
        False sinon

        Result: bool

        O() = 1"""
        return not (self == value)


    ## Renvoie l'instruction sous forme d'un string
    def __str__(self):
        """Renvoie l'instruction sous forme d'un string

        Result: string

        O() = 1"""
        if self._inst == Z:    # Z(a)
            return 'Z({0})'.format(self._a)
        elif self._inst == S:  # S(a)
            return 'S({0})'.format(self._a)
        elif self._inst == T:  # T(a, b)
            return 'T({0},{1})'.format(self._a, self._b)
        else:                  # J(a, b, c)
            assert self._inst == J, self._inst

            return 'J({0},{1},{2})'.format(self._a, self._b, self._c)


    ## Renvoie le 1er argument de l'instruction
    def a(self):
        """Renvoie le 1er argument de l'instruction

        Result: naturel

        O() = 1"""
        return self._a


    ## Renvoie le 2e argument de l'instruction (ou lve une exception s'il n'y en pas)
    def b(self):
        """Renvoie le 2e argument de l'instruction
        (ou lve une exception s'il n'y en pas)

        Result: naturel

        O() = 1"""
        if self._inst >= T:
            return self._b
        else:
            raise 'Not 2nd argument'


    ## Renvoie le 3e argument de l'instruction (ou lve une exception s'il n'y en pas)
    def c(self):
        """Renvoie le 3e argument de l'instruction
        (ou lve une exception s'il n'y en pas)

        Result: naturel

        O() = 1"""
        if self._inst == J:
            return self._c
        else:
            raise 'Not 3rd argument'


    ## Renvoie le "nombre de Gdel" associ  l'instruction
    def godelnumber(self):
        """Renvoie le "nombre de Gdel" associ  l'instruction,
        cd 2**n * 3**(a - 1) * 5**(b - 1) * 7**(c - 1)
        o n == 0, 1, 2 ou 3 pour instruction == Z, S, T ou J

        Result: naturel >= 1

        O() = ..."""
        if (self.inst() == Z) or (self.inst() == S):  # instruction Z(a) ou S(a)
            return 2**self.inst() * 3**(self.a() - 1)
        elif self.inst() == T:                        # instruction T(a,b)
            return 2**self.inst() * 3**(self.a() - 1) * 5**(self.b() - 1)
        elif self.inst() == J:                        # instruction J(a,b,c)
            return 2**self.inst() * 3**(self.a() - 1) * 5**(self.b() - 1) * 7**(self.c() - 1)
        assert False


    ## Renvoie le code associ  l'instruction
    def inst(self):
        """Renvoie le code associ  l'instruction

        Result: naturel

        O() = 1"""
        return self._inst



## Programme (suite de UrmCutlandInstruction) pour la machine virtuelle UrmCutland
class UrmCutlandProg:
    """Programme (suite de UrmCutlandInstruction)
    pour la machine virtuelle UrmCutland"""

    ## self == value ?
    def __eq__(self, value):
        """Renvoie True si self reprsente le mme programme que value
        (cd si exactement la mme suite d'instructions),
        False sinon

        Result: bool

        O() = ..."""
        if (not isinstance(value, UrmCutlandProg)) or (len(self) != len(value)):
            return False
        for i in range(1, len(self) + 1):
            assert isinstance(self[i], UrmCutlandInstruction), type(self[i])
            assert isinstance(value[i], UrmCutlandInstruction), type(value[i])

            if self[i] != value[i]:
                return False
        return True


    ## Renvoie l'instruction numro n
    def __getitem__(self, n):
        """Renvoie l'instruction numro n

        Pre: n: naturel >= 1

        Result: UrmCutlandInstruction

        O() = ..."""
        assert DSPython.natural_is(n), (type(n), n)
        assert n >= 1, n

        return self._insts[n - 1]


    ## Initialise le programme pour la machine virtuelle UrmCutland
    def __init__(self, insts):
        """Initialise le programme pour la machine virtuelle UrmCutland
        Si insts est une squence de UrmCutlandInstruction
        alors initialise simplement le programme
              avec cette squence d'instructions.

        Si insts est un string ou squence de string :

        Si une instruction est incorrecte
        alors une exception est leve.

        Alphabet stricte:
          'Z', 'S', 'T', 'J',
          '(', ')', ', ',
          '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'

        De plus '#' dsigne un commentaire
          (dans lequel tous les caractres sont permis)
          jusqu' la fin du string ou une fin de ligne,
          et est alors ignor
        Tous les espaces ' ', tabulations '\t'
          et  la ligne '\n' sont ignors

        Instructions permises : 'Z(a)'
                                'S(a)'
                                'T(a,b)'
                                'J(a,b,c)'
          o a, b et c reprsentent des naturels >= 1
                       crits en base 10

        Pseudo-instruction include : 'i(prog)'
          o prog dsigne une variable contenant
             un programme qui sera inclus
             (tous les numros d'instructions de prog
              sont dcals en consquence)

        Un numro de ligne
          peut prcder une (pseudo-)instruction : 'n)'
          o n reprsente un naturel >= 1 crit en base 10
          (dans ce cas il doit tre plus grand
           que les numros de ligne dj traits)
          (si des numros d'instructions sont passs
           alors ajoute des instructions neutres 'T(1,1)'
                 pour complter les trous)

        Pre: insts: squence de UrmCutlandInstruction
                    ou string ou squence de string

        O() = ..."""
        assert isinstance(insts, str) or isinstance(insts, list) or isinstance(insts, tuple), \
               type(insts)

        if ((isinstance(insts, list) or isinstance(insts, tuple))
            and ((len(insts) == 0)
                 or isinstance(insts[0],
                               UrmCutlandInstruction))) :  # squence de UrmCutlandInstruction
            if __debug__:
                for inst in insts:
                    assert isinstance(inst, UrmCutlandInstruction), type(inst)

            self._insts = list(insts)  # suite des UrmCutlandInstruction
        else:                                                        # source texte
            # Transforme insts en une liste de string (sans '\n')
            if isinstance(insts, list) or isinstance(insts, tuple):
                insts = '\n'.join(insts)
            insts = insts.split('\n')

            l = []  # liste des instructions dj traites
            t = ''  # string du travail en cours
            num = 1  # numro de l'instruction courante
            already_num = False  # True ssi dj un numro spcifi explicitement
                                 #                    pour l'instruction courante

            ## Pour chaque string (lignes numrotes  partir de 1)
            for i in range(1, len(insts) + 1):
                assert isinstance(insts[i - 1], str), (i, type(insts[i - 1]), insts[i - 1])

                comment = False  # True ssi on est dans un commentaire

                ## Pour chaque caractre (colonnes numrotes  partir de 0)
                for j in range(len(insts[i - 1])):
                    assert insts[i - 1][j] != '\n', (i, j, insts[i - 1][j])

                    c = insts[i - 1][j]  # caractre courant
                    if comment:
                        if c == '\n':
                            comment = False
                        continue

                    if (c != ' ') and (c != '\t'):  # vite les "espaces"
                        if c == '#':    # dbut de commentaire
                            comment = True
                        elif c != ')':  # accumule les caractres jusqu' un ')')
                            if (t[:1] != 'i') and not (c in 'ZSTJi(,0123456789'):
                                raise ("bad character '{0}' in line {1} column {2}: '{3}'"
                                       .format(c, i, j, t))
                            t += c
                        else:           # ')', fin d'un numro d'instruction
                                        #          ou d'une (pseudo)-instruction
                            if t == '':
                                raise "closed ')' with nothing in line {0} column {1}".format(i, j)
                            if t[0] in 'ZSTJ':  # c'est une instruction
                                try:
                                    inst = UrmCutlandInstruction(t + ')')
                                except:
                                    raise ("bad instruction '{0})' in line {1} column {2}"
                                           .format(t, i, j))
                                if len(l) + 1 < num:  # si il faut complter avec des NOTHING
                                    l += [NOTHING]*(num - 1 - len(l))
                                l.append(inst)
                                num += 1
                                already_num = False
                            elif t[0] == 'i':   # c'est la pseudo-instruction i
                                if (len(t) >= 2) and (t[1] != '('):
                                    raise ("not opening '(' in line {0} column {1}: '{2})'"
                                           .format(i, j, t))
                                if len(l) + 1 < num:  # si il faut complter avec des NOTHING
                                    l += [NOTHING]*(num - 1- len(l))
                                try:
                                    include = eval(t[2:])
                                except:
                                    raise ("bad include program '{0})' in line {1} column {2}"
                                           .format(t, i, j))
                                if not (isinstance(include, UrmCutlandProg)):
                                    raise ("not valid include program '{0})' in line {1} column {2}"
                                           .format(t, i, j))
                                for inst in include:
                                    if not isinstance(inst, UrmCutlandInstruction):
                                        raise ("not valid instruction in include program '{0})'"
                                               + ' in line {1} column {2}'
                                               .format(t, i, j))
                                l += include.shiftInstrNum(num - 1)
                                num += len(include)
                                already_num = False
                            else:               # c'est un numro d'instruction
                                if already_num:
                                    raise ('already number {0} to this instruction:'
                                           + " '{1})' in line {2} column {3}"
                                           .format(num, t, i, j))
                                try:
                                    num = int(t)
                                except:
                                    raise ("bad number instruction '{0})'"
                                           + " in line {1} column {2} : '{3}'"
                                           .format(t, i, j, insts[i - 1]))
                                if len(l) > num:
                                    raise ('already number instruction {0} (greater: {1})'
                                           + " in line {2} column {3}: '{4}'"
                                           .format(num, len(l) - 1, i, j, insts[i - 1]))
                                already_num = True
                            t = ''

            if t != '':
                raise "piece of instruction '{0}'".format(t)
            if already_num:
                raise 'number instruction {0} without instruction'.format(num)
            self._insts = l  # suite des UrmCutlandInstruction


    ## Itre sur les instructions du programme
    def __iter__(self):
        """Itre sur les instructions du programme

        O() = 1"""
        for inst in self._insts:
            yield inst


    ## Nombre d'instructions du programme
    def __len__(self):
        """Renvoie le nombre d'instructions du programme

        Result: naturel

        O() = 1"""
        return len(self._insts)


    ## self != value ?
    def __ne__(self, value):
        """Renvoie True si self reprsente un programme diffrent de value
        False sinon

        Result: bool

        O() = ..."""
        return not (value == value)


    ## Initialise la valeur de l'instruction numro n  value
    def __setitem__(self, n, value):
        """Initialise la valeur de l'instruction numro n  value

        Pre: n: naturel >= 1
             value: UrmCutlandInstruction

        O() = ..."""
        assert DSPython.natural_is(n), (type(n), n)
        assert n >= 1, n
        assert isinstance(value, UrmCutlandInstruction), type(value)

        self._insts[n - 1] = value


    ## Renvoie dans un string la suite des instructions du programme
    def __str__(self):
        """Renvoie dans un string
        la suite des instructions du programme
        spare par '\n'

        Result: string

        O() = ..."""
        return '\n'.join([str(inst) for inst in self._insts])


    ##\brief Renvoie une copie du programme dont tous les numros d'instructions == c
    # mentionns dans les instructions J sont changs en new
    def changeInstrNum(self, c, new):
        """Renvoie une copie du programme
        dont tous les numros d'instructions == c
        mentionns dans les instructions J sont changs en new

        Pre: c: naturel >= 1
             new: naturel >= 1

        Result: UrmCutlandProg

        O() = ..."""
        assert DSPython.natural_is(c), (type(c), c)
        assert c >= 1, c
        assert DSPython.natural_is(new), (type(new), new)
        assert new >= 1, new

        l = []  # liste des instructions dj traites

        ## Pour chaque instruction
        for inst in self:
            assert isinstance(inst, UrmCutlandInstruction), type(inst)

            l.append(UrmCutlandInstruction(inst.inst(), inst.a(), inst.b(), new)
                     if (inst.inst() == J) and (inst.c() == c)
                     else inst)

        return UrmCutlandProg(l)


    ## Renvoie le code source correspondant au programme
    def decompile(self, skip=True, num=') ', concat='\n'):
        """Renvoie le code source correspondant au programme
        Si skip == True
          alors passe les instructions T(a,a) avec a naturel
                (dans ce cas il faut num != None)
        Si num != None
          alors chaque instruction est prcde
                de son numro puis de num
        Si concat == None
          alors renvoie une liste de string,
                chacune contenant une instruction
          sinon concatne toutes les instructions
                en un seul string spares par concat

        Pre: skip: bool
             num: None ou string compose de ' ', '\t' ou '\n'
             concat: None ou string compose de ' ', '\t' ou '\n'

        Result: string ou liste de string

        O() = ..."""
        assert ((num == None) and not skip) or isinstance(num, str), (type(num), num, skip)
        assert (concat == None) or isinstance(concat, str), (type(concat), concat)

        n = 1  # numro d'instruction
        l = []  # liste de string des instructions

        ## Pour chaque instruction
        for inst in self:
            assert isinstance(inst, UrmCutlandInstruction), type(inst)

            if (not skip) or (inst.inst() != T) or (inst.a() != inst.b()):
                l.append(str(inst) if num == None
                         else '{0}{1}{2}'.format(n, num, inst))
            n += 1

        return (l if concat == None
                else concat.join(l))


    ## Renvoie le "nombre de Gdel" associ au programme
    def godelnumber(self):
        """Renvoie le "nombre de Gdel" associ au programme,
        cd 2**g_1 * 3**g_2 * 5**g_3 * 7**g_4 * ... * P_k**g_k
        o g_i est le "nombre de Gdel" de l'instruction numro i

        Result: naturel >= 1

        O() = ..."""
        l = []
        for inst in self._insts:
            l.append(inst.godelnumber())
        return factors.godelnumber(l)


    ##???
    def read(f):# ou dans __init__ ???
        """???

        Pre: f: fichier (texte) ouvert en lecture

        O() = ..."""
        assert isinstance(f, file), f

        raise NotImplementedError


    ##\brief Renvoie une copie du programme dont tous les numros d'instructions
    # mentionns dans les instructions J sont dcals de offset
    def shiftInstrNum(self, offset, underflow=1):
        """Renvoie une copie du programme dont tous les numros d'instructions
        mentionns dans les instructions J sont dcals de offset.
        Si une des valeurs devient < 1
        alors elle est remplace par underflow

        Pre: offset: int
             underflow: naturel >= 1

        Result: UrmCutlandProg

        O() = ..."""
        assert isinstance(offset, int), (type(offset), offset)
        assert DSPython.natural_is(underflow), (type(underflow), underflow)
        assert underflow >= 1, underflow

        l = []  # liste des instructions dj traites

        ## Pour chaque instruction
        for inst in self:
            assert isinstance(inst, UrmCutlandInstruction), type(inst)

            l.append(UrmCutlandInstruction(inst.inst(), inst.a(), inst.b(),
                                           (inst.c() + offset if inst.c() + offset > 0
                                            else underflow)) if inst.inst() == J
                     else inst)

        return UrmCutlandProg(l)


    ##???
    def write(f):
        """???

        Pre: f: fichier (texte) ouvert en lecture

        O() = ..."""
        assert isinstance(f, file), f

        raise NotImplementedError



##\brief Machine virtuelle UrmCutland :
# Unlimited Register Machine de <span style="font-variant:small-caps">Cutland</span>
# (variante de la machine de
# <span style="font-variant:small-caps">Shepherdson</span>
# &ndash; <span style="font-variant:small-caps">Sturgis</span>)

##\details
# Machine virtuelle constitue d'une infinit (potentielle) de registres numrotes 1, 2, 3, ...\n
# Chaque registre contient un naturel (initialement 0).
#
# Un programme pour cette machine virtuelle
#   est une suite finie d'instructions numrotes 1, 2, 3, ..., k
#
# Liste des instructions (a, b et c tant des naturels \htmlonly&ge;\endhtmlonly 1) :
# \verbatim
# Zero       Z(a)     : initialiser le registre numro a avec la valeur 0
# Successor  S(a)     : incrmenter la valeur du registre numro a
# Transfer   T(a,b)   : initialiser le registre numro b avec la valeur du registre numro a
# Jump       J(a,b,c) : si la valeur du registre numro a
#                       == la valeur du registre numro b
#                       alors reprendre  partir de l'instruction numro c
#                             (si elle existe, sinon arrter le programme)
#                       sinon ne rien faire\endverbatim
#
# Cf. <i lang="en">Computability: A introduction to recursive function theory</i>
#      (Nigel J. <span style="font-variant:small-caps">Cutland</span>,
#       Cambridge University Press, 2000)
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Theo" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Theo</tt></a>\endhtmlonly
#
# Pour une prsentation image par un "coffee bar"
# cf.
# \htmlonly
# <a lang="en" href="http://www.mail-archive.com/everything-list@eskimo.com/msg14085.html"
#   target="_blank"><i>Re: Key Post 1,
#   toward <span style="font-variant:small-caps">Church</span> Thesis and Lobian machine</i></a>
# \endhtmlonly
# (Bruno <span style="font-variant:small-caps">Marchal</span>)
class UrmCutland:
    """Machine virtuelle UrmCutland :
    Unlimited Register Machine de Cutland
    (variante de la machine de Shepherdson - Sturgis)"""

    ## Renvoie la valeur du registre numro n
    def __getitem__(self, n):
        """Renvoie la valeur du registre numro n

        Pre: n: naturel >= 1

        Result: naturel

        O() = ..."""
        assert DSPython.natural_is(n), (type(n), n)
        assert n >= 1, n

        self._extend(n)
        return self._regs[n - 1]


    ## Initialise la machine virtuelle UrmCutland
    def __init__(self, init=(), nb=1):
        """Initialise la machine virtuelle UrmCutland
        en initialisant ses premiers registres.

        Si init est un naturel
        alors initialise les nb premiers registres avec la valeur init.
        Si init est une squence
        alors initialise les premiers registres
              avec les valeurs de init prises nb fois

        Pre: init: naturel ou squence de naturels
             nb: naturel

        O() = nb"""
        assert DSPython.natural_is(init) or isinstance(init, list) or isinstance(init, tuple), \
               (type(init), init)
        assert DSPython.natural_is(nb), (type(nb), inb)

        # self._regs: liste de valeur des registres
        if DSPython.natural_is(init):  # init est un naturel
            self._regs = [init]*nb
        else:                          # init est une squence
            if __debug__:
                for n in init:
                    assert DSPython.natural_is(n), (n, init)

            self._regs = list(init)*nb


    ## Itre sur les registres de la machine virtuelle
    def __iter__(self):
        """Itre sur les registres de la machine virtuelle

        O() = 1"""
        for reg in self._regs:
            yield reg


    ## Nombre de registres utiliss
    def __len__(self):
        """Renvoie le nombre de registres utiliss
        (en fait le plus grand numro de registre
        parmi les registres ayant t utiliss)

        Result: naturel

        O() = 1"""
        return len(self._regs)


    ## Initialise la valeur du registre numro n  value
    def __setitem__(self, n, value):
        """Initialise la valeur du registre numro n  value

        Pre: n: naturel >= 1
             value: naturel

        O() = ..."""
        assert DSPython.natural_is(n), (type(n), n)
        assert n >= 1, n
        assert DSPython.natural_is(value), (type(value), value)

        self._extend(n)
        self._regs[n - 1] = value


    ## Renvoie dans un string la suite des valeurs des registres numrots de 1  len()
    def __str__(self):
        """Renvoie dans un string la suite des valeurs
        des registres numrots de 1  len()

        Result: string

        O() = ..."""
        return '[' + ' '.join([str(n) for n in self._regs]) + ']'


    ## tend self._regs si ncessaire
    def _extend(self, n):
        """Si le registre numro n n'est pas encore dfini dans self._regs
        alors tend self._regs avec des 0 et renvoie True
        sinon renvoie False

        Pre: n: naturel >= 1

        Result: bool

        O() = ..."""
        assert DSPython.natural_is(n), (type(n), n)
        assert n >= 1, n

        if n > len(self._regs):  # il faut tendre
            self._regs += [0]*(n - len(self._regs))
            return True
        else:
            return False


    ## Renvoie le "nombre de Gdel" associ  la liste des registres
    def godelnumber(self):
        """Renvoie le "nombre de Gdel" associ  la liste des registres,
        cd 2**r_1 * 3**r_2 * 5**r_3 * 7**r_4 * ...
        o r_i est la valeur du registre numro i

        Result: naturel >= 1

        O() = ..."""
        return factors.godelnumber(self._regs)


    ## Registre numro a == registre numro b ?
    def J_test(self, a, b):
        """Renvoie True
        si les registres numro a et numro b contiennent la mme valeur,
        False sinon

        Pre: a: naturel >= 1
             b: naturel >= 1

        Result: bool

        O() = ..."""
        assert DSPython.natural_is(a), (type(a), a)
        assert a >= 1, a
        assert DSPython.natural_is(b), (type(b), b)
        assert b >= 1, b

        self._extend(max(a, b))
        return self._regs[a - 1] == self._regs[b - 1]


    ##\brief Initialise avec les valeurs lues dans le fichier f
    # les nb registres  partir du registre numro n
    # (si nb != None alors lit au plus nb valeurs)
    def read(f, n=1, nb=None):# ou dans __init__ ???
        """Initialise avec les valeurs lues dans le fichier f
        les registres  partir du registre numro n
        (si nb != None alors lit au plus nb valeurs)

        Pre: f: fichier (texte) ouvert en lecture
                compos uniquement des caractres '0'..'9', ' ', ',', ';', '\t', 'n'
                ???
             n: naturel >= 1
             nb: None ou naturel

        O() = ..."""
        assert isinstance(f, file), f
        assert DSPython.natural_is(n), (type(n), n)
        assert n >= 1, n
        assert (nb == None) or _init__.natural_is(nb), (type(nb), nb)

        for line in f:
            line = line.replace('\t', ' ')
            line = line.replace('\n', ' ')
            line = line.replace(';', ' ')
            line = line.replace(',', ' ')
            l = ' '.split(line.replace('  ', ' '))
            for s in l:
                if nb == 0:
                    return
                number = int(s)
                if number <= 0:
                    raise 'number readed dont >= 1'
                self[n] = number
                n += 1
                if nb != None:
                    nb -= 1


    ##\brief Excute au plus nb instructions (ou toutes si nb == None) de prog
    #  partir de l'instruction numro i
    # et renvoie (numro de l'instruction finale, nombre d'instructions excutes)
    def run(self, prog, i=1, nb=None):
        """Excute au plus nb instructions (ou toutes si nb == None) de prog
         partir de l'instruction numro i
        et renvoie (numro de l'instruction finale,
                    nombre d'instructions excutes)
        (le numro de l'instruction finale est en fait
        soit le numro aprs la dernire instruction excute,
        soit le numro d'une instruction qui n'existe pas
             mentionn par la dernire instruction J excute)

        Pre: prog: UrmCutlandProg
             i: naturel >= 1
             nb: None ou naturel

        Result: (naturel >= 1, naturel)

        O() = ..."""
        assert isinstance(prog, UrmCutlandProg), type(prog)
        assert DSPython.natural_is(i), (type(i), i)
        assert i >= 1, i
        assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

        k = 0  # nombre d'instructions excutes

        ## Tant qu'il reste des instructions  excuter
        while (i <= len(prog)) and ((nb == None) or (k < nb)):
            assert isinstance(prog[i], UrmCutlandInstruction), (i, type(prog[i]), prog[i])

            k += 1
            instruction = prog[i]
            i += 1
            if instruction.inst() == Z:                          # Z(a)
                self.Z(instruction.a())
            elif instruction.inst() == S:                        # S(a)
                self.S(instruction.a())
            elif instruction.inst() == T:                        # T(a,b)
                self.T(instruction.a(), instruction.b())
            elif self.J_test(instruction.a(), instruction.b()):  # J(a,b,c) avec reg a == reg b
                assert instruction.inst() == J, instruction.inst()

                i = instruction.c()

        return (i, k)


    ## Instruction Successor : ajoute 1 dans le registre numro a
    def S(self, a):
        """Instruction Successor : ajoute 1 dans le registre numro a

        Pre: a: naturel >= 1

        O() = ..."""
        assert DSPython.natural_is(a), (type(a), a)
        assert a >= 1, a

        self._extend(a)
        self._regs[a - 1] += 1


    ##\brief Instruction Transfer :
    # initialise le registre numro b avec la valeur du registre numro a
    def T(self, a, b):
        """Instruction Transfer :
        initialise le registre numro b
        avec la valeur du registre numro a

        Pre: a: naturel >= 1
             b: naturel >= 1

        O() = ..."""
        assert DSPython.natural_is(a), (type(a), a)
        assert a >= 1, a
        assert DSPython.natural_is(b), (type(b), b)
        assert b >= 1, b

        self._extend(max(a, b))
        self._regs[b - 1] = self._regs[a - 1]


    ##???
    def write(f):
        """???

        Pre: f: fichier (texte) ouvert en lecture

        O() = ..."""
        assert isinstance(f, file), f

        raise NotImplementedError


    ## Instruction Zero : initialise le registre numro a  0
    def Z(self, a):
        """Instruction Zero : initialise le registre numro a  0

        Pre: a: naturel >= 1

        O() = ..."""
        assert DSPython.natural_is(a), (type(a), a)
        assert a >= 1, a

        self._extend(a)
        self._regs[a - 1] = 0



# ############
# Constantes #
##############
## Code associ  l'instruction Z
Z = 0

## Code associ  l'instruction S
S = 1

## Code associ  l'instruction T
T = 2

## Code associ  l'instruction J
J = 3


## Instruction qui ne fait rien
NOTHING = UrmCutlandInstruction(T, 1, 1)



##\brief Programme ngation boolenne de n (cd si n == 0 alors 1 sinon 0)\verbatim
# Pre:  | n                       |
# Post: | ngation boolenne de n |
# Bord: |                         | 0 |
# O() = 1\endverbatim\hideinitializer
NOT = UrmCutlandProg("""
1) Z(2)
2) J(1,2,5)
3) Z(1)
4) J(1,1,6)

5) S(1)
""")

##\brief Programme a <= b (cd si a <= b alors 1 sinon 0)\verbatim
# Pre:  | a      | b |
# Post: | a <= b |
# Bord: |        | b | min(a, b) |
# O() = min(a, b)\endverbatim\hideinitializer
LESSOREQ = UrmCutlandProg("""
             # INV: 0  <= a <= b   ou 0  <= b <= a
1) Z(3)      # INV: _3 <= a <= b   ou _3 <= b <= a

# Boucle _3 de 0  min(a, b) :
2) J(1,3,6)  # INV: _3 <  a <= b   ou _3 <= b <= a
3) J(2,3,9)  # INV: _3 <  a <= b   ou _3 <  b <= a
4) S(3)
5) J(1,1,2)

6) Z(1)      # INV: _3 == a <= b
7) S(1)
8) J(1,1,10)

9) Z(1)      # INV:                   _3 == b <  a
""")

##\brief Programme a < b (cd si a < b alors 1 sinon 0)\verbatim
# Pre:  | a     | b |
# Post: | a < b |
# Bord: |       | b |           | si a == b,
#       |       | b | min(a, b) | sinon
# O() = min(a, b)\endverbatim\hideinitializer
LESS = UrmCutlandProg("""
 1) J(1,2,10)  # INV: 0  <= a < b   ou 0  <= b < a
 2) Z(3)       # INV: _3 <= a < b   ou _3 <= b < a

# Boucle _3 de 0  min(a, b) :
 3) J(1,3,7)   # INV: _3 <  a < b   ou _3 <= b < a
 4) J(2,3,10)  # INV: _3 <  a < b   ou _3 <  b < a
 5) S(3)
 6) J(1,1,3)

 7) Z(1)       # INV: _3 == a < b
 8) S(1)
 9) J(1,1,11)

10) Z(1)      # INV:                  _3 == b < a
""")

##\brief Programme a >= b (cd si a >= b alors 1 sinon 0)\verbatim
# Pre:  | a      | b |
# Post: | a >= b |
# Bord: |        | b | min(a, b) |
# O() = min(a, b)\endverbatim\hideinitializer
GREATOREQ = UrmCutlandProg("""
             # INV: 0  <= a <= b   ou 0  <= b <= a
1) Z(3)      # INV: _3 <= a <= b   ou _3 <= b <= a

# Boucle _3 de 0  min(a, b) :
2) J(2,3,6)  # INV: _3 <= a <= b   ou _3 <  b <= a
3) J(1,3,9)  # INV: _3 <  a <= b   ou _3 <  b <= a
4) S(3)
5) J(1,1,2)

6) Z(1)      # INV:                   _3 == b <= a
7) S(1)
8) J(1,1,10)

9) Z(1)      # INV: _3 == a <  b
""")

##\brief Programme a > b (cd si a > b alors 1 sinon 0)\verbatim
# Pre:  | a     | b |
# Post: | a > b |
# Bord: |       | b |           | si a == b,
#       |       | b | min(a, b) | sinon
# O() = min(a, b)\endverbatim\hideinitializer
GREAT = UrmCutlandProg("""
 1) J(1,2,10)  # INV: 0  <= a < b   ou 0  <= b < a
 2) Z(3)       # INV: _3 <= a < b   ou _3 <= b < a

# Boucle _3 de 0  min(a, b) :
 3) J(2,3,7)   # INV: _3 <= a < b   ou _3 <  b < a
 4) J(1,3,10)  # INV: _3 <  a < b   ou _3 <  b < a
 5) S(3)
 6) J(1,1,3)

 7) Z(1)      # INV:                  _3 == b < a
 8) S(1)
 9) J(1,1,11)

10) Z(1)      # INV: _3 == a < b
""")



##\brief Programme minimum de a et b : min(a, b)\verbatim
# Pre:  | a         | b |
# Post: | min(a, b) |
# Bord: |           | b | min(a, b) |
# O() = min(a, b)\endverbatim\hideinitializer
MIN = UrmCutlandProg("""
             # INV: 0  <= a <= b   ou 0  <= b <= a
1) Z(3)      # INV: _3 <= a <= b   ou _3 <= b <= a

# Boucle _3 de 0  min(a, b) :
2) J(1,3,6)  # INV: _3 <  a <= b   ou _3 <= b <= a
3) J(2,3,6)  # INV: _3 <  a <= b   ou _3 <  b <= a
4) S(3)
5) J(1,1,2)

6) T(3,1)    # INV: _3 == a <= b   ou _3 == b <  a
""")

##\brief Programme maximum de a et b : max(a, b)\verbatim
# Pre:  | a         | b |
# Post: | max(a, b) |
# Bord: |           | b | min(a, b) |
# O() = min(a, b)\endverbatim\hideinitializer
MAX = UrmCutlandProg("""
             # INV: 0  <= a <= b   ou 0  <= b <= a
1) Z(3)      # INV: _3 <= a <= b   ou _3 <= b <= a

# Boucle _3 de 0  min(a, b) :
2) J(1,3,6)  # INV: _3 <  a <= b   ou _3 <= b <= a
3) J(2,3,7)  # INV: _3 <  a <= b   ou _3 <  b <= a
4) S(3)
5) J(1,1,2)

6) T(2,1)    # INV: _3 == a <= b
""")



##\brief Programme n + 1\verbatim
# Pre:  | n |
# Post: | n |
# O() = 1\endverbatim
INC = UrmCutlandProg((UrmCutlandInstruction(S, 1), ))

##\brief Programme n - 1\verbatim
# Pre:  | n     |
# Post: | n - 1 |         si n > 0, sinon ne s'arrte jamais
# Bord: |       | n | n |
# O() = n\endverbatim\hideinitializer
DEC = UrmCutlandProg("""
1) T(1,3)
2) Z(1)
3) Z(2)
4) S(2)

# Boucle _2 de 1  n :
5) J(2,3,9)
6) S(1)
7) S(2)
8) J(1,1,5)
""")

##\brief Programme a + b\verbatim
# Pre:  | a     | b |
# Post: | a + b |
# Bord: |       | b | b |
# O() = b\endverbatim\hideinitializer
ADD = UrmCutlandProg("""
1) Z(3)

# Boucle _3 de 0  b :
2) J(2,3,6)
3) S(1)
4) S(3)
5) J(1,1,2)
""")

##\brief Programme a - b\verbatim
# Pre:  | a     | b |
# Post: | a - b |         si a >= b, sinon ne s'arrte jamais
# Bord: |       | a | a |
# O() = a - b\endverbatim\hideinitializer
SUB = UrmCutlandProg("""
1) T(1,3)
2) Z(1)

# Boucle _2 de b  a :
3) J(2,3,7)
4) S(1)
5) S(2)
6) J(1,1,3)
""")

##\brief Programme a*b\verbatim
# Pre:  | a   | b |
# Post: | a*b |
# Bord: |     | a |   | b | b | si b == 0,
#       |     | a | a | b | b | sinon
# O() = a*b\endverbatim\hideinitializer
MUL = UrmCutlandProg("""
 1) T(2,4)
 2) T(1,2)
 3) Z(1)
 4) Z(5)

# Boucle _5 de 0  b :
 5) J(4,5,13)
 6) i(ADD)
11) S(5)
12) J(1,1,5)
""")

##\brief Programme division euclidienne : quotient entier a//b et reste a%b\verbatim
# Pre:  | a    | b   |
# Post: | a//b | a%b |             si b != 0, sinon ne s'arrte jamais
# Bord: |      |     | a | b | a |
# O() = a\endverbatim\hideinitializer
DIV = UrmCutlandProg("""
 1) T(1,3)
 2) T(2,4)
 3) Z(1)
 4) J(1,2,4)  # si b == 0
 5) Z(2)
 6) Z(5)

# Boucle _5 de 0  a :
 7) J(3,5,15)
 8) S(2)  # reste += 1
 9) S(5)
10) J(2,4,12)  # si reste == b
11) J(1,1,7)

12) S(1)  # quotient += 1
13) Z(2)  # reste = 0
14) J(1,1,7)
""")

##\brief Programme a<sup>b</sup> == a**b\verbatim
# Pre:  | a    | b |
# Post: | a**b |
# Bord: |      | a | 0          | 0 | 0 | b | b | si b == 0,
#       |      | a |            | a | a | b | b | si b > 0 et a == 0,
#       |      | a | a**(b - 1) | a | a | b | b | sinon
# O() = a**b\endverbatim\hideinitializer
POW = UrmCutlandProg("""
 1) T(2,6)
 2) T(1,2)
 3) Z(1)
 4) S(1)
 5) Z(7)

# Boucle _7 de 0  b :
 6) J(6,7,22)
 7) i(MUL)
19) S(7)
20) T(4,2)
21) J(1,1,6)
""")


##\brief Programme plus grand commun diviseur : gcd(a, b)\verbatim
# Pre:  | a         | b |
# Post: | gcd(a, b) |   |                               si a != 0 ou a!= b, sinon ne s'arrte jamais
# Bord: |           | 0 | 0           | gcd(a, b) | 0 | si a == 0 ou b == 0,
#       |           | 0 | k*gcd(a, b) | gcd(a, b) | 0 | sinon
#                         (avec k naturel >= 1 et gcd(a, b) <= k*gcd(a, b) <= max(a, b))
# O() = ...\endverbatim\hideinitializer
GCD = UrmCutlandProg("""
# Boucle tant que _2 > 0 : (algorithme d'Euclide)
 1) Z(5)
 2) J(2,5,19)
 3) i(DIV)  # | a//b | a%b | a | b | a |
17) T(4,1)  # | b    | a%b | a | b | a |
18) J(1,1,1)
19) J(1,5,19)
""")

##\brief Programme n<sup>e</sup> et (n + 1)<sup>e</sup> nombres de Fibonacci :
# F<sub>n</sub> et F<sub>n + 1</sub>\verbatim
# Pre:  | n   |
# Post: | F_n | F_(n+1) |
# Bord: |     |         |     | n | n | si n == 0,
#       |     |         | F_n | n | n | sinon
# O() = F_(n+1)\endverbatim\hideinitializer
FIBONACCI2 = UrmCutlandProg("""
 1) T(1,4)
 2) Z(1)
 3) Z(2)
 4) S(2)
 5) Z(5)

# Boucle _5 de 0  n :
 6) J(4,5,16)
 7) i(ADD)
12) T(1,2)
13) T(3,1)
14) S(5)
15) J(1,1,6)
""")

##\brief Programme n!\verbatim
# Pre:  | n  |
# Post: | n! |
# Bord: |    | n |          |   |   | n | si n == 0,
#       |    | n | (n - 1)! | n | n | n | sinon
# O() = n!\endverbatim\hideinitializer
FACTORIAL = UrmCutlandProg("""
 1) T(1,6)
 2) Z(1)
 3) S(1)
 4) Z(2)

# Boucle _2 de 0  n :
 5) J(2,6,21)
 6) S(2)
 7) i(MUL)
19) T(4,2)
20) J(1,1,5)
""")



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import fractions, math, sys

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        import DSPython.debug as debug
        import DSPython.nat32 as nat32

        debug.test_begin(VERSION, __debug__)

        print('Z == {0}'.format(Z)); sys.stdout.flush()
        assert DSPython.natural_is(Z), (type(Z), Z)
        assert Z == 0, Z

        print('S == {0}'.format(S)); sys.stdout.flush()
        assert DSPython.natural_is(S), (type(S), S)
        assert S == 1, S

        print('T == {0}'.format(T)); sys.stdout.flush()
        assert DSPython.natural_is(T), (type(T), T)
        assert T == 2, T

        print('J == {0}'.format(J)); sys.stdout.flush()
        assert DSPython.natural_is(J), (type(J), J)
        assert J == 3, J



        print()
        print('UrmCutlandInstruction()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.__eq__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.__ne__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.__str__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.a()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.b()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.c()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.godelnumber()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandInstruction.inst()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print("NOTHING == '%s'" % NOTHING); sys.stdout.flush()
        assert isinstance(NOTHING, UrmCutlandInstruction), type(NOTHING)
        assert str(NOTHING) == 'T(1,1)', str(NOTHING)



        print()
        print('UrmCutlandProg()...', end=''); sys.stdout.flush()
        print('???', end='')
        prog = UrmCutlandProg((' \n    \t  ',
                               """   \t
                               \t   """))
        assert prog == UrmCutlandProg([]), [str(i) for i in prog]

        prog = UrmCutlandProg('1) S(1)')
        assert prog == INC, [str(i) for i in prog]

        prog = UrmCutlandProg(('Z ( \t  \n   3  )#####',
                               ' J(2, 3,  000 # comment',
                               '  00   6  ) S(1) # comment \n S(3) J(1,1,\t2)'))
        assert prog == ADD, [str(i) for i in prog]

        prog = UrmCutlandProg("""
            T(2,4)  # initialisation
            T(1,2)
            Z(1)
            Z(5)
            J(4,5,13)
            Z(3)  # [ADD
            J(2,3,11)
            S(1)
            S(3)
            J(1,1,7)  # ADD]
            S(5)  # boucle sur ADD
            J(1,1,5)""")
        assert prog == MUL, [str(i) for i in prog]
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__eq__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__getitem__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__iter__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__len__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__ne__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__setitem__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.__str__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.changeInstrNum()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.decompile()...', end=''); sys.stdout.flush()
        print('???', end='')
        src = ADD.decompile(concat='   ')
        assert src == '1) Z(3)   2) J(2,3,6)   3) S(1)   4) S(3)   5) J(1,1,2)', src

        src = ADD.decompile(concat=None)
        assert src == ['1) Z(3)',
                       '2) J(2,3,6)',
                       '3) S(1)',
                       '4) S(3)',
                       '5) J(1,1,2)'], src

        src = MUL.decompile(num=') \t')
        assert src == """1) \tT(2,4)
2) \tT(1,2)
3) \tZ(1)
4) \tZ(5)
5) \tJ(4,5,13)
6) \tZ(3)
7) \tJ(2,3,11)
8) \tS(1)
9) \tS(3)
10) \tJ(1,1,7)
11) \tS(5)
12) \tJ(1,1,5)""", src
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.godelnumber()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.read()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.shiftInstrNum()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutlandProg.write()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()



        print()
        print('UrmCutland()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.__getitem__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.__iter__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.__len__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.__setitem__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.__str__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland._extend()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.godelnumber()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.J_test()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.read()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.run()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.S()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.T()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.write()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('UrmCutland.Z()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()



        print()
        print('NOT...', end=''); sys.stdout.flush()
        assert isinstance(NOT, UrmCutlandProg), type(NOT)
        for inst in NOT:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for n in range(100):
            m = UrmCutland(init=n)
            (i, nb) = m.run(NOT)
            assert i == len(NOT) + 1, (n, i, len(NOT), nb)
            assert nb == 3 + (n != 0), (n, i, nb)
            assert len(m) == 2, (n, len(m))
            assert m[1] == (not bool(n)), (n, m[1])
            assert m[2] == 0, (n, m[2])
        print('ok'); sys.stdout.flush()


        print('LESSOREQ...', end=''); sys.stdout.flush()
        assert isinstance(LESSOREQ, UrmCutlandProg), type(LESSOREQ)
        for inst in LESSOREQ:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(LESSOREQ)
                assert i == len(LESSOREQ) + 1, (a, b, i, len(LESSOREQ), nb)
                assert nb <= len(LESSOREQ)*(min(a, b) + 1), (a, b, i, len(LESSOREQ), nb,
                                                             len(LESSOREQ)*(min(a, b) + 1))
                assert len(m) == 3, (a, b, len(m))
                assert m[1] == (a <= b), (a, b, m[1], a <= b)
                assert m[2] == b, (a, b, m[2])
                assert m[3] == min(a, b), (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('LESS...', end=''); sys.stdout.flush()
        assert isinstance(LESS, UrmCutlandProg), type(LESS)
        for inst in LESS:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(LESS)
                assert i == len(LESS) + 1, (a, b, i, len(LESS), nb)
                assert nb <= len(LESS)*(min(a, b) + 1), (a, b, i, len(LESS), nb,
                                                         len(LESS)*(min(a, b) + 1))
                assert len(m) == (2 + (a != b)), (a, b, len(m))
                assert m[1] == (a < b), (a, b, m[1], a < b)
                assert m[2] == b, (a, b, m[2])
                if a != b:
                    assert m[3] == min(a, b), (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('GREATOREQ...', end=''); sys.stdout.flush()
        assert isinstance(GREATOREQ, UrmCutlandProg), type(GREATOREQ)
        for inst in GREATOREQ:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(GREATOREQ)
                assert i == len(GREATOREQ) + 1, (a, b, i, len(GREATOREQ), nb)
                assert nb <= len(GREATOREQ)*(min(a, b) + 1), (a, b, i, len(GREATOREQ), nb,
                                                              len(GREATOREQ)*(min(a, b) + 1))
                assert len(m) == 3, (a, b, len(m))
                assert m[1] == (a >= b), (a, b, m[1], a >= b)
                assert m[2] == b, (a, b, m[2])
                assert m[3] == min(a, b), (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('GREAT...', end=''); sys.stdout.flush()
        assert isinstance(GREAT, UrmCutlandProg), type(GREAT)
        for inst in GREAT:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(GREAT)
                assert i == len(GREAT) + 1, (a, b, i, len(GREAT), nb)
                assert nb <= len(GREAT)*(min(a, b) + 1), (a, b, i, len(GREAT), nb,
                                                          len(GREAT)*(min(a, b) + 1))
                assert len(m) == (2 + (a != b)), (a, b, len(m))
                assert m[1] == (a > b), (a, b, m[1], a > b)
                assert m[2] == b, (a, b, m[2])
                if a != b:
                    assert m[3] == min(a, b), (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('MIN...', end=''); sys.stdout.flush()
        assert isinstance(MIN, UrmCutlandProg), type(MIN)
        for inst in MIN:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(MIN)
                assert i == len(MIN) + 1, (a, b, i, len(MIN), nb)
                assert nb <= len(MIN)*(min(a, b) + 1), (a, b, i, len(MIN), nb,
                                                        len(MIN)*(min(a, b) + 1))
                assert len(m) == 3, (a, b, len(m))
                assert m[1] == min(a, b), (a, b, m[1], min(a, b))
                assert m[2] == b, (a, b, m[2])
                assert m[3] == min(a, b), (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('MAX...', end=''); sys.stdout.flush()
        assert isinstance(MAX, UrmCutlandProg), type(MAX)
        for inst in MAX:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(MAX)
                assert i == len(MAX) + 1, (a, b, i, len(MAX), nb)
                assert nb <= len(MAX)*(min(a, b) + 1), (a, b, i, len(MAX), nb,
                                                        len(MAX)*(min(a, b) + 1))
                assert len(m) == 3, (a, b, len(m))
                assert m[1] == max(a, b), (a, b, m[1], max(a, b))
                assert m[2] == b, (a, b, m[2])
                assert m[3] == min(a, b), (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('INC...', end=''); sys.stdout.flush()
        assert isinstance(INC, UrmCutlandProg), type(INC)
        for inst in INC:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for n in range(100):
            m = UrmCutland(init=n)
            (i, nb) = m.run(INC)
            assert i == len(INC) + 1, (n, i, len(INC), nb)
            assert nb == 1, (n, nb)
            assert len(m) == 1, (n, len(m))
            assert m[1] == n + 1, (n, m[1])

            assert m[5] == 0, (n, m[5])
            assert len(m) == 5, (n, len(m))
            assert m[1] == n + 1, (n, m[1])
            assert m[2] == 0, (n, m[2])
            assert m[3] == 0, (n, m[3])
            assert m[4] == 0, (n, m[4])
            assert len(m) == 5, (n, len(m))
        print('ok'); sys.stdout.flush()


        print('DEC...', end=''); sys.stdout.flush()
        assert isinstance(DEC, UrmCutlandProg), type(DEC)
        for inst in DEC:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        m = UrmCutland(init=0)
        (i, nb) = m.run(DEC, nb=10000)
        assert nb == 10000, (i, nb)

        for n in range(1, 100):
            m = UrmCutland(init=n)
            (i, nb) = m.run(DEC)
            assert i == len(DEC) + 1, (n, i, len(DEC), nb)
            assert nb <= len(DEC)*n, (n, i, len(DEC), nb, len(DEC)*(min(a, b) + 1))
            assert len(m) == 3, (n, len(m))
            assert m[1] == n - 1, (n, m[1])
            assert m[2] == n, (n, m[2])
            assert m[3] == n, (n, m[3])

            assert m[5] == 0, (n, m[5])
            assert len(m) == 5, (n, len(m))
            assert m[1] == n - 1, (n, m[1])
            assert m[2] == n, (n, m[2])
            assert m[3] == n, (n, m[3])
            assert m[4] == 0, (n, m[4])
            assert len(m) == 5, (n, len(m))
        print('ok'); sys.stdout.flush()


        print('ADD...', end=''); sys.stdout.flush()
        assert isinstance(ADD, UrmCutlandProg), type(ADD)
        for inst in ADD:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(ADD)
                assert i == len(ADD) + 1, (a, b, i, len(ADD), nb)
                assert nb <= len(ADD)*(b + 1), (a, b, i, len(ADD), nb, len(ADD)*(b + 1))
                assert len(m) == 3, (a, b, len(m))
                assert m[1] == a + b, (a, b, m[1], a + b)
                assert m[2] == b, (a, b, m[2])
                assert m[3] == b, (a, b, m[3])
        print('ok'); sys.stdout.flush()


        print('SUB...', end=''); sys.stdout.flush()
        assert isinstance(SUB, UrmCutlandProg), type(SUB)
        for inst in SUB:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(20):
            for b in range(15):
                m = UrmCutland(init=(a, b))
                if a >= b:
                    (i, nb) = m.run(SUB)
                    assert i == len(SUB) + 1, (a, b, i, len(SUB), nb)
                    assert nb < 100, (a, b, i, nb)
                    assert nb <= len(SUB)*(a - b + 1), (a, b, i, len(SUB), nb,
                                                        len(SUB)*(a - b + 1))
                    assert len(m) == 3, (a, b, len(m))
                    assert m[1] == a - b, (a, b, m[1], a - b)
                    assert m[2] == a, (a, b, m[2])
                    assert m[3] == a, (a, b, m[3])
                elif debug.assertspeed >= debug.ASSERT_NORMAL:
                    (i, nb) = m.run(SUB, nb=1000)
                    assert nb == 1000, (a, b, i, nb)
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('MUL...', end=''); sys.stdout.flush()
        assert isinstance(MUL, UrmCutlandProg), type(MUL)
        for inst in MUL:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(40 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
            for b in range(15 if debug.assertspeed >= debug.ASSERT_NORMAL else 10):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(MUL)
                assert i == len(MUL) + 1, (a, b, i, len(MUL), nb)
                assert nb <= len(MUL)*((a + 1)*b + 1), (a, b, i, len(MUL), nb,
                                                        len(MUL)*((a + 1)*b + 1))
                assert len(m) == 5, (a, b, len(m))
                assert m[1] == a*b, (a, b, m[1], a*b)
                assert m[2] == a, (a, b, m[2])
                if b == 0:
                    assert m[3] == 0, (a, b, m[3])
                else:
                    assert m[3] == a, (a, b, m[3])
                assert m[4] == b, (a, b, m[4])
                assert m[5] == b, (a, b, m[5])
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('DIV...', end=''); sys.stdout.flush()
        assert isinstance(DIV, UrmCutlandProg), type(DIV)
        for inst in DIV:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
            m = UrmCutland(init=(a, 0))
            if debug.assertspeed >= debug.ASSERT_NORMAL:
                (i, nb) = m.run(DIV, nb=5000)
                assert nb == 5000, (a, i, nb)
            for b in range(1, 20):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(DIV)
                assert i == len(DIV) + 1, (a, b, i, len(DIV), nb)
                assert nb < 500, (a, b, i, nb)
                assert nb <= len(DIV)*(a + 1), (a, b, i, len(DIV), nb, len(DIV)*(a + 1))
                assert len(m) == 5, (a, b, len(m))
                assert m[1] == a//b, (a, b, m[1], a//b)
                assert m[2] == a%b, (a, b, m[2], a%b)
                assert m[3] == a, (a, b, m[3])
                assert m[4] == b, (a, b, m[4])
                assert m[5] == a, (a, b, m[5])
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('POW...', end=''); sys.stdout.flush()
        assert isinstance(POW, UrmCutlandProg), type(POW)
        for inst in POW:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(10 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for b in range(6):
                m = UrmCutland(init=(a, b))
                (i, nb) = m.run(POW)
                assert nb <= len(POW)*((a + 2)**(b + 1)), (a, b, i, len(POW), nb,
                                                           len(POW)*((a + 1)**(b + 1)))
                assert i == len(POW) + 1, (a, b, i, len(POW), nb)
                assert len(m) == 7, (a, b, len(m))
                assert m[1] == a**b, (a, b, m[1], a**b)
                assert m[2] == a, (a, b, m[2])
                if b == 0:
                    assert m[3] == 0, (a, b, m[3])
                    assert m[4] == 0, (a, b, m[4])
                    assert m[5] == 0, (a, b, m[5])
                else:
                    if a == 0:
                        assert m[3] == 0, (a, b, m[3])
                    else:
                        assert m[3] == a**(b - 1), (a, b, m[3], a**(b - 1))
                    assert m[4] == a, (a, b, m[4])
                    assert m[5] == a, (a, b, m[5])
                assert m[6] == b, (a, b, m[6])
                assert m[7] == b, (a, b, m[7])
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('GCD...', end=''); sys.stdout.flush()
        assert isinstance(GCD, UrmCutlandProg), type(GCD)
        for inst in GCD:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for a in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
            for b in range(20):
                m = UrmCutland(init=(a, b))
                if (a == 0) and (b == 0):
                    (i, nb) = m.run(GCD, nb=10000)
                    assert nb == 10000, (i, nb)
                else:
                    (i, nb) = m.run(GCD)
                    assert i == len(GCD) + 1, (a, b, i, len(GCD), nb)
                    assert nb < 1000, (a, b, i, nb)
                    assert nb <= len(GCD)*(max(a, b) + 2), (a, b, i, len(GCD), nb,
                                                            len(GCD)*(max(a, b) + 2))
                    assert len(m) == 5, (a, b, len(m))
                    gcd_a_b = fractions.gcd(a, b)
                    assert m[1] == gcd_a_b, (a, b, m[1], gcd_a_b)
                    assert m[2] == 0, (a, b, m[2])
                    if (a == 0) or (b == 0):
                        assert m[3] == 0, (a, b, m[3])
                    else:
                        assert m[3] != 0, (a, b)
                        assert gcd_a_b <= m[3], (a, b, m[3], gcd_a_b)
                    assert m[3]%gcd_a_b == 0, (a, b, m[3], gcd_a_b,
                                                       m[3]%gcd_a_b)
                    assert m[3] <= max(a, b), (a, b, m[3])
                    if b == 0:
                        assert m[4] == 0, (a, b, m[4])
                    else:
                        assert m[4] == gcd_a_b, (a, b, m[4], gcd_a_b)
                    assert m[5] == 0, (a, b, m[5])
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('FIBONACCI2...', end=''); sys.stdout.flush()
        assert isinstance(FIBONACCI2, UrmCutlandProg), type(FIBONACCI2)
        for inst in FIBONACCI2:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for n in range(20 if debug.assertspeed >= debug.ASSERT_NORMAL else 10):
            m = UrmCutland(init=n)
            (i, nb) = m.run(FIBONACCI2)
            assert i == len(FIBONACCI2) + 1, (n, i, len(FIBONACCI2), nb)
            assert nb <= len(FIBONACCI2)*(nat32.fibonacci(n + 1) + 1), \
                   (n, i, len(FIBONACCI2),
                    nb, nat32.fibonacci(n + 1),len(FIBONACCI2)*(nat32.fibonacci(n + 1) + 1))
            assert len(m) == 5, (n, len(m))
            assert m[1] == nat32.fibonacci(n), (n, m[1], nat32.fibonacci(n))
            assert m[2] == nat32.fibonacci(n + 1), (n, m[2], nat32.fibonacci(n + 1))
            assert m[3] == nat32.fibonacci(n), (n, m[3], nat32.fibonacci(n))
            assert m[4] == n, (n, m[4])
            assert m[5] == n, (n, m[5])
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('FACTORIAL...', end=''); sys.stdout.flush()
        assert isinstance(FACTORIAL, UrmCutlandProg), type(FACTORIAL)
        for inst in FACTORIAL:
            assert (inst.inst() != T) or (inst.a() != inst.b()), inst

        for n in range(9 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            m = UrmCutland(init=n)
            (i, nb) = m.run(FACTORIAL)
            assert i == len(FACTORIAL) + 1, (n, i, len(FACTORIAL), nb)
            assert nb <= len(FACTORIAL)*(math.factorial(n) + 1), \
                   (n, i, len(FACTORIAL), nb, math.factorial(n),
                    len(FACTORIAL)*(math.factorial(n) + 1))
            assert len(m) == 6, (n, len(m))
            assert m[1] == math.factorial(n), (n, m[1], math.factorial(n))
            assert m[2] == n, (n, m[2])
            if n == 0:
                assert m[3] == 0, (n, m[3])
            else:
                assert m[3] == math.factorial(n - 1), (n, m[3], math.factorial(n - 1))
            assert m[4] == n, (n, m[4])
            assert m[5] == n, (n, m[5])
            assert m[6] == n, (n, m[6])
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
