#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.tnp1 Problme 3n + 1
#
# Cf. \htmlonly <a href="http://www.opimedia.be/3nP1/" target="_blank">
#   <tt>http://www.opimedia.be/3nP1/</tt></a>\endhtmlonly

##\file
# Problme 3n + 1
#
# Cf. \htmlonly <a href="http://www.opimedia.be/3nP1/" target="_blank">
#   <tt>http://www.opimedia.be/3nP1/</tt></a>\endhtmlonly

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 14 juillet 2006
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'tnp1 --- 2010 April 12'

import numbers

import DSPython
import DSPython.natural as natural



# Fonction du problme original de (Lothar) Collatz :
#
# Pour tout n entier :
#   L(n) :=  2n     /3   si n mod 3 = 0
#           (4n - 1)/3   si n mod 3 = 1
#           (4n + 1)/3   si n mod 3 = 2
#
#     ==> L(-3) == -2
#         L(-2) == -3
#         L(-1) == -1
#         L( 0) ==  0
#         L( 1) ==  1
#         L( 2) ==  3
#         L( 3) ==  2
#
# Pour tout n entier :
#   L*(n) := L^(n)(n)


# Fonctions lies au problme 3n+1 :
#
# Pour tout n entier :
#   C(n) := n/2      si n pair
#           3n + 1   si n impair
#
#     ==> C(-3) == -8
#         C(-2) == -1
#         C(-1) == -2
#         C( 0) ==  0
#         C( 1) ==  4
#         C( 2) ==  1
#         C( 3) == 10
#
#
#                n
#            3 ----- + 1
#               2^k
#   F(n) := -------------   avec k = nombre de fois pour lequel n est divisible par 2
#                2^l             l = nombre de fois pour lequel le numrateur est divisible par 2
#
#     ==> F(-3) == -1
#         F(-2) == -1
#         F(-1) == -1
#         F( 0) ==  1
#         F( 1) ==  1
#         F( 2) ==  1
#         F( 3) ==  5
#
#
#   T(n) := n/2          si n pair
#           (3n + 1)/2   si n impair
#
#     ==> T(-3) == -4
#         T(-2) == -1
#         T(-1) == -1
#         T( 0) ==  0
#         T( 1) ==  2
#         T( 2) ==  1
#         T( 3) ==  5
#
#   T'(n) := L      o T o L^(-1)
#   T"(n) := L^(-1) o T o L
#
#
#   U(n) := -1         si n == -1
#            0         si n == 0
#           T^(k)(n)   sinon
#                      avec k naturel tel que
#                        n, T(n), T^(2)(n), T^(3)(n), ... et T^(k-1)(n) soient de mme parit
#                        et T^(k)(n) soit de parit contraire
#
#     ==> U(-3) == -4
#         U(-2) == -1
#         U(-1) == -1
#         U( 0) ==  0
#         U( 1) ==  2
#         U( 2) ==  1
#         U( 3) ==  8
#
#
# Pour tout n naturel :
#   C*(n)  := C^(n)(n)
#   T*(n)  := T^(n)(n)
#   T'*(n) := T'^(n)(n)
#   T"*(n) := T"^(n)(n)
#   F*(n)  := F^(n)(n)
#   U*(n)  := U^(n)(n)



# ###########
# Fonctions #
#############
# C()
######
## C(n)
def C(n):
    """Renvoie la valeur de C(n) := n/2      si n pair
                                 3n + 1   si n impair

    Pre: n: Integral

    Result: Integral

    O(n) = 1"""
    assert isinstance(n, numbers.Integral), n

    return (n*3 + 1 if n&1
            else n>>1)


## C <sup>(k)</sup>(n)
def Ck(n, k):
    """Renvoie la valeur de C^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    if n == 0:
        return 0
    while k > 0:
        if n&1 == 0:  # tape(s) pair
            s = natural.scan1(abs(n))  # |n| commence par s bits 0   (s >= 1)
            if s < k:  # s tapes paires
                n >>= s
                k -= s
            else:      # k tapes paires
                return n >> k
        else:         # tape(s) impair
            if n == 1:
                return (1, 4, 2)[k%3]
            elif n > 0:
                s = natural.scan0(n)  # n commence par s bits 1   (s >= 1)
                if (s << 1) < k:  # s doubles tapes impaires, paires
                    n = natural.pow3(s) * ((n >> s) + 1) - 1
                    # == (n' + 1) 3**s - 1 == n' 3**s + 3**s - 1   avec n' = n / (2**s)
                    k -= s << 1
                elif k&1 == 0:    # k doubles tapes impaires, paires
                    return natural.pow3(k>>1) * ((n >> (k>>1)) + 1) - 1
                else:             # k doubles tapes impaires, paires puis 1 tape impaire
                    return (natural.pow3(k>>1) * ((n >> (k>>1)) + 1) - 1) * 3 + 1
            else:
                if k > 1:         # 1 tape impaire puis 1 tape paire
                    n = (n * 3 + 1) >> 1
                    k -= 2
                else:             # 1 tape impaire
                    n = n * 3 + 1
                    k -= 1
    return n


## C*(n)
def CS(n):
    """Renvoie la valeur de C*(n) == C^(n)(n)

    Pre: n: naturel

    Result: Integral

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    return Ck(n, n)


## C* <sup>(k)</sup>(n)
def CSk(n, k):
    """Renvoie la valeur de C*^(k)(n)

    Pre: n: naturel
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(k), k

    while k > 0:
        if 0 <= n <= 2:
            return ((0, 4, 4)[n] if k%2
                    else (0, 2, 2)[n])
        n = CS(n)
        k -= 1
    return n


# F()
######
## F(n)
def F(n):
    """_                                 n
                                  3 ----- + 1
                                     2^k
    Renvoie la valeur de F(n) := -------------
                                      2^l
      avec k = nombre de fois pour lequel n est divisible par 2
           l = nombre de fois pour lequel le numrateur est divisible par 2

    Pre: n: Integral

    Result: Integral

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    if not (0 <= n <= 2):
        if n&1 == 0:  # n pair
            n >>= natural.scan1(abs(n))
        n = n * 3 + 1
        return (n if n&1
                else n >> natural.scan1(abs(n)))
    else:                  # n == 0 ou 1 ou 2
        return 1


## F <sup>(k)</sup>(n)
def Fk(n, k):
    """Renvoie la valeur de F^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    while (k > 0) and (n != 1):
        n = F(n)
        k -= 1
    return n


## F*(n)
def FS(n):
    """Renvoie la valeur de F*(n) == F^(n)(n)

    Pre: n: naturel

    Result: Integral

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    return Fk(n, n)


## F* <sup>(k)</sup>(n)
def FSk(n, k):
    """Renvoie la valeur de F*^(k)(n)

    Pre: n: naturel
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(k), k

    while (k > 0) and (n != 1):
        n = FS(n)
        k -= 1
    return n


# L()
######
## L(n)
def L(n):
    """Renvoie la valeur L(n) :=  2n     /3   si n mod 3 = 0
                              (4n - 1)/3   si n mod 3 = 1
                              (4n + 1)/3   si n mod 3 = 2

    Pre: n: Integral

    Result: Integral

    O(n) = 1"""
    assert isinstance(n, numbers.Integral), n

    n, m = divmod(n, 3)
    return (n << 1 if m == 0
            else (n << 2) + (1 if m == 1
                             else 3))


## L <sup>(-1)</sup>(n)
def L_rec(n):
    """Renvoie la valeur L^(-1)(n) =  3n     /2   si n pair
                                  (3n + 1)/4   si n mod 4 = 1
                                  (3n - 1)/4   si n mod 4 = 3

    Pre: n: Integral

    Result: Integral

    O(n) = 1"""
    assert isinstance(n, numbers.Integral), n

    return ((n >> 1) * 3 if n&1 == 0
            else ((n * 3 - 1) if n&2
                  else (n * 3 + 1)) >> 2)


## L <sup>(k)</sup>(n)
def Lk(n, k):
    """Renvoie la valeur L^(k)(n)

    Pre: n: Integral
         k: Integral

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert isinstance(k, numbers.Integral), k

    if k >= 0:
        while k > 0:
            n = L(n)
            k -= 1
    else:
        while k < 0:
            n = L_rec(n)
            k += 1
    return n


## L*(n)
def LS(n):
    """Renvoie la valeur de L*(n) == L^(n)(n)

    Pre: n: Integral

    Result: Integral

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    return Lk(n, n)


## L* <sup>(k)</sup>(n)
def LSk(n, k):
    """Renvoie la valeur de L*^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    while k > 0:
        n = LS(n)
        k -= 1
    return n


# T()
######
## T(n)
def T(n):
    """Renvoie la valeur de T(n) := n/2          si n pair
                                 (3n + 1)/2   si n impair

    Pre: n: Integral

    Result: Integral

    O(n) = 1"""
    assert isinstance(n, numbers.Integral), n

    return (n*3 + 1 if n&1
            else n) >> 1


## T <sup>(k)</sup>(n)
def Tk(n, k):
    """Renvoie la valeur de T^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    if n == 0:
        return 0
    while k > 0:
        if n&1 == 0:  # tape(s) pair
            s = natural.scan1(abs(n))  # |n| commence par s bits 0   (s >= 1)
            if s < k:  # s tapes paires
                n >>= s
                k -= s
            else:      # k tapes paires
                return n >> k
        else:         # tape(s) impair
            if n == 1:
                return (1, 2)[k%2]
            elif n > 0:
                s = natural.scan0(n)  # n commence par s bits 1   (s >= 1)
                if s < k:  # s tapes impaires
                    n = natural.pow3(s) * ((n >> s) + 1) - 1
                    # == (n' + 1) 3**s - 1 == n' 3**s + 3**s - 1   avec n' = n / (2**s)
                    k -= s
                else:      # k tapes impaires
                    return natural.pow3(k) * ((n >> k) + 1) - 1
            else:          # 1 tape impaire
                n = (n * 3 + 1) >> 1
                k -= 1
    return n


## T*(n)
def TS(n):
    """Renvoie la valeur de T*(n) == T^(n)(n)

    Pre: n: naturel

    Result: Integral

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    return Tk(n, n)


## T* <sup>(k)</sup>(n)
def TSk(n, k):
    """Renvoie la valeur de T*^(k)(n)

    Pre: n: naturel
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(k), k

    if n != 0:
        while (k > 0) and (n != 2):
            n = TS(n)
            k -= 1
    return n


# T'()
#######
## T'(n)
def TP(n):
    """Renvoie la valeur de T'(n) := L o T o L^(-1) (n) = L( T( L^(-1)(n) ) )

    Pre: n: Integral

    Result: Integral

    O(n) = 1"""
    assert isinstance(n, numbers.Integral), n

    return L(T(L_rec(n)))


## T' <sup>(k)</sup>(n)
def TPk(n, k):
    """Renvoie la valeur de T'^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    return L(Tk(L_rec(n), k))


## T'*(n)
def TPS(n):
    """Renvoie la valeur de T'*(n) == T'^(n)(n)

    Pre: n: naturel

    Result: Integral

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    return TPk(n, n)


## T'* <sup>(k)</sup>(n)
def TPSk(n, k):
    """Renvoie la valeur de T'*^(k)(n)

    Pre: n: naturel
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(k), k

    while k > 0:
        n = TPS(n)
        k -= 1
    return n


# T"()
#######
## T&quot;(n)
def TP2(n):
    """Renvoie la valeur de T"(n) := L^(-1) o T o L (n) = L^(-1)( T( L(n) ) )

    Pre: n: Integral

    Result: Integral

    O(n) = 1""" #"
    assert isinstance(n, numbers.Integral), n

    return L_rec(T(L(n)))


## T&quot; <sup>(k)</sup>(n)
def TP2k(n, k):
    """Renvoie la valeur de T"^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ...""" #"
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    return L_rec(Tk(L(n), k))


## T&quot;*(n)
def TP2S(n):
    """Renvoie la valeur de T"*(n) == T"^(n)(n)

    Pre: n: naturel

    Result: Integral

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    return TP2k(n, n)


## T&quot;* <sup>(k)</sup>(n)
def TP2Sk(n, k):
    """Renvoie la valeur de T"*^(k)(n)

    Pre: n: naturel
         k: naturel

    Result: Integral

    O(n, k) = ...""" #"
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(k), k

    while k > 0:
        n = TP2S(n)
        k -= 1
    return n


# U()
######
## U(n)
def U(n):
    """Renvoie la valeur de U(n) := -1         si n == -1
                                 0          si n == 0
                                 T^(k)(n)   sinon
       avec k naturel tel que
         n, T(n), T^(2)(n), T^(3)(n)... et T^(k-1)(n) soient de mme parit
         et T^(k)(n) de parit contraire

    Pre: n: Integral

    Result: Integral

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    if n&1 == 0:  # 1 tape paire
        return (n >> natural.scan1(abs(n)) if n != 0
                else 0)
    else:         # 1 tape impaire
        if n > 0:
            s = natural.scan0(n)  # n commence par s bits 1   (s >= 1)
            return natural.pow3(s) * ((n >> s) + 1) - 1
        else:      # 1 tape impaire
            if n != -1:
                while n&1:
                    n = (n * 3 + 1) >> 1
            return n


## U <sup>(k)</sup>(n)
def Uk(n, k):
    """Renvoie la valeur de U^(k)(n)

    Pre: n: Integral
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert DSPython.natural_is(k), k

    while (k > 0) and not(0 <= n <= 2):
        n = U(n)
        k -= 1
    return (U(n) if k%2
            else n)


## U*(n)
def US(n):
    """Renvoie la valeur de U*(n) == U^(n)(n)

    Pre: n: naturel

    Result: Integral

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    return Uk(n, n)


## U* <sup>(k)</sup>(n)
def USk(n, k):
    """Renvoie la valeur de U*^(k)(n)

    Pre: n: naturel
         k: naturel

    Result: Integral

    O(n, k) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(k), k

    if n != 0:
        while k > 0:
            if 1 <= n <= 4:
                return 2
            n = US(n)
            k -= 1
    return n



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        import DSPython.debug as debug

        debug.test_begin(VERSION, __debug__)

        # C()
        print('C()...', end=''); sys.stdout.flush()
        assert C(-5) == -14, C(-5)
        assert C(-4) ==  -2, C(-4)
        assert C(-3) ==  -8, C(-3)
        assert C(-2) ==  -1, C(-2)
        assert C(-1) ==  -2, C(-1)
        assert C( 0) ==   0, C( 0)
        assert C( 1) ==   4, C( 1)
        assert C( 2) ==   1, C( 2)
        assert C( 3) ==  10, C( 3)
        assert C( 4) ==   2, C( 4)
        assert C( 5) ==  16, C( 5)
        print('ok'); sys.stdout.flush()


        print('Ck()...', end=''); sys.stdout.flush()
        assert Ck(-5, 1) == -14, Ck(-5, 1)
        assert Ck(-4, 1) ==  -2, Ck(-4, 1)
        assert Ck(-3, 1) ==  -8, Ck(-3, 1)
        assert Ck(-2, 1) ==  -1, Ck(-2, 1)
        assert Ck(-1, 1) ==  -2, Ck(-1, 1)
        assert Ck( 0, 1) ==   0, Ck( 0, 1)
        assert Ck( 1, 1) ==   4, Ck( 1, 1)
        assert Ck( 2, 1) ==   1, Ck( 2, 1)
        assert Ck( 3, 1) ==  10, Ck( 3, 1)
        assert Ck( 4, 1) ==   2, Ck( 4, 1)
        assert Ck( 5, 1) ==  16, Ck( 5, 1)
        for n in range(-100, 100):
            assert Ck(n, 0) == n, (n, Ck(n))
            assert Ck(n, 1) == C(n), (n, Ck(n, 1), C(n))
            assert Ck(n, 2) == C(C(n)), (n, Ck(n, 2), C(C(n)))
            for k in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
                assert Ck(n, k+1) == C(Ck(n, k)) == Ck(C(n), k), \
                       (n, k, Ck(n, k+1), C(Ck(n, k)), Ck(C(n), k))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('CS()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert CS(n) == Ck(n, n), (n, CS(n), Ck(n, n))
        print('ok'); sys.stdout.flush()


        print('CSk()...', end=''); sys.stdout.flush()
        for n in range(50):
            assert CSk(n, 0) == n, (n, CSk(n, 0))
            assert CSk(n, 1) == CS(n), (n, CSk(n, 1), CS(n))
            assert CSk(n, 2) == CS(CS(n)), (n, CSk(n, 2), CS(CS(n)))
            for k in range(20):
                assert CSk(n, k+1) == CS(CSk(n, k)), (n, k, CSk(n, k+1), CS(CSk(n, k)))
        print('ok'); sys.stdout.flush()


        # F()
        print('F()...', end=''); sys.stdout.flush()
        assert F(-5) == -7, F(-5)
        assert F(-4) == -1, F(-4)
        assert F(-3) == -1, F(-3)
        assert F(-2) == -1, F(-2)
        assert F(-1) == -1, F(-1)
        assert F( 0) ==  1, F( 0)
        assert F( 1) ==  1, F( 1)
        assert F( 2) ==  1, F( 2)
        assert F( 3) ==  5, F( 3)
        assert F( 4) ==  1, F( 4)
        assert F( 5) ==  1, F( 5)
        print('ok'); sys.stdout.flush()


        print('Fk()...', end=''); sys.stdout.flush()
        assert Fk(-5, 1) == -7, Fk(-5, 1)
        assert Fk(-4, 1) == -1, Fk(-4, 1)
        assert Fk(-3, 1) == -1, Fk(-3, 1)
        assert Fk(-2, 1) == -1, Fk(-2, 1)
        assert Fk(-1, 1) == -1, Fk(-1, 1)
        assert Fk( 0, 1) ==  1, Fk( 0, 1)
        assert Fk( 1, 1) ==  1, Fk( 1, 1)
        assert Fk( 2, 1) ==  1, Fk( 2, 1)
        assert Fk( 3, 1) ==  5, Fk( 3, 1)
        assert Fk( 4, 1) ==  1, Fk( 4, 1)
        assert Fk( 5, 1) ==  1, Fk( 5, 1)
        for n in range(-50, 50):
            assert Fk(n, 0) == n, (n, F(n))
            assert Fk(n, 1) == F(n), (n, Fk(n, 1), F(n))
            assert Fk(n, 2) == F(F(n)), (n, Fk(n, 2), F(F(n)))
            for k in range(25):
                assert Fk(n, k+1) == F(Fk(n, k)) == Fk(F(n), k), \
                       (n, k, Fk(n, k+1), F(Fk(n, k)), Fk(F(n), k))
        print('ok'); sys.stdout.flush()


        print('FS()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert FS(n) == Fk(n, n), (n, FS(n), Fk(n, n))
        print('ok'); sys.stdout.flush()


        print('FSk()...', end=''); sys.stdout.flush()
        for n in range(50):
            assert FSk(n, 0) == n, (n, FSk(n, 0))
            assert FSk(n, 1) == FS(n), (n, FSk(n, 1), FS(n))
            assert FSk(n, 2) == FS(FS(n)), (n, FSk(n, 2), FS(FS(n)))
            for k in range(20):
                assert FSk(n, k+1) == FS(FSk(n, k)), (n, k, FSk(n, k+1), FS(FSk(n, k)))
        print('ok'); sys.stdout.flush()


        # L()
        print('L()...', end=''); sys.stdout.flush()
        assert L(-5) == -7, L(-5)
        assert L(-4) == -5, L(-4)
        assert L(-3) == -2, L(-3)
        assert L(-2) == -3, L(-2)
        assert L(-1) == -1, L(-1)
        assert L( 0) ==  0, L( 0)
        assert L( 1) ==  1, L( 1)
        assert L( 2) ==  3, L( 2)
        assert L( 3) ==  2, L( 3)
        assert L( 4) ==  5, L( 4)
        assert L( 5) ==  7, L( 5)
        print('ok'); sys.stdout.flush()


        print('L_rec()...', end=''); sys.stdout.flush()
        assert L_rec(-5) == -4, L_rec(-5)
        assert L_rec(-4) == -6, L_rec(-4)
        assert L_rec(-3) == -2, L_rec(-3)
        assert L_rec(-2) == -3, L_rec(-2)
        assert L_rec(-1) == -1, L_rec(-1)
        assert L_rec( 0) ==  0, L_rec( 0)
        assert L_rec( 1) ==  1, L_rec( 1)
        assert L_rec( 2) ==  3, L_rec( 2)
        assert L_rec( 3) ==  2, L_rec( 3)
        assert L_rec( 4) ==  6, L_rec( 4)
        assert L_rec( 5) ==  4, L_rec( 5)
        print('ok'); sys.stdout.flush()


        print('Lk()...', end=''); sys.stdout.flush()
        assert Lk(-5, 1) == -7, Lk(-5, 1)
        assert Lk(-4, 1) == -5, Lk(-4, 1)
        assert Lk(-3, 1) == -2, Lk(-3, 1)
        assert Lk(-2, 1) == -3, Lk(-2, 1)
        assert Lk(-1, 1) == -1, Lk(-1, 1)
        assert Lk( 0, 1) ==  0, Lk( 0, 1)
        assert Lk( 1, 1) ==  1, Lk( 1, 1)
        assert Lk( 2, 1) ==  3, Lk( 2, 1)
        assert Lk( 3, 1) ==  2, Lk( 3, 1)
        assert Lk( 4, 1) ==  5, Lk( 4, 1)
        assert Lk( 5, 1) ==  7, Lk( 5, 1)
        for n in range(-50, 50):
            assert Lk(n, 0) == n, (n, L(n))
            assert Lk(n, 1) == L(n), (n, Lk(n, 1), L(n))
            assert Lk(n, 2) == L(L(n)), (n, Lk(n, 2), L(L(n)))
            assert Lk(n, -1) == L_rec(n), (n, Lk(n, -1), L_rec(n))
            assert Lk(n, -2) == L_rec(L_rec(n)), (n, Lk(n, -2), L_rec(L_rec(n)))
            for k in range(25):
                assert Lk(n, k+1) == L(Lk(n, k)) == Lk(L(n), k), \
                       (n, k, Lk(n, k+1), L(Lk(n, k)), Lk(L(n), k))
                assert Lk(n, -(k+1)) == L_rec(Lk(n, -k)) == Lk(L_rec(n), -k), \
                       (n, -k, Lk(n, -(k+1)), L_rec(Lk(n, -k)), Lk(L_rec(n), -k))
        print('ok'); sys.stdout.flush()


        print('LS()...', end=''); sys.stdout.flush()
        for n in range(-100, 100):
            assert LS(n) == Lk(n, n), (n, LS(n), Lk(n, n))
        print('ok'); sys.stdout.flush()


        print('LSk()...', end=''); sys.stdout.flush()
        for n in range(30 if debug.assertspeed >= debug.ASSERT_NORMAL else 10):
            assert LSk(n, 0) == n, (n, LSk(n, 0))
            assert LSk(n, 1) == LS(n), (n, LSk(n, 1), LS(n))
            assert LSk(n, 2) == LS(LS(n)), (n, LSk(n, 2), LS(LS(n)))
            for k in range(10):
                if ((n == 8)
                    or (n >= 10)
                    and ((k == 3) or ((n >= 17) and (k == 2)))):
                    break
                assert LSk(n, k+1) == LS(LSk(n, k)), (n, k, LSk(n, k+1), LS(LSk(n, k)))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        # T()
        print('T()...', end=''); sys.stdout.flush()
        assert T(-5) == -7, T(-5)
        assert T(-4) == -2, T(-4)
        assert T(-3) == -4, T(-3)
        assert T(-2) == -1, T(-2)
        assert T(-1) == -1, T(-1)
        assert T( 0) ==  0, T( 0)
        assert T( 1) ==  2, T( 1)
        assert T( 2) ==  1, T( 2)
        assert T( 3) ==  5, T( 3)
        assert T( 4) ==  2, T( 4)
        assert T( 5) ==  8, T( 5)
        for n in range(-100, 100):
            if n&1 == 0:  # n pair
                assert T(n) == C(n), (n, T(n), C(n))
            else:         # n impair
                assert T(n) == Ck(n, 2), (n, T(n), Ck(n, 2))
        print('ok'); sys.stdout.flush()


        print('Tk()...', end=''); sys.stdout.flush()
        assert Tk(-5, 1) == -7, Tk(-5, 1)
        assert Tk(-4, 1) == -2, Tk(-4, 1)
        assert Tk(-3, 1) == -4, Tk(-3, 1)
        assert Tk(-2, 1) == -1, Tk(-2, 1)
        assert Tk(-1, 1) == -1, Tk(-1, 1)
        assert Tk( 0, 1) ==  0, Tk( 0, 1)
        assert Tk( 1, 1) ==  2, Tk( 1, 1)
        assert Tk( 2, 1) ==  1, Tk( 2, 1)
        assert Tk( 3, 1) ==  5, Tk( 3, 1)
        assert Tk( 4, 1) ==  2, Tk( 4, 1)
        assert Tk( 5, 1) ==  8, Tk( 5, 1)
        for n in range(-100, 100):
            assert Tk(n, 0) == n, (n, Tk(n, 0))
            assert Tk(n, 1) == T(n), (n, Tk(n, 1), T(n))
            assert Tk(n, 2) == T(T(n)), (n, Tk(n, 2), T(T(n)))
            for k in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
                assert Tk(n, k+1) == T(Tk(n, k)) == Tk(T(n), k), \
                       (n, k, Tk(n, k+1), T(Tk(n, k)), Tk(T(n), k))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('TS()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert TS(n) == Tk(n, n), (n, TS(n), Tk(n, n))
        print('ok'); sys.stdout.flush()


        print('TSk()...', end=''); sys.stdout.flush()
        for n in range(50):
            assert TSk(n, 0) == n, (n, TSk(n, 0))
            assert TSk(n, 1) == TS(n), (n, TSk(n, 1), TS(n))
            assert TSk(n, 2) == TS(TS(n)), (n, TSk(n, 2), TS(TS(n)))
            for k in range(20):
                assert TSk(n, k+1) == TS(TSk(n, k)), (n, k, TSk(n, k+1), TS(TSk(n, k)))
        print('ok'); sys.stdout.flush()


        # T'()
        print('TP()...', end=''); sys.stdout.flush()
        assert TP(-5) == -3, TP(-5)
        assert TP(-4) == -2, TP(-4)
        assert TP(-3) == -1, TP(-3)
        assert TP(-2) == -5, TP(-2)
        assert TP(-1) == -1, TP(-1)
        assert TP( 0) ==  0, TP( 0)
        assert TP( 1) ==  3, TP( 1)
        assert TP( 2) ==  7, TP( 2)
        assert TP( 3) ==  1, TP( 3)
        assert TP( 4) ==  2, TP( 4)
        assert TP( 5) ==  3, TP( 5)
        for n in range(-100, 100):
            assert TP(n) == L(T(L_rec(n))), (n, TP(n), L(T(L_rec(n))))
        print('ok'); sys.stdout.flush()


        print('TPk()...', end=''); sys.stdout.flush()
        assert TPk(-5, 1) == -3, TPk(-5, 1)
        assert TPk(-4, 1) == -2, TPk(-4, 1)
        assert TPk(-3, 1) == -1, TPk(-3, 1)
        assert TPk(-2, 1) == -5, TPk(-2, 1)
        assert TPk(-1, 1) == -1, TPk(-1, 1)
        assert TPk( 0, 1) ==  0, TPk( 0, 1)
        assert TPk( 1, 1) ==  3, TPk( 1, 1)
        assert TPk( 2, 1) ==  7, TPk( 2, 1)
        assert TPk( 3, 1) ==  1, TPk( 3, 1)
        assert TPk( 4, 1) ==  2, TPk( 4, 1)
        assert TPk( 5, 1) ==  3, TPk( 5, 1)
        for n in range(-100, 100):
            assert TPk(n, 0) == n, (n, TPk(n, 0))
            assert TPk(n, 1) == TP(n), (n, TPk(n, 1), TP(n))
            assert TPk(n, 2) == TP(TP(n)), (n, TPk(n, 2), TP(TP(n)))
            for k in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 25):
                assert TPk(n, k+1) == TP(TPk(n, k)) == TPk(TP(n), k), \
                       (n, k, TPk(n, k+1), TP(TPk(n, k)), TPk(TP(n), k))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('TPS()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert TPS(n) == TPk(n, n), (n, TPS(n), TPk(n, n))
        print('ok'); sys.stdout.flush()


        print('TPSk()...', end=''); sys.stdout.flush()
        for n in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 25):
            assert TPSk(n, 0) == n, (n, TPSk(n, 0))
            assert TPSk(n, 1) == TPS(n), (n, TPSk(n, 1), TPS(n))
            assert TPSk(n, 2) == TPS(TPS(n)), (n, TPSk(n, 2), TPS(TPS(n)))
            for k in range(20):
                assert TPSk(n, k+1) == TPS(TPSk(n, k)), (n, k, TPSk(n, k+1), TPS(TPSk(n, k)))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        # T"()
        print('TP2()...', end=''); sys.stdout.flush()
        assert TP2(-5) == -15, TP2(-5)
        assert TP2(-4) ==  -5, TP2(-4)
        assert TP2(-3) ==  -1, TP2(-3)
        assert TP2(-2) ==  -6, TP2(-2)
        assert TP2(-1) ==  -1, TP2(-1)
        assert TP2( 0) ==   0, TP2( 0)
        assert TP2( 1) ==   3, TP2( 1)
        assert TP2( 2) ==   4, TP2( 2)
        assert TP2( 3) ==   1, TP2( 3)
        assert TP2( 4) ==  12, TP2( 4)
        assert TP2( 5) ==   8, TP2( 5)
        for n in range(-100, 100):
            assert TP2(n) == L_rec(T(L(n))), (n, TP2(n), L_rec(T(L(n))))
        print('ok'); sys.stdout.flush()


        print('TP2k()...', end=''); sys.stdout.flush()
        assert TP2k(-5, 1) == -15, TP2k(-5, 1)
        assert TP2k(-4, 1) ==  -5, TP2k(-4, 1)
        assert TP2k(-3, 1) ==  -1, TP2k(-3, 1)
        assert TP2k(-2, 1) ==  -6, TP2k(-2, 1)
        assert TP2k(-1, 1) ==  -1, TP2k(-1, 1)
        assert TP2k( 0, 1) ==   0, TP2k( 0, 1)
        assert TP2k( 1, 1) ==   3, TP2k( 1, 1)
        assert TP2k( 2, 1) ==   4, TP2k( 2, 1)
        assert TP2k( 3, 1) ==   1, TP2k( 3, 1)
        assert TP2k( 4, 1) ==  12, TP2k( 4, 1)
        assert TP2k( 5, 1) ==   8, TP2k( 5, 1)
        for n in range(-100, 100):
            assert TP2k(n, 0) == n, (n, TP2k(n, 0))
            assert TP2k(n, 1) == TP2(n), (n, TP2k(n, 1), TP2(n))
            assert TP2k(n, 2) == TP2(TP2(n)), (n, TP2k(n, 2), TP2(TP2(n)))
            for k in range(50):
                assert TP2k(n, k+1) == TP2(TP2k(n, k)) == TP2k(TP2(n), k), \
                       (n, k, TP2k(n, k+1), TP2(TP2k(n, k)), TP2k(TP2(n), k))
        print('ok'); sys.stdout.flush()


        print('TP2S()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert TP2S(n) == TP2k(n, n), (n, TP2S(n), TP2k(n, n))
        print('ok'); sys.stdout.flush()


        print('TP2Sk()...', end=''); sys.stdout.flush()
        for n in range(50 if debug.assertspeed >= debug.ASSERT_NORMAL else 25):
            assert TP2Sk(n, 0) == n, (n, TP2Sk(n, 0))
            assert TP2Sk(n, 1) == TP2S(n), (n, TP2Sk(n, 1), TP2S(n))
            assert TP2Sk(n, 2) == TP2S(TP2S(n)), (n, TP2Sk(n, 2), TP2S(TP2S(n)))
            for k in range(20):
                assert TP2Sk(n, k+1) == TP2S(TP2Sk(n, k)), (n, k, TP2Sk(n, k+1), TP2S(TP2Sk(n, k)))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()

        # U()
        print('U()...', end=''); sys.stdout.flush()
        assert U(-5) == -10, U(-5)
        assert U(-4) ==  -1, U(-4)
        assert U(-3) ==  -4, U(-3)
        assert U(-2) ==  -1, U(-2)
        assert U(-1) ==  -1, U(-1)
        assert U( 0) ==   0, U( 0)
        assert U( 1) ==   2, U( 1)
        assert U( 2) ==   1, U( 2)
        assert U( 3) ==   8, U( 3)
        assert U( 4) ==   1, U( 4)
        assert U( 5) ==   8, U( 5)
        for n in range(-100, -1):
            t = T(n)
            while t & 1 == n & 1:  # t et n de mme parit
                t = T(t)
            assert U(n) == t, (n, U(n), t)
        for n in range(1, 100):
            t = T(n)
            while t & 1 == n & 1:  # t et n de mme parit
                t = T(t)
            assert U(n) == t, (n, U(n), t)
        print('ok'); sys.stdout.flush()


        print('Uk()...', end=''); sys.stdout.flush()
        assert Uk(-5, 1) == -10, Uk(-5, 1)
        assert Uk(-4, 1) ==  -1, Uk(-4, 1)
        assert Uk(-3, 1) ==  -4, Uk(-3, 1)
        assert Uk(-2, 1) ==  -1, Uk(-2, 1)
        assert Uk(-1, 1) ==  -1, Uk(-1, 1)
        assert Uk( 0, 1) ==   0, Uk( 0, 1)
        assert Uk( 1, 1) ==   2, Uk( 1, 1)
        assert Uk( 2, 1) ==   1, Uk( 2, 1)
        assert Uk( 3, 1) ==   8, Uk( 3, 1)
        assert Uk( 4, 1) ==   1, Uk( 4, 1)
        assert Uk( 5, 1) ==   8, Uk( 5, 1)
        for n in range(-50, 50):
            assert Uk(n, 0) == n, (n, U(n))
            assert Uk(n, 1) == U(n), (n, Uk(n, 1), U(n))
            assert Uk(n, 2) == U(U(n)), (n, Uk(n, 2), U(U(n)))
            for k in range(25):
                assert Uk(n, k+1) == U(Uk(n, k)) == Uk(U(n), k), \
                       (n, k, Uk(n, k+1), U(Uk(n, k)), Uk(U(n), k))
        print('ok'); sys.stdout.flush()


        print('US()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert US(n) == Uk(n, n), (n, US(n), Uk(n, n))
        print('ok'); sys.stdout.flush()


        print('USk()...', end=''); sys.stdout.flush()
        for n in range(50):
            assert USk(n, 0) == n, (n, USk(n, 0))
            assert USk(n, 1) == US(n), (n, USk(n, 1), US(n))
            assert USk(n, 2) == US(US(n)), (n, USk(n, 2), US(US(n)))
            for k in range(20):
                assert USk(n, k+1) == US(USk(n, k)), (n, k, USk(n, k+1), US(USk(n, k)))
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
