#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.polynomial Polynmes

##\file
# Polynmes

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 11 mars 2007
####################################
from __future__ import division
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'polynomial --- 2010 March 16'

import numbers, string

import DSPython
import DSPython.numbernone as numbernone



# ############
# Constantes #
##############
## String contenant les caractres alphabtiques majuscules puis minuscules
ALPHA = string.ascii_uppercase + string.ascii_lowercase

## String contenant les caractres alphabtiques minuscules puis majuscules
ALPHA_DOWN = string.ascii_lowercase + string.ascii_uppercase



# #########
# Classes #
###########
## Terme (tuple de Number ou de None reprsentant les exposants pour chaque variable)
class Term(tuple):
    """Terme(tuple de Number ou de None
    reprsentant les exposants pour chaque variable)"""

    ## Renvoie le terme sous forme de string "'Term(tuple des exposants)'"
    def __repr__(self):
        """Renvoie le terme sous forme de string "'Term(tuple des exposants)'"

        Result: string

        O() = ..."""
        return "'Term({0})'".format(tuple.__repr__(self))


    ## Renvoie le terme sous forme de string
    def __str__(self, varsstr=ALPHA, nonestr='?',
                mulstr='*', expstr='^', exprstr='', simplify=True):
        """Renvoie le terme sous forme de string.
        varsstr spcifie le caractre des variables utilises
        nonestr spcifie la chane de caractres
          reprsentant les valeurs indtermines None
        Les facteurs sont spars par multstr.
        expstr est plac  gauche de chaque exposant et exprstr  droite.
        Si simplify alors chaque lment sera prsent dans le rsultat,
                    sinon passe les variables d'exposant 0
                          et passe les exposants 1.
        Si le rsultat ne contient aucun lment
          alors finalement renvoie '1'.

        Pre: varsstr: string assez grand
                        pour que chaque variable ait un caractre
             nonestr: string
             multstr: string
             expstr: string
             exprstr: string
             simplify: boolean

        Result: string

        O() = ..."""
        assert isinstance(varsstr, str), type(varsstr)
        assert len(varsstr) >= len(self), (len(varsstr), len(self))
        assert isinstance(nonestr, str), type(nonestr)
        assert isinstance(mulstr, str), type(mulstr)
        assert isinstance(expstr, str), type(expstr)
        assert isinstance(exprstr, str), type(exprstr)

        l = []  # liste des facteurs sous forme de string
        for i in range(len(self)):
            e = self[i]

            assert isinstance(e, numbers.Number) or (e == None), (i, e)

            if simplify and (e == 1):
                l.append(varsstr[i])
            elif (not simplify) or (e != 0):
                l.append('{0}{1}{2}{3}'.format(varsstr[i], expstr, (e if e != None
                                                                    else nonestr), exprstr))
        return (mulstr.join(l) if l != []
                else '1')


    ## ??? Compare les exposants (d'aprs l'ordre dtermin par order) des 2 Term
    def cmp_order(self, other, order=None):
        """Compare les exposants (d'aprs l'ordre dtermin par order)
        des 2 Term et renvoie -1 si self <  other
                               0         ==
                               1         >
        Si order == None alors utilise l'ordre normal

        Pre: other: Term
             order: None
                    ou tuple de naturels assez grand
                       pour que chaque variable ait un indice

        Result: -1, 0 ou 1

        O() = ..."""
        assert isinstance(other, Term), type(other)
        assert len(self) == len(other), (len(self), len(other))
        assert order == None or isinstance(order, tuple), type(order)
        assert order == None or len(order) >= len(self), (len(order), len(self))

        if order == None:
            for i in range(len(self)):
                c = cmp(self[i], other[i])
                if c != 0:
                    return c
                i += 1
            return 0
        else:
            for i in range(len(self)):
                c = order[i]
                c = cmp(self[c], other[c])
                if c != 0:
                    return c
                i += 1
            return 0


    ## Renvoie la k<sup>e</sup> drive partielle pour la variable d'indice i
    def deriv(self, i=0, k=1):
        """Renvoie la kme drive partielle pour la variable d'indice i
        Si k > 0 alors pour le terme (..., n, ...)
          renvoie (n(n - 1)...(n - k + 1), Term((..., n - k, ...)))
          En particulier si k == 1
            alors renvoie (n, Term((..., n - 1, ...)))
        Si k == 0 alors renvoie (1, Term(self))
        Si k < 0 alors pour le terme (..., n, ...)
          renvoie (1/((n + 1)...(n - k)), Term((..., n - k, ...)))
          (Si k <= n < 0 alors renvoie (None, Term((..., n - k, ...))))
        Si n == None alors renvoie (None, Term((..., n, ...)))
        Donc pour (..., n, ...) renvoie toujours
          (numbernone.falling_factorial_pow(n, k), Term((..., n - k, ...)))

        Pre: i: natural < len(self)
             k: Integral

        Result: (Number ou None, Term)

        O() = ..."""
        assert DSPython.natural_is(i), i
        assert i < len(self), (i, len(self))
        assert isinstance(k, numbers.Integral), k

        return ((numbernone.falling_factorial_pow(self[i], k),
                 Term(self[:i] + (self[i] - k, ) + self[i + 1:])) if self[i] != None
                else (None, Term(self)))


    ## Renvoie le terme valu en fonction des valeurs de values
    def eval(self, values=1):
        """Renvoie le terme valu en fonction des valeurs de values
        Pour le terme (a, b, c, ...)
        si values est un Number
          alors renvoie values^a * values^b * values^c * ...
        si values est une squence
          alors renvoie values[0]^a * values[1]^b * values[2]^c * ...
        Si un des ces Number est None alors renvoie None

        Pre: value: Number ou None
                    ou tuple (de Number ou de None) assez grand
                       pour que chaque variable ait une valeur
                    ou list (de Number ou de None) assez grand
                       pour que chaque variable ait une valeur

        Result: Number ou None

        O() = ..."""
        assert isinstance(values, numbers.Number) or (values == None) \
               or isinstance(values, tuple) or isinstance(values, list), type(values)
        assert isinstance(values, numbers.Number) or (values == None) \
               or (len(values) >= len(self)), (len(values), len(self))

        if isinstance(values, numbers.Number):
            if values == 1:
                return 1
            else:
                nb = 0
                for e in self:
                    if e == None:
                        return None
                    if e != 0:
                        nb += e
                return values**nb
        elif values != None:
            assert len(values) >= len(self), (len(values), len(self))

            r = 1
            for i in range(len(self)):
                e = self[i]
                if (e == None) or (values[i] == None):
                    return None
                if (e != 0) and (values[i] != 1):
                    r *= values[i]**e
            return r
        else:
            return None


    ##\brief Renvoie une "valuation partielle" du terme sous forme de (coef, Term)
    # en fonction des valeurs de values
    def partialeval(self, values):
        """Renvoie une "valuation partielle" du terme
        sous forme de (coef, Term) en fonction des valeurs de values.
        Pour chaque Number de values,
          la variable correspondante est value,
          le rsultat multiplie coef et l'exposant est rinitialis  0,
        alors que pour chaque None de values,
          la variable correspondante est ignore
          et son exposant conserv.
        Si le terme est vide alors renvoie (1, Term())
        Si le terme contient un exposant None qui doit tre valu
          alors renvoie (None, Term) o les exposants None sont conservs

        Pre: value: tuple de Number ou de None assez grand
                       pour que chaque variable ait une valeur
                    ou list de Number ou de None assez grand
                       pour que chaque variable ait une valeur

        Result: (Number ou None, Term)

        O() = ..."""
        assert isinstance(values, tuple) or isinstance(values, list), type(values)
        assert len(values) >= len(self), (len(values), len(self))

        notnone = True
        coef = 1
        l = list(self)
        for i in range(len(self)):
            n = values[i]
            if n != None:
                if self[i] != None:
                    coef *= n**self[i]
                    l[i] = 0
                else:
                    coef = 0
                    notnone = False
        return (coef if notnone
                else None, Term(l))


##\brief Polynme (dictionnaire[Term] = coefficient,
# avec varsstr, une chane de caractres pour les variables)
class Polynomial(dict):
    """Polynme (dictionnaire[Term] = coefficient,
    avec varsstr, une chane de caractres pour les variables)"""

    ## Initialise le polynme  value * coef
    def __init__(self, value=(), coef=1, varsstr=ALPHA):
        """Initialise le polynme  value * coef et
        et spcifie le caractre des variables par varsstr.
        Si value est un Polynomial
        alors varsstr est ignor et value.varsstr est utilis

        Pre: value: Polynomial
                    ou dict de Term de mme taille pour les cls
                            et de Number pour les valeurs
                    ou Term
                    ou tuple de Term de mme taille
                    ou liste de Term de mme taille
             coef: Number
             varsstr: string assez grand
                        pour que chaque variable ait un caractre

        Result: None

        O() = ..."""
        assert isinstance(value, Polynomial) or isinstance(value, Term) or isinstance(value, dict) \
               or isinstance(value, tuple) or isinstance(value, list), type(value)
        assert isinstance(coef, numbers.Number), coef
        assert isinstance(varsstr, str), type(varsstr)

        ## String contenant les caractres pour chaque variable
        self.varsstr = (value.varsstr if isinstance(value, Polynomial)
                        else varsstr)
        if coef != 0:
            if isinstance(value, Polynomial) or isinstance(value, dict):
                if (coef == 1) and isinstance(value, Polynomial):
                    dict.__init__(self, value)
                else:
                    dict.__init__(self)
                    for t in value:
                        assert isinstance(t, Term), t
                        assert len(t) == len(list(value.keys())[0]), (len(t), len(value.keys()[0]))

                        if value[t]*coef != 0:
                            self[t] = value[t] * coef
            elif isinstance(value, Term):
                assert len(varsstr) >= len(value), (len(varsstr), len(value))

                dict.__init__(self, {value: coef})
            else:  # value est un tuple de Term ou une liste de Term
                dict.__init__(self)
                for t in value:
                    assert isinstance(t, Term), type(t)
                    assert len(t) == len(value[0]), (len(t), len(value[0]))

                    if t not in self:
                        self[t] = coef
                    else:
                        self[t] += coef
                        if self[t] == 0:
                            del self[t]
        else:
            dict.__init__(self)


    ## ??? Renvoie le polynme auquel est ajout value * coef
    def __iadd__(self, value, coef=1):
        """Renvoie le polynme auquel est ajout value * coef

        Pre: value: Polynomial de mme taille des termes
                    ou Term (de taille quelconque si le polynme est vide,
                             de taille lenterms() sinon)
                    ou Number (si le polynme n'est pas vide)
             coef: Number

        Result: Polynomial

        O() = ..."""
        if coef != 0:
            if isinstance(value, Term):
                if (not self) or (self.lenterms() == len(value)):
                    if value not in self:
                        self[value] = coef
                    else:
                        self[value] += coef
                        if self[value] == 0:
                            del self[value]
                    return self
            elif isinstance(value, numbers.Number):
                if self:
                    t = Term((0, ) * self.lenterms())
                    coef *= value
                    if coef != 0:
                        if t in self:
                            self[t] += coef
                            if self[t] == 0:
                                del self[t]
                        else:
                            self[t] = coef
                    return self
            elif isinstance(value, Polynomial) and (value.lenterms() == self.lenterms()):
                for t in value:
                    if t not in self:
                        self[t] = value[t] * coef
                    else:
                        self[t] += value[t] * coef
                    if self[t] == 0:
                        del self[t]
                return self
            raise NotImplementedError
        else:
            return self


    ## Renvoie le polynme sous forme de string "'Polynomial(tuple des exposants, varsstr="...")'"
    def __repr__(self):
        """Renvoie le polynme sous forme
        de string "'Polynomial(tuple des exposants, varsstr="...")'"

        Result: string

        O() = ..."""
        return "'Polynomial({0}, varsstr=\"{1}\")'".format(dict.__repr__(self), self.varsstr)


    ## ??? Renvoie le polynme sous forme de string
    def __str__(self, varsstr=None, nonestr='?', addstr=' + ', substr=' - ', mulstr='*',
                expstr='^', exprstr='', simplify=True, order=None):
        """Renvoie le polynme sous forme de string.
        varsstr spcifie le caractre des variables utilises
          (si varsstr == None alors utilise self.varsstr).
        nonestr spcifie la chane de caractres
          reprsentant les valeurs indtermines None
        Les termes de coefficients >= 0 sont spars par addstr,
                                   <  0                  substr.
        Les facteurs sont spars par mulstr.
        expstr est plac  gauche de chaque exposant et exprstr  droite.
        Si simplify alors chaque lment sera prsent dans le rsultat,
                    sinon passe le coefficient si == 1,
                          passe les variables d'exposant 0
                          et passe les exposants 1.
        order ???
        Si le rsultat ne contient aucun lment alors renvoie ''.

        Pre: varsstr: None
                      ou string assez grand
                        pour que chaque variable ait un caractre
             nonestr: string
             addstr: string
             substr: string
             multstr: string
             expstr: string
             exprstr: string
             simplify: boolean
             order: None
                    ou tuple de naturels ou liste de tuple de naturels
                       assez grand pour que chaque variable ait un indice

        Result: string

        O() = ..."""
        if varsstr == None:
            varsstr = self.varsstr

        assert isinstance(varsstr, str), type(varsstr)
        assert (self.lenterms() == None) or (len(varsstr) >= self.lenterms()), \
               (len(varsstr), self.lenterms())#???
        assert isinstance(nonestr, str), type(nonestr)
        assert isinstance(addstr, str), type(addstr)
        assert isinstance(substr, str), type(substr)
        assert isinstance(mulstr, str), type(mulstr)
        assert isinstance(expstr, str), type(expstr)
        assert isinstance(exprstr, str), type(exprstr)
        assert order == None or isinstance(order, tuple) or isinstance(order, list), type(order)
        assert order == None or len(order) >= len(self), (len(order), len(self))

        if order != None:
            raise NotImplementedError

        l = []  # liste des termes sous forme de string
        terms = sorted(self)
        if simplify:
            for t in terms:
                assert isinstance(t, Term), type(t)

                if self[t] >= 0:
                    l.append(addstr)
                    if self[t] != 1:
                        l.append(str(self[t]))
                    t_str = t.__str__(varsstr=varsstr, nonestr=nonestr,
                                      mulstr=mulstr, expstr=expstr, exprstr=exprstr)
                    if (self[t] != 1) and (t_str != '1'):
                        l.append(mulstr)
                    if (t_str != '1') or (self[t] == 1):
                        l.append(t_str)
                else:
                    if l == []:
                        l.append(substr)
                    l.append(substr)
                    if self[t] != -1:
                        l.append(str(-self[t]))
                    t_str = t.__str__(varsstr=varsstr, nonestr=nonestr,
                                      mulstr=mulstr, expstr=expstr, exprstr=exprstr)
                    if (self[t] != -1) and (t_str != '1'):
                        l.append(mulstr)
                    if (t_str != '1') or (self[t] == -1):
                        l.append(t_str)
        else:
            for t in terms:
                assert isinstance(t, Term), type(t)

                if self[t] >= 0:
                    l.append(addstr)
                    l.append(str(self[t]))
                    t_str = t.__str__(varsstr=varsstr, nonestr=nonestr,
                                      mulstr=mulstr, expstr=expstr, exprstr=exprstr, simplify=False)
                else:
                    if l == []:
                        l.append(substr)
                    l.append(substr)
                    l.append(str(-self[t]))
                    t_str = t.__str__(varsstr=varsstr, nonestr=nonestr,
                                      mulstr=mulstr, expstr=expstr, exprstr=exprstr, simplify=False)
                l.append(mulstr)
                l.append(t_str)
        return ''.join(l[1:])  # ignore le premier addstr ajout dans l


    ## Renvoie la k<sup>e</sup> drive partielle pour la variable d'indice i
    def deriv(self, i=0, k=1):
        """Renvoie la kme drive partielle pour la variable d'indice i
        (c.--d. la somme des drives de ses termes)

        Pre: i: natural < self.lenterms()
             k: Integral

        Result: (Number ou None, Term)

        O() = ..."""
        assert DSPython.natural_is(i), i
        assert i < self.lenterms(), (i, self.lenterms())
        assert isinstance(k, numbers.Integral), k

        d = {}
        for t in self:
            (c, dt) = t.deriv(i=i, k=k)
            d[dt] = (self[t] * c if c != None
                     else None)
        return Polynomial(d, varsstr=self.varsstr)


    ## Renvoie le polynme valu en fonction des valeurs de values
    def eval(self, values=1):
        """Renvoie le polynme valu en fonction des valeurs de values

        Pre: value: Number
                    ou tuple de Number assez grand
                       pour que chaque variable ait une valeur
                    ou list de Number assez grand
                       pour que chaque variable ait une valeur

        Result: Number ou None ???

        O() = ..."""
        assert isinstance(values, numbers.Number) or isinstance(values, tuple) \
               or isinstance(values, list), type(values)
        assert isinstance(values, numbers.Number) or (len(values) >= self.lenterms()), \
               (len(values), self.lenlenterms())

        r = 0
        for t in self:
            p = t.eval(values) * self[t]
            if p != 0:
                r += p
        return r


    ## Renvoie la taille des termes
    def lenterms(self):
        """Renvoie la taille des termes ou None si le polynme est vide

        Result: int ou None

        O() = ..."""
        return (len(list(self.keys())[0]) if len(self) > 0
                else None)


    ## Renvoie une "valuation partielle" du polynme en fonction des valeurs de values.
    def partialeval(self, values):
        """Renvoie une "valuation partielle" du polynme
          en fonction des valeurs de values.
        Pour chaque Number de values,
          la variable correspondante est value,
          le rsultat multiplie le coefficient du terme
          et l'exposant est rinitialis  0,
        alors que pour chaque None de values,
          la variable correspondante est ignore et son exposant conserv.
        Si le polynme est vide alors renvoie Polynomial()

        Pre: value: tuple de Number ou de None assez grand
                       pour que chaque variable ait une valeur
                    ou list de Number ou de None assez grand
                       pour que chaque variable ait une valeur

        Result: Polynomial

        O() = ..."""
        assert isinstance(values, tuple) or isinstance(values, list), type(values)
        assert (self.lenterms() == None) or (len(values) >= self.lenterms()), \
               (len(values), self.lenterms())#???

        p = Polynomial()
        for t in self:
            (c, new_t) = t.partialeval(values)
            c *= self[t]
            if c != 0:
                if new_t not in p:
                    p[new_t] = c
                else:
                    p[new_t] += c
                    if p[new_t] == 0:
                        del p[new_t]
        return p



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import decimal, sys

        import DSPython.debug as debug

        debug.test_begin(VERSION, __debug__)

        print("ALPHA == '{0}'".format(ALPHA)); sys.stdout.flush()
        print("ALPHA_DOWN == '{0}'".format(ALPHA_DOWN)); sys.stdout.flush()



        print()
        print('Term()...', end=''); sys.stdout.flush()
        assert Term() == (), Term()
        assert Term(()) == (), Term(())
        assert Term((1, 2)) == (1, 2), Term((1, 2))
        assert Term([1, 2]) == (1, 2), Term([1, 2])
        assert Term((1, None, 2)) == (1, None, 2), Term((1, None, 2))
        assert Term((3, -5, 7)) == (3, -5, 7), Term((3, -5, 7))
        assert Term((3, 5.0, decimal.Decimal('7'))) == (3, 5.0, decimal.Decimal('7')), \
               Term((3, 5.0, decimal.Decimal('7')))
        assert Term(Term((3, 5, 7))) == (3, 5, 7), Term(Term((3, 5, 7)))
        assert isinstance(Term((3, 5, 7)), Term), Term((3, 5, 7))
        assert isinstance(Term((3, 5, 7)), tuple), Term((3, 5, 7))
        assert not isinstance(Term((3, 5, 7)), list), Term((3, 5, 7))
        print('ok'); sys.stdout.flush()


        print('Term.__repr__()...', end=''); sys.stdout.flush()
        assert repr(Term()) == "'Term(())'", repr(Term())
        assert repr(Term((0, 0))) == "'Term((0, 0))'", repr(Term((0, 0)))
        assert repr(Term((1, 2))) == "'Term((1, 2))'", repr(Term((1, 2)))
        assert repr(Term([1, 2])) == "'Term((1, 2))'", repr(Term([1, 2]))
        assert repr(Term((1, None, 2))) == "'Term((1, None, 2))'", repr(Term((1, None, 2)))
        assert repr(Term((3, -5, 7))) == "'Term((3, -5, 7))'", repr(Term((3, -5, 7)))
        assert repr(Term((3, 5.0, decimal.Decimal('7')))) == "'Term((3, 5.0, Decimal('7')))'",\
               repr(Term((3, 5.0, decimal.Decimal('7'))))
        assert repr(Term((3, 1, 7))) == "'Term((3, 1, 7))'", repr(Term((3, 1, 7)))
        assert repr(Term((3, 0, 7))) == "'Term((3, 0, 7))'", repr(Term((3, 0, 7)))
        print('ok'); sys.stdout.flush()


        print('Term.__str__()...', end=''); sys.stdout.flush()
        assert str(Term()) == '1', str(Term())
        assert str(Term((0, 0))) == '1', str(Term((0, 0)))
        assert str(Term((1, 2))) == 'A*B^2', str(Term((1, 2)))
        assert str(Term([1, 2])) == 'A*B^2', str(Term([1, 2]))
        assert str(Term((3, -5, 7))) == 'A^3*B^-5*C^7', str(Term((3, -5, 7)))
        assert str(Term((3, 5.0, decimal.Decimal('7')))) == 'A^3*B^5.0*C^7', \
               str(Term((3, 5.0, decimal.Decimal('7'))))
        assert str(Term((3, 1, 7))) == 'A^3*B*C^7', str(Term((3, 1, 7)))
        assert str(Term((3, 0, 7))) == 'A^3*C^7', str(Term((3, 0, 7)))
        assert str(Term((1, None, 0, 2))) == 'A*B^?*D^2', str(Term((1, None, 0, 2)))
        t = Term((3, 0, 7, None, 1))
        assert str(t) == 'A^3*C^7*D^?*E', str(t)
        assert Term.__str__(t, mulstr='') == 'A^3C^7D^?E', (t, Term.__str__(t, mulstr=''))
        assert Term.__str__(t, mulstr=' . ') == 'A^3 . C^7 . D^? . E', \
               (t, Term.__str__(t, mulstr=' . '))
        assert Term.__str__(t, simplify=True) == 'A^3*C^7*D^?*E', \
               (t, Term.__str__(t, simplify=True))
        assert Term.__str__(t, simplify=False) == 'A^3*B^0*C^7*D^?*E^1', \
               (t, Term.__str__(t, simplify=False))
        assert Term.__str__(t, varsstr='tuXYZ') == 't^3*X^7*Y^?*Z', \
               (t, Term.__str__(t, varsstr='tuXY'))
        assert Term.__str__(t, varsstr='tuXYZ', nonestr='None') == 't^3*X^7*Y^None*Z', \
               (t, Term.__str__(t, varsstr='tuXY', nonestr='None'))
        assert Term.__str__(t, varsstr='tuXYZ', mulstr=' . ', simplify=False) \
               == 't^3 . u^0 . X^7 . Y^? . Z^1', \
               (t, Term.__str__(t, varsstr='tuXYZ', mulstr=' . ', simplify=False))
        assert Term.__str__(t, varsstr='tuXYZ', expstr='^(', exprstr=')') == 't^(3)*X^(7)*Y^(?)*Z',\
               (t, Term.__str__(t, varsstr='tuXYZ', expstr='^(', exprstr=')'))
        print('ok'); sys.stdout.flush()


        print('Term.cmp_order()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Term.deriv()...', end=''); sys.stdout.flush()
        for n in range(-100, 100):
            assert Term.deriv(Term((n, ))) == (n, Term((n - 1, ))), (n, Term.deriv(Term((n, ))))
            assert Term.deriv(Term((n, 33))) == (n, Term((n - 1, 33))), \
                   (n, Term.deriv(Term((n, 33))))
            assert Term.deriv(Term((0, 1, 2, n, 4, 5)), i=3) == (n, Term((0, 1, 2, n - 1, 4, 5))), \
                   (n, Term.deriv(Term((0, 1, 2, n, 4, 5)), i=3))
            assert Term.deriv(Term((n, 33)), k=0) == (1, Term((n, 33))), \
                   (n, Term.deriv(Term((n, 33)), k=0))
            assert Term.deriv(Term((n, 33)), k=2) == (n*(n - 1), Term((n - 2, 33))), \
                   (n, Term.deriv(Term((n, 33)), k=2))
            assert Term.deriv(Term((n, 33)), k=3) == (n*(n - 1)*(n - 2), Term((n - 3, 33))), \
                   (n, Term.deriv(Term((n, 33)), k=3))
            if -1 <= n < 0:
                assert Term.deriv(Term((n, 33)), k=-1) == (None, Term((n + 1, 33))), \
                       (n, Term.deriv(Term((n, 33)), k=-1))
            else:
                assert Term.deriv(Term((n, 33)), k=-1) == (1/(n + 1), Term((n + 1, 33))), \
                       (n, Term.deriv(Term((n, 33)), k=-1))
            if -2 <= n < 0:
                assert Term.deriv(Term((n, 33)), k=-2) == (None, Term((n + 2, 33))), \
                       (n, Term.deriv(Term((n, 33)), k=-2))
            else:
                assert Term.deriv(Term((n, 33)), k=-2) \
                       == (1/((n + 1)*(n + 2)), Term((n + 2, 33))), \
                       (n, Term.deriv(Term((n, 33)), k=-2))
            if -3 <= n < 0:
                assert Term.deriv(Term((n, 33)), k=-3) == (None, Term((n + 3, 33))), \
                       (n, Term.deriv(Term((n, 33)), k=-3))
            else:
                assert Term.deriv(Term((n, 33)), k=-3) \
                       == (1/((n + 1)*(n + 2)*(n + 3)), Term((n + 3, 33))), \
                       (n, Term.deriv(Term((n, 33)), k=-3))
            assert Term.deriv(Term((None, 33)), k=n) == (None, Term((None, 33))), \
                   (None, Term.deriv(Term((None, 33)), k=n))
        print('ok'); sys.stdout.flush()


        print('Term.eval()...', end=''); sys.stdout.flush()
        t = Term()
        assert t.eval()  == 1, (t, t.eval())
        assert t.eval(0) == 1, (t, t.eval(0))
        assert t.eval(1) == 1, (t, t.eval(1))
        assert t.eval(2) == 1, (t, t.eval(2))
        assert t.eval(3) == 1, (t, t.eval(3))
        assert t.eval(None) == None, (t, t.eval(None))
        t = Term((2, ))
        assert t.eval()  == 1, (t, t.eval())
        assert t.eval(0) == 0, (t, t.eval(0))
        assert t.eval(1) == 1, (t, t.eval(1))
        assert t.eval(2) == 4, (t, t.eval(2))
        assert t.eval(3) == 9, (t, t.eval(3))
        assert t.eval(None) == None, (t, t.eval(None))
        t = Term((2, 3))
        assert t.eval()  == 1,    (t, t.eval())
        assert t.eval(0) == 0,    (t, t.eval(0))
        assert t.eval(1) == 1,    (t, t.eval(1))
        assert t.eval(2) == 4*8,  (t, t.eval(2))
        assert t.eval(3) == 9*27, (t, t.eval(3))
        assert t.eval(None) == None, (t, t.eval(None))
        t = Term((0, 3))
        assert t.eval()  == 1,  (t, t.eval())
        assert t.eval(0) == 0,  (t, t.eval(0))
        assert t.eval(1) == 1,  (t, t.eval(1))
        assert t.eval(2) == 8,  (t, t.eval(2))
        assert t.eval(3) == 27, (t, t.eval(3))
        assert t.eval(None) == None, (t, t.eval(None))

        t = Term()
        assert t.eval((0, 0)) == 1, (t, t.eval((0, 0)))
        assert t.eval((4, 0)) == 1, (t, t.eval((4, 0)))
        assert t.eval((4, 5)) == 1, (t, t.eval((4, 5)))
        assert t.eval(None) == None, (t, t.eval(None))
        t = Term((2, ))
        assert t.eval((0, 0)) == 0,  (t, t.eval((0, 0)))
        assert t.eval((4, 0)) == 16, (t, t.eval((4, 0)))
        assert t.eval((4, 5)) == 16, (t, t.eval((4, 5)))
        assert t.eval((4, None)) == 16, (t, t.eval((4, None)))
        t = Term((2, 3))
        assert t.eval((0, 0)) == 0,      (t, t.eval((0, 0)))
        assert t.eval((4, 0)) == 0,      (t, t.eval((4, 0)))
        assert t.eval((4, 5)) == 16*125, (t, t.eval((4, 5)))
        assert t.eval([4, 5]) == 16*125, (t, t.eval([4, 5]))
        assert t.eval((4, None)) == None, (t, t.eval((4, None)))
        t = Term((0, 3))
        assert t.eval((0, 0))    == 0,   (t, t.eval((0, 0)))
        assert t.eval((4, 0))    == 0,   (t, t.eval((4, 0)))
        assert t.eval((0, 5))    == 125, (t, t.eval((4, 0)))
        assert t.eval((4, 5))    == 125, (t, t.eval((4, 5)))
        assert t.eval((4, 5, 6)) == 125, (t, t.eval((4, 5, 6)))
        assert t.eval((4, None)) == None, (t, t.eval((4, None)))
        t = Term((-2, 3))
        assert t.eval((4, 0)) == 0,       (t, t.eval((4, 0)))
        assert t.eval((4, 5)) == 125/16,  (t, t.eval((4, 5)))
        assert t.eval((4, None)) == None, (t, t.eval((4, None)))
        t = Term((2, None))
        assert t.eval((4, 0)) == None, (t, t.eval((4, 0)))
        assert t.eval((4, 5)) == None, (t, t.eval((4, 5)))
        assert t.eval((4, None)) == None, (t, t.eval((4, None)))
        print('ok'); sys.stdout.flush()


        print('Term.partialeval()...', end=''); sys.stdout.flush()
        t = Term()
        assert t.partialeval(())           == (1, t), (t, t.partialeval(()))
        assert t.partialeval((0, 5))       == (1, t), (t, t.partialeval((0, 5)))
        assert t.partialeval((4, 5))       == (1, t), (t, t.partialeval((4, 5)))
        assert t.partialeval((None, 5))    == (1, t), (t, t.partialeval((None, 5)))
        assert t.partialeval((4, None))    == (1, t), (t, t.partialeval((4, None)))
        assert t.partialeval((4, None, 6)) == (1, t), (t, t.partialeval((4, None, 6)))
        t = Term((2, 3))
        assert t.partialeval((0, 5))       == (0, (0, 0)),      (t, t.partialeval((0, 5)))
        assert t.partialeval((4, 5))       == (16*125, (0, 0)), (t, t.partialeval((4, 5)))
        assert t.partialeval((None, 5))    == (125, (2, 0)),    (t, t.partialeval((None, 5)))
        assert t.partialeval((4, None))    == (16, (0, 3)),     (t, t.partialeval((4, None)))
        assert t.partialeval((4, None, 6)) == (16, (0, 3)),     (t, t.partialeval((4, None, 6)))
        t = Term((-2, 3))
        assert t.partialeval((4, 5))       == (125/16, (0, 0)), (t, t.partialeval((4, 5)))
        assert t.partialeval((None, 5))    == (125, (-2, 0)),   (t, t.partialeval((None, 5)))
        assert t.partialeval((4, None))    == (1/16, (0, 3)),   (t, t.partialeval((4, None)))
        assert t.partialeval((4, None, 6)) == (1/16, (0, 3)),   (t, t.partialeval((4, None, 6)))
        t = Term((2, None))
        assert t.partialeval((0, 5))       == (None, (0, None)), (t, t.partialeval((0, 5)))
        assert t.partialeval((4, 5))       == (None, (0, None)), (t, t.partialeval((4, 5)))
        assert t.partialeval((None, 5))    == (None, (2, None)), (t, t.partialeval((None, 5)))
        assert t.partialeval((4, None))    == (16, (0, None)),   (t, t.partialeval((4, None)))
        assert t.partialeval((4, None, 6)) == (16, (0, None)),   (t, t.partialeval((4, None, 6)))
        print('ok'); sys.stdout.flush()



        print()
        print('Polynomial()...', end=''); sys.stdout.flush()
        assert Polynomial() == {}, Polynomial()
        assert Polynomial(()) == {}, Polynomial(())
        assert Polynomial(Term()) == {Term(): 1}, Polynomial(Term())
        assert Polynomial(Term((0, 0))) == {Term((0, 0)): 1}, Polynomial(Term((0, 0)))
        t = Term((2, 3))
        assert Polynomial(t) == {t: 1}, (t, Polynomial(t))
        u = Term((5, 4))
        assert Polynomial((t, u)) == {t: 1, u: 1}, (t, u, Polynomial((t, u)))
        assert Polynomial([t, u]) == {t: 1, u: 1}, (t, u, Polynomial([t, u]))
        assert Polynomial((t, t)) == {t: 2}, (t, u, Polynomial((t, t)))
        assert Polynomial({t:8, t:5}) == {t: 5}, (t, u, Polynomial({t:8, t:5}))

        assert Polynomial(coef=3) == {}, Polynomial(coef=3)
        assert Polynomial((t, u), coef=-7) == {t: -7, u: -7}, (t, u, Polynomial((t, u), coef=-7))
        assert Polynomial([t, u], coef=-7) == {t: -7, u: -7}, (t, u, Polynomial([t, u], coef=-7))
        assert Polynomial((t, t), coef=-7) == {t: -14}, (t, u, Polynomial((t, t), coef=-7))
        assert Polynomial([t, u, t], coef=-7) == {t: -14, u: -7}, \
               (t, u, Polynomial([t, u, t], coef=-7))
        assert Polynomial({t:8, t:5}, coef=-7) == {t: -35}, (t, u, Polynomial({t:8, t:5}, coef=-7))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.__iadd__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.__repr__()...', end=''); sys.stdout.flush()
        assert repr(Polynomial()) == "'Polynomial({{}}, varsstr=\"{0}\")'".format(ALPHA), \
               repr(Polynomial())
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.__str__()...', end=''); sys.stdout.flush()
        assert str(Polynomial()) == '', str(Polynomial())
        assert str(Polynomial(Term())) == '1', str(Polynomial(Term()))
        assert str(Polynomial(Term((0, 0)))) == '1', str(Polynomial(Term((0, 0))))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.deriv()...', end=''); sys.stdout.flush()
        p = Polynomial({Term((3, 0, 7, 1)):5, Term((-4, 2, 3, 1)):-3})
        assert p.deriv() == Polynomial({Term((2, 0, 7, 1)):15, Term((-5, 2, 3, 1)):12}), p.deriv()
        assert p.deriv(k=2) == Polynomial({Term((1, 0, 7, 1)):30, Term((-6, 2, 3, 1)):-60}), \
               p.deriv(k=2)
        assert p.deriv(k=-1) == Polynomial({Term((4, 0, 7, 1)):1.25, Term((-3, 2, 3, 1)):1}), \
               p.deriv(k=-1)
        assert p.deriv(i=1) == Polynomial({Term((3, -1, 7, 1)):0, Term((-4, 1, 3, 1)):-6}), \
               p.deriv(i=1)
        assert p.deriv(i=2) == Polynomial({Term((3, 0, 6, 1)):35, Term((-4, 2, 2, 1)):-9}), \
               p.deriv(i=2)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.eval()...', end=''); sys.stdout.flush()
        assert Polynomial().eval() == 0, Polynomial().eval()
        assert Polynomial(Term()).eval() == 1, Polynomial(Term()).eval()
        assert Polynomial(Term((0, 0))).eval() == 1, Polynomial().eval(Term((0, 0)))
        p = Polynomial(Term((3, 0, 7, 1)))
        assert p.eval() == 1, (p, p.eval())
        assert p.eval(1) == 1, (p, p.eval(1))
        assert p.eval(2) == 8*128*2, (p, p.eval(2))
        assert p.eval((5, 6, 2, 13)) == 125*128*13, (p, p.eval((5, 6, 2, 13)))
        assert p.eval([5, 6, 2, 13]) == 125*128*13, (p, p.eval([5, 6, 2, 13]))
        assert p.eval((5, 0, 2, 13)) == 125*128*13, (p, p.eval((5, 0, 2, 13)))
        assert p.eval((5, 6, 2, 0))  == 0,          (p, p.eval((5, 6, 2, 0)))
        p = Polynomial(Term((3, 0, 7, 1)), coef=-3)
        assert p.eval() == -3, (p, p.eval())
        assert p.eval(1) == -3, (p, p.eval(1))
        assert p.eval(2) == -3*8*128*2, (p, p.eval(2))
        assert p.eval((5, 6, 2, 13)) == -3*125*128*13, (p, p.eval((5, 6, 2, 13)))
        assert p.eval([5, 6, 2, 13]) == -3*125*128*13, (p, p.eval([5, 6, 2, 13]))
        assert p.eval((5, 0, 2, 13)) == -3*125*128*13, (p, p.eval((5, 0, 2, 13)))
        assert p.eval((5, 6, 2, 0))  == 0,             (p, p.eval((5, 6, 2, 0)))
        p = Polynomial((Term((3, 0, 7, 1)), Term((-2, 2, 3, 1))), coef=-3)
        assert p.eval() == -6, (p, p.eval())
        assert p.eval(1) == -6, (p, p.eval(1))
        assert p.eval(2) == -3*8*128*2 - 3/4*4*8*2,    (p, p.eval(2))
        assert p.eval((5, 6, 2, 13)) == -3*125*128*13 - 3/25*36*8*13, (p, p.eval((5, 6, 2, 13)))
        assert p.eval((5, 0, 2, 13)) == -3*125*128*13, (p, p.eval((5, 0, 2, 13)))
        assert p.eval((5, 6, 2, 0))  == 0,             (p, p.eval((5, 6, 2, 0)))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.lenterms()...', end=''); sys.stdout.flush()
        assert Polynomial().lenterms() == None, Polynomial().lenterms()
        assert Polynomial(Term()).lenterms() == 0, Polynomial(Term()).lenterms()
        assert Polynomial(Term((0, 0))).lenterms() == 2, Polynomial(Term((0, 0))).lenterms()
        assert Polynomial(Term((2, 3))).lenterms() == 2, Polynomial(Term((2, 3))).lenterms()
        assert Polynomial((Term((2, 3)), Term((4, 5)), Term((6, 7)))).lenterms() == 2, \
               Polynomial((Term((2, 3)), Term((4, 5)), Term((6, 7)))).lenterms()
        assert Polynomial(Term((7, 8, 9))).lenterms() == 3, Polynomial(Term((7, 8, 9))).lenterms()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Polynomial.partialeval()...', end=''); sys.stdout.flush()
        assert Polynomial().partialeval(()) == {}, Polynomial().partialeval(())
        p = Polynomial()
        assert p.partialeval(())        == {}, (p, p.partialeval(()))
        assert p.partialeval((0, 5))    == {}, (p, p.partialeval((0, 5)))
        assert p.partialeval((None, 5)) == {}, (p, p.partialeval((None, 5)))
        p = Polynomial(Term((2, 3)))
        assert p.partialeval((0, 5)) == Polynomial(Term((0, 0)), coef=0), \
               (p, p.partialeval((0, 5)))
        assert p.partialeval((4, 5)) == Polynomial(Term((0, 0)), coef=16*125), \
               (p, p.partialeval((4, 5)))
        assert p.partialeval([4, 5]) == Polynomial(Term((0, 0)), coef=16*125), \
               (p, p.partialeval([4, 5]))
        assert p.partialeval((None, 5)) == Polynomial(Term((2, 0)), coef=125), \
               (p, p.partialeval((None, 5)))
        assert p.partialeval((4, None)) == Polynomial(Term((0, 3)), coef=16), \
               (p, p.partialeval((4, None)))
        assert p.partialeval((4, None, 6)) == Polynomial(Term((0, 3)), coef=16), \
               (p, p.partialeval((4, None, 6)))
        p = Polynomial((Term((3, 0, 7, 1)), Term((-2, 2, 3, 1))), coef=-3)
        assert p.partialeval((5, 0, 2, 13)) == Polynomial(Term((0, 0, 0, 0)), coef=-3*125*128*13), \
               (p, p.partialeval((5, 0, 2, 13)))
        assert p.partialeval((5, 6, 2, 13)) \
               == Polynomial(Term((0, 0, 0, 0)), coef= -3*125*128*13 - 3/25*36*8*13), \
               (p, p.partialeval((5, 6, 2, 13)))
        assert p.partialeval((5, None, 2, 13)) \
               == Polynomial({Term((0, 0, 0, 0)): -3*125*128*13,
                              Term((0, 2, 0, 0)): -3/25*8*13}), \
                              (p, p.partialeval((5, None, 2, 13)))
        print('???', end='')
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
