#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.nbsystem Systmes de numration
# !!! Work in progress !!!

##\file
# Systmes de numration
# !!! Work in progress !!!

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 2 aot 2006
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'nbsystem --- 2010 March 16'

import DSPython
import DSPython.natural as natural



# ###########
# Fonctions #
#############
## Renvoie le chiffre correspondant au caractre c
def chr_to_fig(c):
    """Renvoie le chiffre correspondant au caractre c

    Pre: c: caractre alphanumrique

    Result: naturel <= 36

    O(1) = ..."""
    assert isinstance(c, str), type(c)
    assert c.isalnum(), (type(c), c)

    return (ord(c) - ord('0') if c.isdigit()
            else ord(c.upper()) + 10 - ord('A'))


## Naturel correspondant  la reprsentation de n dans le systme de numration de Fibonacci
def fibonacci(n):
    """Renvoie le naturel reprsentant n (en binaire)
      dans le systme de numration de Fibonacci
                    k                                k
    C.--d. si n = Sum(b_i * F_(i+2)) alors renvoie Sum(b_i * 2**i)
                   i=0                              i=0
      o b_i = 0 ou 1 tel que si b_i = 1 alors b_(i+1) = 0
         F_(i+2) = (i+2)me nombre de Fibonacci

    Pre: n: naturel

    Result: naturel "normalis"
              suivant le systme de numration de Fibonacci

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    if n > 2:
        # Cherche un i tel que n <= F_(i+2)
        i = natural.lg(n)
        (fi_1, fi_2) = natural.fibonacci2(i + 2)  # (F_(i+1), F_(i+2))
        while n > fi_2:
            i += 1
            (fi_1, fi_2) = (fi_2, fi_2 + fi_1)

        # Calcul f par algorithme glouton
        f = 0
        while n > 0:
            while fi_2 > n:
                i -= 1
                (fi_1, fi_2) = (fi_2 - fi_1, fi_1)
            n -= fi_2
            f = natural.bitset1(f, i)
            # Le suivant ne peut convenir car pas deux '1' conscutifs
            i -= 1
            (fi_1, fi_2) = (fi_2 - fi_1, fi_1)
        return f
    else:
        return n


## Naturel gal  la valeur de f exprime dans le systme de numration de Fibonacci
def fibonacci_to_n(f):
    """Renvoie le naturel gal
       la valeur de f exprime dans le systme de numration de Fibonacci
                    k                             k
    C.--d. si f = Sum(b_i * 2**i) alors renvoie Sum(b_i * F_(i+2))
                   i=0                           i=0
      o b_i = 0 ou 1
         F_(i+2) = (i+2)me nombre de Fibonacci

    Pre: f: naturel ("normalis" ou non)

    Result: naturel

    O(f) = ..."""
    assert DSPython.natural_is(f), f

    if f > 2:
        n = 0
        from_i = natural.scan1(f)
        (fi_1, fi_2) = natural.fibonacci2(from_i + 2)  # (F_(i+1), F_(i+2))
        for i in range(from_i, natural.rscan1(f) + 1):
            if natural.bit(f, i):  # chiffre '1'
                n += fi_2
            (fi_1, fi_2) = (fi_2, fi_2 + fi_1)
            i += 1
        return n
    else:
        return f


## "Normalise" f
def fibonacci_to_normal(f):
    """Renvoie n "normalis" suivant le systme de numration de Fibonacci,
    c.--d. renvoie le naturel reprsentant le mme nombre que f
      dans le systme de numration de Fibonacci
      mais ne contenant pas deux chiffres '1' conscutifs

    Pre: f: naturel

    Result: naturel "normalis"
              suivant le systme de numration de Fibonacci

    O(f) = ..."""
    assert DSPython.natural_is(f), f

    if f > 2:
        k = natural.rscan1(f) - 1
        mask111  = 7 << k
        mask11   = 3 << k
        mask1100 = mask11 << 2
        k = 0
        # Parcours f de la gauche vers la droite
        while True:
            while f&mask11 != mask11:  # avance tant que pas deux chiffres '1' conscutifs
                k += 1
                mask11 >>= 1
            if mask11 <= 1:  # parcours termin
                return f
            mask1100 >>= k
            mask111  >>= k

            f ^= mask111  # remplace '011' par '100'
            if f&mask1100 == mask1100:  # faut reculer car form '1100'
                k = 0
                mask11     = mask1100
                mask1100 <<= 2
                mask111  <<= 2
            else:                       # avancer
                k = 2
                mask11 >>= 2
    else:
        return f


## ime chiffre de n en base b
def fig(n, i, b=2):
    """Renvoie le ime chiffre de n en base b

    Pre: n: naturel
         i: naturel
         b: naturel >= 2

    Result: naturel < b

    O(n, i, b) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(i), i
    assert DSPython.natural_is(b), b
    assert b >= 2, b

    if b == 2:                     # b == 2
        return int(natural.bit(n, i))
    elif natural.nbbits1(b) == 1:  # b == 2**k
        k = natural.scan1(b)
        return (n >> (k*i)) % b
    else:                          # b > 2
        while (i > 0) and (n > 0):
            i -= 1
            n //= b
        return n%b


## Renvoie le caractre correspondant au chiffre n
def fig_to_chr(n, caps=False):
    """Renvoie le caractre correspondant au chiffre n
      Si caps alors utlise les lettres majuscules pour reprsenter les chiffers > 9
              sinon utlise les lettres minuscules

    Pre: n: naturel <= 36
         caps: boolean

    Result: caractre alphanumrique

    O(1) = ..."""
    assert DSPython.natural_is(n), n
    assert n <= 36, n

    return (chr(n + ord('0')) if n < 10
            else chr(n + (ord('A') if caps
                          else ord('a')) - 10))


## Change le ime chiffre de n en base b
def figset(n, i, c, b=2):
    """Renvoie n avec son ime chiffre en base b chang en c

    Pre: n: naturel
         i: naturel
         c: naturel < 2
         b: naturel >= 2

    Result: naturel

    O(n, i, c, b) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(i), i
    assert DSPython.natural_is(c), c
    assert c < b, (c, b)
    assert DSPython.natural_is(b), b
    assert b >= 2, b

    if b == 2:  # b == 2
        return natural.bitset(n, i, c)
    else:       # b > 2
        p = b**i
        (q, r) = divmod(n, p)  # "reste" : chiffres qui prcdent
        q //= b                # "quotient" : chiffres qui suivent
        return q*(p*b) + (c*p + r)


## Renvoie le naturel correspondant  la liste s de chiffres en base b
def list_to(s, b=2):
    """Renvoie le naturel correspondant  la liste s de chiffres en base b
      (Si n == [] alors renvoie 0)

    Pre: s: liste de naturels
         b: naturel >= 2

    Result: naturel

    O(n, b) = ..."""
    assert DSPython.natural_is(b), b
    assert b >= 2, b

    n = 0
    if b == 2:
        for i in range(len(s)):
            assert DSPython.natural_is(s[i]), s[i]

            n = natural.bitset(n, i, s[i])
    else:
        p = 1
        for c in s:
            assert DSPython.natural_is(c), c

            n += p*c
            p *= b
    return n


## Renvoie le naturel reprsent par s en base b
def str_to(s, b=2):
    """Renvoie le naturel reprsent par s en base b
      (Si n == 0 alors renvoie '0')

    Pre: s: string ne contenant que des chiffres < b
         b: 2 <= naturel <= 36

    Result: naturel

    O(s, b) = ..."""
    assert isinstance(s, str), type(s)
    assert DSPython.natural_is(b), b
    assert 2 <= b <= 36, b

    n = 0
    if b == 2:
        for i in range(len(s)):
            assert s[-i-1].isalnum(), (type(s[-i-1]), s[-i-1])

            n = natural.bitset(n, i, s[-i-1] != '0')
    else:
        p = 1
        for i in range(len(s)):
            assert s[-i-1].isalnum(), (type(s[-i-1]), s[-i-1])

            n += p*chr_to_fig(s[-i-1])
            p *= b
    return n


## Somme des chiffres de n en base b
def sum_figs(n, b=2):
    """Renvoie la somme des chiffres de n reprsent en base b

    Pre: n: naturel
         b: naturel >= 2

    Result: naturel

    O(n, b) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(b), b
    assert b >= 2, b

    if b == 2:                     # b == 2
        return natural.nbbits1(n)
    elif natural.nbbits1(b) == 1:  # b == 2**k
        k = natural.scan1(b)
        m_k = natural.mersenne(k)
        s = 0
        while n > 0:
            s += n & m_k
            n >>= k
        return s
    else:                          # autre b
        s = 0
        while n > 0:
            (n, r) = divmod(n, b)
            s += r
        return s


## Somme des chiffres alterne de n en base b
def sum_figs_alt(n, b=2):
    """Renvoie la somme des chiffres altern de n reprsent en base b
    (1er chiffre en positif, 2e en ngatif, 3e en positif ...)

    Pre: n: naturel
         b: naturel >= 2

    Result: naturel

    O(n, b) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(b), b
    assert b >= 2, b

    s = 0
    while n > 0:
        (n, r) = divmod(n, b)
        s += r
        (n, r) = divmod(n, b)
        s -= r
    return s


## Renvoie la liste des chiffres de n crit en base b
def to_list(n, b=2):
    """Renvoie la liste des chiffres de n crit en base b
      (Si n == 0 alors renvoie [])

    Pre: n: naturel
         b: naturel >= 2

    Result: liste de naturels (dont le dernier lmnt est non nul)

    O(n, b) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(b), b
    assert b >= 2, b

    l = []
    if b == 2:
        while n > 0:
            l.append(n & 1)
            n >>= 1
    else:
        while n > 0:
            (n, r) = divmod(n, b)
            l.append(r)
    return l


## n crit en base b dans un string
def to_str(n, b=2, caps=False):
    """Renvoie un string reprsentant n en base b
      (Si n == 0 alors renvoie '0')
      Si caps alors utlise les lettres majuscules pour reprsenter les chiffers > 9
              sinon utlise les lettres minuscules

    Pre: n: naturel
         b: 2 <= naturel <= 36
         caps: boolean

    Result: string (dont le premier caractre est diffrent de '0')

    O(n, b) = ..."""
    assert DSPython.natural_is(n), n
    assert DSPython.natural_is(b), b
    assert 2 <= b <= 36, b

    if n > 0:
        if b == 2:
            return natural.bin(n)
        else:
            l = []
            while n > 0:
                (n, r) = divmod(n, b)
                l.insert(0, fig_to_chr(r, caps=caps))
            return ''.join(l)
    else:
        return '0'



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        import DSPython.debug as debug
        import DSPython.bit32 as bit32
        import DSPython.nat32 as nat32

        debug.test_begin(VERSION, __debug__)

        print('chr_to_fig()...', end=''); sys.stdout.flush()
        assert chr_to_fig('0') == 0,  chr_to_fig('0')
        assert chr_to_fig('1') == 1,  chr_to_fig('1')
        assert chr_to_fig('2') == 2,  chr_to_fig('2')
        assert chr_to_fig('9') == 9,  chr_to_fig('9')
        assert chr_to_fig('a') == 10, chr_to_fig('a')
        assert chr_to_fig('A') == 10, chr_to_fig('A')
        assert chr_to_fig('z') == 35, chr_to_fig('z')
        assert chr_to_fig('Z') == 35, chr_to_fig('Z')
        print('ok'); sys.stdout.flush()


        print('fibonacci()...', end=''); sys.stdout.flush()
        assert fibonacci(0) == 0, (fibonacci(0), bit32.bin(fibonacci(0)))
        assert fibonacci(1) == 1, (fibonacci(1), bit32.bin(fibonacci(1)))
        assert fibonacci(2) == 2, (fibonacci(2), bit32.bin(fibonacci(2)))
        assert fibonacci(3) == 4, (fibonacci(3), bit32.bin(fibonacci(3)))
        assert fibonacci(4) == 5, (fibonacci(4), bit32.bin(fibonacci(4)))
        assert fibonacci(5) == 8, (fibonacci(5), bit32.bin(fibonacci(5)))
        for n in range(10000):
            f = fibonacci(n)
            assert bit32.bin(f).find('11') == -1, (n, f, bit32.bin(f))
            s = 0
            for i in range(bit32.nbbits(f)):
                if bit32.bit(f, i):
                    s += nat32.fibonacci(i + 2)
            assert s == n, (n, s, f, bit32.bin(f))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('fibonacci_to_n()...', end=''); sys.stdout.flush()
        assert fibonacci_to_n(0) == 0, (fibonacci_to_n(0), bit32.bin(fibonacci_to_n(0)))
        assert fibonacci_to_n(1) == 1, (fibonacci_to_n(1), bit32.bin(fibonacci_to_n(1)))
        assert fibonacci_to_n(2) == 2, (fibonacci_to_n(2), bit32.bin(fibonacci_to_n(2)))
        assert fibonacci_to_n(3) == 3, (fibonacci_to_n(3), bit32.bin(fibonacci_to_n(3)))
        assert fibonacci_to_n(4) == 3, (fibonacci_to_n(4), bit32.bin(fibonacci_to_n(4)))
        assert fibonacci_to_n(5) == 4, (fibonacci_to_n(5), bit32.bin(fibonacci_to_n(5)))
        for k in range(500):
            assert fibonacci_to_n(1<<k) == natural.fibonacci(k + 2), \
                   (k, fibonacci_to_n(1<<k), natural.fibonacci(k + 2))
            assert fibonacci_to_n((1<<k) + 1) == natural.fibonacci(k + 2) + 1, \
                   (k, fibonacci_to_n((1<<k) + 1), natural.fibonacci(k + 2) + 1)
            assert fibonacci_to_n((1<<k) + (1<<(k+1))) == natural.fibonacci(k + 4), \
                   (k, fibonacci_to_n((1<<k) + (1<<(k+1))), natural.fibonacci(k + 4))
            assert fibonacci_to_n((1<<k) + (1<<(k+2))) \
                   == natural.fibonacci(k + 2) + natural.fibonacci(k + 4), \
                   (k, fibonacci_to_n((1<<k) + (1<<(k+2))),
                    natural.fibonacci(k + 2) + natural.fibonacci(k + 4))
        for n in range(10000):
            f = fibonacci(n)
            assert fibonacci_to_n(f) == n, (f, fibonacci_to_n(f), n,
                                            bit32.bin(fibonacci_to_n(f)), bit32.bin(n))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('fibonacci_to_normal()...', end=''); sys.stdout.flush()
        assert fibonacci_to_normal(0) == 0, \
               (bit32.bin(0), fibonacci_to_normal(0), bit32.bin(fibonacci_to_normal(0)))
        assert fibonacci_to_normal(1) == 1, \
               (bit32.bin(1), fibonacci_to_normal(1), bit32.bin(fibonacci_to_normal(1)))
        assert fibonacci_to_normal(2) == 2, \
               (bit32.bin(2), fibonacci_to_normal(2), bit32.bin(fibonacci_to_normal(2)))
        assert fibonacci_to_normal(3) == 4, \
               (bit32.bin(3), fibonacci_to_normal(3), bit32.bin(fibonacci_to_normal(3)))
        assert fibonacci_to_normal(4) == 4, \
               (bit32.bin(4), fibonacci_to_normal(4), bit32.bin(fibonacci_to_normal(4)))
        assert fibonacci_to_normal(5) == 5, \
               (bit32.bin(5), fibonacci_to_normal(5), bit32.bin(fibonacci_to_normal(5)))
        for n in range(10000):
            f = fibonacci_to_normal(n)
            assert bit32.bin(f).find('11') == -1, (n, bit32.bin(n), bit32.bin(f), f)
            assert f == fibonacci(fibonacci_to_n(n)), (n, f, fibonacci(fibonacci_to_n(n)))
            assert fibonacci_to_n(n) == fibonacci_to_n(f), (n, fibonacci_to_n(n), fibonacci_to_n(f))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('fig()...', end=''); sys.stdout.flush()
        for n in range(500):
            for b in range(2, 20):
                assert fig(n, 0, b) == n%b, (n, b, fig(n, 0, b), n%b)
                for k in range(10):
                    assert fig(n, k, b) == (n//(b**k))%b, (n, b, k, fig(n, k, b), (n//(b**k))%b)
        print('ok'); sys.stdout.flush()


        print('fig_to_chr()...', end=''); sys.stdout.flush()
        assert fig_to_chr(0) == '0', fig_to_chr(0)
        assert fig_to_chr(1) == '1', fig_to_chr(1)
        assert fig_to_chr(2) == '2', fig_to_chr(2)
        assert fig_to_chr(9) == '9', fig_to_chr(9)
        assert fig_to_chr(10)            == 'a', fig_to_chr(10)
        assert fig_to_chr(10, caps=True) == 'A', fig_to_chr(10, caps=True)
        assert fig_to_chr(35)            == 'z', fig_to_chr(35)
        assert fig_to_chr(35, caps=True) == 'Z', fig_to_chr(35, caps=True)
        print('ok'); sys.stdout.flush()


        print('figset()...', end=''); sys.stdout.flush()
        for n in (0, 123):
            for b in range(2, 40 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
                for c in range(b):
                    for k in range(10):
                        new = figset(n, k, c, b)
                        assert fig(new, k, b) == c, (new, b, k, fig(new, k, b), c)
                        for l in range(10):
                            if l != k:
                                assert fig(new, l, b) == fig(n, l, b), \
                                       (new, b, k, l, fig(new, l, b), fig(n, l, b))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('list_to()...', end=''); sys.stdout.flush()
        assert list_to([0, 0, 1])   == 4,  list_to([0, 0, 1])
        assert list_to((0, 0, 1))   == 4,  list_to((0, 0, 1))
        assert list_to([1, 0, 1])   == 5,  list_to([1, 0, 1])
        assert list_to([2, 1], 3)   == 5,  list_to([2, 1], 3)
        assert list_to([5], 10)     == 5,  list_to([5], 10)
        assert list_to([14, 1], 16) == 30, list_to([14, 1], 16)
        for b in range(2, 50):
            for n in range(1000):
                s = to_list(n, b)
                new = list_to(s, b)
                assert s == to_list(new, b), (b, n, new, s, to_list(new, b))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('str_to()...', end=''); sys.stdout.flush()
        assert str_to('100')    == 4,  str_to('100')
        assert str_to('101')    == 5,  str_to('101')
        assert str_to('12', 3)  == 5,  str_to('12', 3)
        assert str_to('5', 10)  == 5,  str_to('5', 10)
        assert str_to('1e', 16) == 30, str_to('1e', 16)
        for b in range(2, 37):
            for n in range(1000):
                s = to_str(n, b)
                new = str_to(s, b)
                assert s == to_str(new, b), (b, n, new, s, to_str(new, b))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('sum_figs()...', end=''); sys.stdout.flush()
        for k in range(500):
            assert sum_figs(2**k - 1) == k, (k, sum_figs(2**k - 1))
            assert sum_figs(2**k) == 1, (k, sum_figs(2**k))
        for b in range(2, 50):
            assert sum_figs(0, b) == 0, (b, sum_figs(0, b))
            assert sum_figs(1, b) == 1, (b, sum_figs(1, b))
            for n in range(2**10):
                assert sum_figs(n, b) == sum_figs(n//b, b) + n%b, \
                       (b, n, sum_figs(n, b), sum_figs(n//b, b))
            for k in range(50):
                assert sum_figs(b**k, b) == 1, (b, k, sum_figs(b**k, b))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('sum_figs_alt()...', end=''); sys.stdout.flush()
        for k in range(50):
            assert sum_figs_alt(2**(2*k) - 1) == 0, (k, sum_figs_alt(2**(2*k) - 1))
            assert sum_figs_alt(2**(2*k)) == 1, (k, sum_figs_alt(2**(2*k)))

            assert sum_figs_alt(2**(2*k+1) - 1) == 1, (k, sum_figs_alt(2**(2*k+1) - 1))
            assert sum_figs_alt(2**(2*k+1)) == -1, (k, sum_figs_alt(2**(2*k+1)))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('to_list()...', end=''); sys.stdout.flush()
        assert to_list(0)      == [],        to_list(0)
        assert to_list(4)      == [0, 0, 1], to_list(4)
        assert to_list(5)      == [1, 0, 1], to_list(5)
        assert to_list(5, 3)   == [2, 1],    to_list(5, 3)
        assert to_list(5, 10)  == [5],       to_list(5, 10)
        assert to_list(30, 16) == [14, 1],   to_list(30, 16)
        for n in range(1, 1000):
            l = [int(c) for c in natural.bin(n)]
            l.reverse()
            assert to_list(n) == l, (n, to_list(n), l)
        for b in range(2, 50):
            assert to_list(0, b) == [], (b, to_list(0, b))
            assert to_list(1, b) == [1], (b, to_list(1, b))
            for n in range(1, b):
                assert to_list(n, b) == [n], (b, n, to_list(n, b))
            for i in range(1, 50):
                assert to_list(b**i - 1, b) == [b - 1]*i, \
                       (b, i, to_list(b**i - 1, b))  # [b-1,b-1,...,b-1]
                assert to_list(b**i, b) == [0]*i + [1], \
                       (b, i, to_list(b**i, b))      # [0,0,...,0,1]
                assert to_list(b**i + 1, b) == [1] + [0]*(i - 1) + [1], \
                       (b, i, to_list(b**i + 1, b))  # [1,0,...,0,1]
            for n in range(1, 100):
                assert list_to(to_list(n, b), b) == n, \
                       (b, n, to_list(n, b), list_to(to_list(n, b), b))
            for n in range(1, 2**40, 10000000000):
                assert list_to(to_list(n, b), b) == n, \
                       (b, n, to_list(n, b), list_to(to_list(n, b), b))
            for n in range(2**32 - 100, 2**32 + 100):
                assert list_to(to_list(n, b), b) == n, \
                       (b, n, to_list(n, b), list_to(to_list(n, b), b))
        print('ok'); sys.stdout.flush()


        print('to_str()...', end=''); sys.stdout.flush()
        assert to_str(0)     == '0',   to_str(0)
        assert to_str(4)     == '100', to_str(4)
        assert to_str(5)     == '101', to_str(5)
        assert to_str(5, 3)  == '12',  to_str(5, 3)
        assert to_str(5, 10) == '5',   to_str(5, 10)
        assert to_str(30, 16)            == '1e', to_str(30, 16)
        assert to_str(30, 16, caps=True) == '1E', to_str(30, 16, caps=True)
        for n in range(1000):
            assert to_str(n) == natural.bin(n), (n, to_str(n), natural.bin(n))
        for b in range(2, 37):
            assert to_str(0, b) == '0', (b, to_str(0, b))
            assert to_str(1, b) == '1', (b, to_str(1, b))
            for n in range(0, b):
                assert to_str(n, b) == fig_to_chr(n), (b, n, to_str(n, b), fig_to_chr(n))
            for i in range(1, 50):
                assert to_str(b**i - 1, b) == fig_to_chr(b - 1)*i, \
                       (b, i, to_str(b**i - 1, b))  #  c...cc
                assert to_str(b**i, b) == '1' + '0'*i, \
                       (b, i, to_str(b**i, b))      # 10...00
                assert to_str(b**i + 1, b) == '1' + '0'*(i - 1) + '1', \
                       (b, i, to_str(b**i + 1, b))  # 10...01
            for n in range(1, 100):
                assert int(to_str(n, b), b) == n, (b, n, to_str(n, b))
            if debug.assertspeed >= debug.ASSERT_NORMAL:
                for n in range(1, 2**40, 10000000000):
                    assert int(to_str(n, b), b) == n, (b, n, to_str(n, b))
                for n in range(2**32 - 100, 2**32 + 100):
                    assert int(to_str(n, b), b) == n, (b, n, to_str(n, b))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
