#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.intmod Entiers modulo k (lments de \htmlonly &#8484;<sub>k</sub>\endhtmlonly)

##\file
# Entiers modulo k (lments de \htmlonly &#8484;<sub>k</sub>\endhtmlonly)

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 12 dcembre 2009
####################################
from __future__ import division
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'intmod --- 2010 March 16'

import numbers

import DSPython



# ########
# Classe #
##########
##\brief Type nombre entier modulo k
# \htmlonly(c.--d.\endhtmlonly un lment de \htmlonly&#8484;<sub>k</sub>\endhtmlonly)
class Intmod(int):
    """Type nombre entier modulo k (c.--d. un lment de Z_k) :
    0, 1, 2, &hellip;, k-1"""

    ## Cre l'lment
    def __new__(cls, value, k, base=10):
        """Cre l'lment

        Pre: value: Integral ou String
             k: Integral >= 1"""
        assert isinstance(value, numbers.Integral) or isinstance(value, str), value
        assert DSPython.natural_is(k), k
        assert k >= 1, k

        if isinstance(value, str):
            value = int(value, base=base)

        assert isinstance(value, numbers.Integral), value

        return int.__new__(cls, value%k)


    ## Initialise l'lment  self%k
    def __init__(self, value, k):
        """Initialise l'lment  self%k

        Pre: value: Integral ou String
             k: Integral >= 1"""
        assert isinstance(value, numbers.Integral) or isinstance(value, str), value
        assert DSPython.natural_is(k), k
        assert k >= 1, k

        self._k = k



    ## Renvoie self + other
    def __add__(self, other):
        """Renvoie self + other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__add__(self, other), self._k)


    ## Renvoie le naturel correspondant  la valeur self
    def __int__(self):
        """Renvoie le naturel correspondant  la valeur self

        Result: Integral < self.k()"""
        return int.__int__(self)


    ## Renvoie ~self
    def __invert__(self):
        """Renvoie ~self (modulo self.k())

        Result: Intmod(, k=self.k())"""
        return Intmod(int.__invert__(self), self._k)


    ## Renvoie self<<other
    def __lshift__(self, other):
        """Renvoie self<<other (modulo self.k())

        Pre: other: Intmod ou (Integral >= 0)

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__lshift__(self, other), self._k)


    ## Renvoie self*other
    def __mul__(self, other):
        """Renvoie self*other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__mul__(self, other), self._k)


    ## Renvoie -self
    def __neg__(self):
        """Renvoie -self (modulo self.k())

        Result: Intmod(, k=self.k())"""
        return Intmod(int.__neg__(self), self._k)


    ## Renvoie self | other
    def __or__(self, other):
        """Renvoie self | other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__or__(self, other), self._k)


    ## Renvoie self**other
    def __pow__(self, other):
        """Renvoie self**other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__pow__(self, (other if isinstance(other, Intmod) or (other >= 0)
                                         else self._k - other), self._k), self._k)


    ## Renvoie self - other
    def __sub__(self, other):
        """Renvoie self - other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__sub__(self, other), self._k)


    ## Renvoie self ^ other
    def __xor__(self, other):
        """Renvoie self ^ other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: Intmod(, k=self.k())"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return Intmod(int.__xor__(self, other), self._k)


    ## Renvoie la valeur du modulo k
    def k(self):
        """Renvoie la valeur du modulo k

        Result: Integral >= 1"""
        return self._k


#__i...__ ???
#__r...__ ???

    ## Renvoie (self//other, self%other)
    def __divmod__(self, other):
        """Renvoie (self//other, self%other) (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: NotImplemented"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return NotImplemented


    ## Renvoie self//other
    def __floordiv__(self, other):
        """Renvoie self//other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: NotImplemented"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return NotImplemented


    ## Renvoie self%other
    def __mod__(self, other):
        """Renvoie self%other (modulo self.k())

        Pre: other: Intmod ou Integral

        Result: NotImplemented"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return NotImplemented


    ## Renvoie NotImplemented
    def __truediv__(self, other):
        """Renvoie NotImplemented

        Pre: other: Intmod ou Integral

        Result: NotImplemented"""
        assert isinstance(other, Intmod) or isinstance(other, numbers.Integral), other

        return NotImplemented



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        import DSPython.debug as debug

        debug.test_begin(VERSION, __debug__)

        print('Intmod()...', end='') ; sys.stdout.flush()
        for k in range(1, 11):
            for n in range(-k*2, k*3):
                a = Intmod(n, k)
                assert a._k == k, (k, n, a, a_k)
                assert int(a) == n%k, (k, n, a, int(a))
        print('ok') ; sys.stdout.flush()


        print('Intmod.__add__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a + b == int(a + b)%k, (k, a, b, a + b, int(a + b)%k)
                    assert a + b == b + a, (k, a, b)
                for b in range(-k, k*2):
                    assert a + b == int(a + b)%k, (k, a, b, a + b, int(a + b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__and__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a & b == int(a & b)%k, (k, a, b, a & b, int(a & b)%k)
                    assert a & b == b & a, (k, a, b)
                for b in range(-k, k*2):
                    assert a & b == int(a & b)%k, (k, a, b, a & b, int(a & b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__divmod__()...', end='') ; sys.stdout.flush()
        for k in range(1, 5):
            for a in range(-k*2, k*2):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)
                    try:
                        assert divmod(a, b), (k, a, b)
                    except TypeError:
                        pass
        print('ok') ; sys.stdout.flush()


        print('Intmod.__floordiv__()...', end='') ; sys.stdout.flush()
        for k in range(1, 5):
            for a in range(-k*2, k*2):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)
                    try:
                        assert a//b, (k, a, b)
                    except TypeError:
                        pass
        print('ok') ; sys.stdout.flush()


        print('Intmod.__int__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11):
            for n in range(-k*2, k*3):
                a = Intmod(n, k)
                assert a == n%k, (k, n, a)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__invert__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                assert ~a == int(~a)%k, (k, a, ~a, int(~a)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__lshift__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a<<b == int(a<<b)%k, (k, a, b, a<<b, int(a<<b)%k)
                for b in range(k*2):
                    assert a<<b == int(a<<b)%k, (k, a, b, a<<b, int(a<<b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__mod__()...', end='') ; sys.stdout.flush()
        for k in range(1, 5):
            for a in range(-k*2, k*2):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)
                    try:
                        assert a%b, (k, a, b)
                    except TypeError:
                        pass
        print('ok') ; sys.stdout.flush()


        print('Intmod.__mul__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a*b == int(a*b)%k, (k, a, b, a*b, int(a*b)%k)
                    assert a*b == b*a, (k, a, b)
                for b in range(-k, k*2):
                    assert a*b == int(a*b)%k, (k, a, b, a*b, int(a*b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__neg__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                assert -a == int(-a)%k, (k, a, -a, int(-a)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__or__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a | b == int(a | b)%k, (k, a, b, a | b, int(a | b)%k)
                    assert a | b == b | a, (k, a, b)
                for b in range(-k, k*2):
                    assert a | b == int(a | b)%k, (k, a, b, a | b, int(a | b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__pow__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a**b == int(a**b)%k, (k, a, b, a**b, int(a**b)%k)
                for b in range(-k, k*2):
                    assert a**b == int(a**b)%k, (k, a, b, a**b, int(a**b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__rshift__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a>>b == int(a>>b)%k, (k, a, b, a>>b, int(a>>b)%k)
                for b in range(k*2):
                    assert a>>b == int(a>>b)%k, (k, a, b, a>>b, int(a>>b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__str__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11):
            for n in range(-k*2, k*3):
                a = Intmod(n, k)
                assert str(a) == str(n%k), (k, n, str(a))
        print('ok') ; sys.stdout.flush()


        print('Intmod.__sub__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a - b == int(a - b)%k, (k, a, b, a - b, int(a - b)%k)
                    assert a - b == -(b - a), (k, a, b)
                for b in range(-k, k*2):
                    assert a - b == int(a - b)%k, (k, a, b, a - b, int(a - b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.__truediv__()...', end='') ; sys.stdout.flush()
        for k in range(1, 5):
            for a in range(-k*2, k*2):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)
                    try:
                        assert a/b, (k, a, b)
                    except TypeError:
                        pass
        print('ok') ; sys.stdout.flush()


        print('Intmod.__xor__()...', end='') ; sys.stdout.flush()
        for k in range(1, 11 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                for b in range(-k, k*2):
                    b = Intmod(b, k)

                    assert a ^ b == int(a ^ b)%k, (k, a, b, a ^ b, int(a ^ b)%k)
                    assert a ^ b == b ^ a, (k, a, b)
                for b in range(-k, k*2):
                    assert a ^ b == int(a ^ b)%k, (k, a, b, a ^ b, int(a ^ b)%k)
        print('ok') ; sys.stdout.flush()


        print('Intmod.k()...', end='') ; sys.stdout.flush()
        for k in range(1, 11):
            for a in range(-k*2, k*3):
                a = Intmod(a, k)
                assert a.k() == k, (k, a, a.k())
        print('ok') ; sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
