#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.integer Entiers : fonctions arithmtiques

##\file
# Entiers : fonctions arithmtiques

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 9 juillet 2006
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'integer --- 2010 March 16'

import numbers, types

import DSPython.natural as natural



# ###########
# Fonctions #
#############
## n en binaire dans un string
def bin(n, sign=False):
    """Renvoie un string reprsentant n en binaire
      (Si n == 0 alors renvoie '0' ou '+0' si sign=True)
      Si n < 0
        alors renvoie la reprsentation de -n prcde du caractre '-'
      Si sign == True
        alors ajoute toujours le signe mme si c'est '+'

    Pre: n: Integral

    Result: string

    O(n) = lg(n)"""
    assert isinstance(n, numbers.Integral), n

    if n < 0:
        sign = '-'
        n = -n
    elif sign:
        sign = '+'
    else:
        sign = ''
    return sign + natural.bin(n)


## Coefficient binomial de n et k
def binom(n, k):
    """Renvoie le coefficient binomial (n)
                                    (k)

    Pre: n: Integral
         k: Integral

    Result: naturel

    O(n, k) = ..."""
    assert isinstance(n, numbers.Integral), n
    assert isinstance(k, numbers.Integral), k

    if n < 0:
        raise NotImplementedError
    return (natural.binom(n, k) if k >= 0
            else 0)


## F<sub>k</sub>
def fibonacci(k):
    """Renvoie F_k, le kime nombre de Fibonacci

    Pre: k: Integral

    Result: Integral

    O(k) = ..."""
    assert isinstance(k, numbers.Integral), k

    return fibonacci2(k)[1]


## (F<sub>k-1</sub>, F<sub>k</sub>)
def fibonacci2(k):
    """Renvoie (F_(k-1), F_k), les (k-1)ime et kime nombres de Fibonacci

    Pre: k: Integral

    Result: couple d'Integral

    O(k) = ..."""
    assert isinstance(k, numbers.Integral), k

    if k >= 0:
        return natural.fibonacci2(k)
    else:
        f2 = natural.fibonacci2(-k + 1)
        return ((f2[1], -f2[0]) if k&1 == 0
                else (-f2[1], f2[0]))


## n est un nombre de Fibonacci ?
def fibonacci_is(n):
    """Renvoie True si n est un nombre de Fibonacci,
      False sinon

    Pre: n: Integral

    Result: booleen

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    if n >= 0:
        return natural.fibonacci_is(n)
    elif n != -1:
        k = natural.fibonacci_to_index(-n)
        return k != None and k&1 == 0
    else:
        return True


## Indice du nombre de Fibonacci correspondant  n
def fibonacci_to_index(n):
    """Si n == F_k alors renvoie k
                      (si n == 1 alors renvoie toujours 1, jamais 2
                       et si n >= 0 alors renvoie toujours le k positif,
                                          jamais le ngatif),
                sinon renvoie None

    Pre: n Integral

    Result: Integral

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    if n >= 0:
        return natural.fibonacci_to_index(n)
    elif n != -1:
        k = natural.fibonacci_to_index(-n)
        return (-k if (k != None) and (k&1 == 0)
                else None)
    else:
        return -2


## f(k)
def fibonacci_gen(f_0, f_1, k):
    """Renvoie f(k),
    la valeurs en k de la fonction f de Fibonacci
    (c.--d. telle que f(n+2) = f(n) + f(n+1))
    de conditions initiales f(0) == f_0 et f(1) == f_1

    Pre: f_0: Integral
         f_1: Integral
         k: Integral

    Result: Integral

    O(k) = ..."""
    assert isinstance(f_0, numbers.Integral), f_0
    assert isinstance(f_1, numbers.Integral), f_1
    assert isinstance(k, numbers.Integral), k

    return fibonacci2_gen(f_0, f_1, k)[1]


## (f(k-1), f(k))
def fibonacci2_gen(f_0, f_1, k):
    """Renvoie (f(k-1), f(k)),
    les valeurs en k-1 et k de la fonction f de Fibonacci
    (c.--d. telle que f(n+2) = f(n) + f(n+1))
    de conditions initiales f(0) == f_0 et f(1) == f_1

    Pre: f_0: Integral
         f_1: Integral
         k: Integral

    Result: couple d'Integral

    O(k) = ..."""
    assert isinstance(f_0, numbers.Integral), f_0
    assert isinstance(f_1, numbers.Integral), f_1
    assert isinstance(k, numbers.Integral), k

    Fk_2, Fk_1 = fibonacci2(k - 1)
    return (Fk_2*f_0 + Fk_1*f_1,
            Fk_1*f_0 + (Fk_1 + Fk_2)*f_1)


## L<sub>k</sub>
def lucas(k):
    """Renvoie L_k, le kime nombre de Lucas

    Pre: k: Integral

    Result: Integral

    O(k) = ..."""
    assert isinstance(k, numbers.Integral), k

    Fk_1, Fk = fibonacci2(k)
    return (Fk_1 << 1) + Fk


## (L<sub>k-1</sub>, L<sub>k</sub>)
def lucas2(k):
    """Renvoie (L_(k-1), L_k), les (k-1)ime et kime nombres de Lucas

    Pre: k: Integral

    Result: couple d'Integral

    O(k) = ..."""
    assert isinstance(k, numbers.Integral), k

    Fk_2, Fk_1= fibonacci2(k - 1)
    Fk = Fk_1 + Fk_2
    return (Fk + Fk_2, (Fk_1 << 1) + Fk)


## n est un nombre de Lucas ?
def lucas_is(n):
    """Renvoie True si n est un nombre de Lucas,
      False sinon

    Pre: n: Integral

    Result: booleen

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    if n >= 0:
        return natural.lucas_is(n)
    else:
        k = natural.lucas_to_index(-n)
        return k != None and k&1 != 0


## Indice du nombre de Lucas correspondant  n
def lucas_to_index(n):
    """Si n == F_k alors renvoie k
                      (si n == 1 alors renvoie toujours 1, jamais 2
                       et si n >= 0 alors renvoie toujours le k positif,
                                          jamais le ngatif),
                sinon renvoie None

    Pre: n Integral

    Result: Integral

    O(n) = ..."""
    assert isinstance(n, numbers.Integral), n

    if n >= 0:
        return natural.lucas_to_index(n)
    else:
        k = natural.lucas_to_index(-n)
        return (-k if (k != None) and (k&1 != 0)
                else None)


## Produit des valeurs de f value sur les lments de s
def prod(f, s):
    """Produit des valeurs de f value sur les lments de s

    Pre: f: fonction  un paramtre Integral, renvoyant un Integral
         s: collection d'Integral sur lesquels f est dfinie

    Result: Integral

    O(f, s) = ..."""
    assert isinstance(f, types.FunctionType), type(f)

    r = 1
    for i in s:
        r *= f(i)
        if r == 0:
            return 0
    return r


## Somme des valeurs de f value sur les lments de s
def sum(f, s):
    """Somme des valeurs de f value sur les lments de s

    Pre: f: fonction  un paramtre Integral, renvoyant un Integral
         s: collection d'Integral sur lesquels f est dfinie

    Result: Integral

    O(f, s) = ..."""
    assert isinstance(f, types.FunctionType), type(f)

    r = 0
    for i in s:
        r += f(i)
    return r



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        import math

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        import DSPython.debug as debug
        import DSPython.natseq as natseq

        debug.test_begin(VERSION, __debug__)

        print('bin()...', end=''); sys.stdout.flush()
        assert bin(-1) == '-1', bin(1)
        assert bin(0)  == '0', bin(0)
        assert bin(1)  == '1', bin(1)

        assert bin(-1, False) == '-1', bin(-1, False)
        assert bin(0,  False) == '0',  bin(0,  False)
        assert bin(1,  False) == '1',  bin(1,  False)

        assert bin(-1, True) == '-1', bin(-1, True)
        assert bin(0,  True) == '+0', bin(0,  True)
        assert bin(1,  True) == '+1', bin(1,  True)
        for i in range(1, 50):
            assert bin(2**i - 1) == '1'*i, (i, bin(2**i - 1))                    #  1...11
            assert bin(2**i) == '1' + '0'*i, (i, bin(2**i))                      # 10...00
            assert bin(2**i + 1) == '1' + '0'*(i - 1) + '1', (i, bin(2**i + 1))  # 10...01
        for n in range(-16384, 16384):
            assert int(bin(n), 2) == n, (n, bin(n))
        for n in range(1, 2**40, 1000000000):
            assert int(bin(n), 2) == n, (n, bin(n))
        for n in range(2**32 - 16384, 2**32 + 16384):
            assert int(bin(n), 2) == n, (n, bin(n))
        print('ok'); sys.stdout.flush()


        print('binom()...', end=''); sys.stdout.flush()
        for n in range(100):
            assert binom(n, 0) == 1, (n, binom(n, 0))
            assert binom(n, 1) == n, (n, binom(n, 1))
            for k in range(1, 10):
                assert binom(n, n+k) == 0, (n, k, binom(n, n+k))
            s = 0
            for k in range(n + 1):
                s += binom(n, k)
            assert s == 2**n, (n, s, 2**n)
        for n in range(1, 50):
            for k in range(1, n + 1):
                assert binom(n, k) == binom(n, n - k), (n, k, binom(n, k), binom(n, n - k))
                assert binom(n, k) == binom(n - 1, k - 1) * n // k, \
                       (n, k, binom(n, k), binom(n - 1, k - 1))
                assert binom(n, k) == binom(n, k - 1) * (n - k + 1) // k, \
                       (n, k, binom(n, k), binom(n, k - 1) * (n - k + 1))
        for n in range(3, 100):
            assert binom(n, 2) == ((n - 1)*n)//2, (n, binom(n, 2), ((n - 1)*n)//2)
        print('ok'); sys.stdout.flush()


        print('fibonacci()...', end=''); sys.stdout.flush()
        Fk_1 = 1
        Fk  = 0
        for k in range(5000):
            assert fibonacci(k) == Fk, (k, fibonacci(k))
            Fk_1, Fk = Fk, Fk + Fk_1
        Fk_1 = 1
        Fk  = 0
        for k in range(0, -1000, -1):
            assert fibonacci(k) == Fk, (k, fibonacci(k))
            Fk_1, Fk = Fk - Fk_1, Fk_1
        print('ok'); sys.stdout.flush()


        print('fibonacci2()...', end=''); sys.stdout.flush()
        Fk_1 = 1
        Fk  = 0
        for k in range(5000):
            assert fibonacci2(k) == (Fk_1, Fk), (k, fibonacci2(k))
            Fk_1, Fk = Fk, Fk + Fk_1
        Fk_1 = 1
        Fk  = 0
        for k in range(0, -1000, -1):
            assert fibonacci2(k) == (Fk_1, Fk), (k, fibonacci2(k))
            Fk_1, Fk = Fk - Fk_1, Fk_1
        print('ok'); sys.stdout.flush()


        print('fibonacci_is()...', end=''); sys.stdout.flush()
        assert not fibonacci_is(-5)
        assert not fibonacci_is(-4)
        assert fibonacci_is(-3)
        assert not fibonacci_is(-2)
        assert fibonacci_is(-1)
        assert fibonacci_is(0)
        assert fibonacci_is(1)
        assert fibonacci_is(2)
        assert fibonacci_is(3)
        assert not fibonacci_is(4)
        for k in range(-1000, 1000):
            assert fibonacci_is(fibonacci(k)), (k, fibonacci(k))
        for n in range(-16384, 16384):
            if fibonacci_is(n):
                assert -46 <= fibonacci_to_index(n) <= 47, (n, fibonacci_to_index(n))
            else:
                assert fibonacci_to_index(n) == None, (n, fibonacci_to_index(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(-(2**45), 2**45, 1000000000):
                if fibonacci_is(n):
                    assert -46 <= fibonacci_to_index(n) <= 47, (n, fibonacci_to_index(n))
                else:
                    assert fibonacci_to_index(n) == None, (n, fibonacci_to_index(n))
            for n in range(-2**32 - 32768, -2**32 + 32768):
                if fibonacci_is(n):
                    assert -46 <= fibonacci_to_index(n) <= 47, (n, fibonacci_to_index(n))
                else:
                    assert fibonacci_to_index(n) == None, (n, fibonacci_to_index(n))
            for n in range(2**32 - 32768, 2**32 + 32768):
                if fibonacci_is(n):
                    assert 0 <= fibonacci_to_index(n) <= 47, (n, fibonacci_to_index(n))
                else:
                    assert fibonacci_to_index(n) == None, (n, fibonacci_to_index(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('fibonacci_to_index()...', end=''); sys.stdout.flush()
        assert fibonacci_to_index(-5) == None, fibonacci_to_index(-5)
        assert fibonacci_to_index(-4) == None, fibonacci_to_index(-4)
        assert fibonacci_to_index(-3) == -4,   fibonacci_to_index(-3)
        assert fibonacci_to_index(-2) == None, fibonacci_to_index(-2)
        assert fibonacci_to_index(-1) == -2,   fibonacci_to_index(-1)
        assert fibonacci_to_index(0) == 0,    fibonacci_to_index(0)
        assert fibonacci_to_index(1) == 1,    fibonacci_to_index(1)
        assert fibonacci_to_index(2) == 3,    fibonacci_to_index(2)
        assert fibonacci_to_index(3) == 4,    fibonacci_to_index(3)
        assert fibonacci_to_index(4) == None, fibonacci_to_index(4)
        for k in range(-1000, 0):
            assert fibonacci_to_index(fibonacci(k)) == (-1)**(k&1) * k, \
                   (k, fibonacci_to_index(fibonacci(k)))
        for k in range(3, 1000):
            assert fibonacci_to_index(fibonacci(k)) == k, \
                   (k, fibonacci_to_index(fibonacci(k)))
        for n in range(-16384, 16384):
            k = fibonacci_to_index(n)
            if k != None:
                assert fibonacci(k) == n, (n, k, fibonacci(k))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(-(2**45), 2**45, 1000000000):
                k = fibonacci_to_index(n)
                if k != None:
                    assert fibonacci(k) == n, (n, k, fibonacci(k))
            for n in range(-2**32 - 32768, -2**32 + 32768):
                k = fibonacci_to_index(n)
                if k != None:
                    assert fibonacci(k) == n, (n, k, fibonacci(k))
            for n in range(2**32 - 32768, 2**32 + 32768):
                k = fibonacci_to_index(n)
                if k != None:
                    assert fibonacci(k) == n, (n, k, fibonacci(k))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('fibonacci_gen()...', end=''); sys.stdout.flush()
        for k in range(-100, 1000):
            Fk_1, Fk = fibonacci2(k)
            assert fibonacci_gen(0, 1, k) == Fk, (k, fibonacci_gen(0, 1, k), Fk)
            assert fibonacci_gen(2, 1, k) == Fk_1*2 + Fk, (k, fibonacci_gen(2, 1, k), Fk_1*2 + Fk)
            for a in range(-10, 10):
                assert fibonacci_gen(0, a, k) == Fk*a, (k, fibonacci_gen(0, a, k), Fk*a)
                assert fibonacci_gen(a, 0, k) == Fk_1*a, (k, fibonacci_gen(a, 0, k), Fk_1*a)
                assert fibonacci_gen(a, a, k) == (Fk + Fk_1)*a, \
                       (k, fibonacci_gen(a, a, k), (Fk + Fk_1)*a)
                assert fibonacci_gen(-a, a, k) == (Fk - Fk_1)*a, \
                       (k, fibonacci_gen(-a, a, k), (Fk - Fk_1)*a)
        print('ok'); sys.stdout.flush()


        print('fibonacci2_gen()...', end=''); sys.stdout.flush()
        for k in range(-500, 3000):
            assert fibonacci2_gen(0, 1, k) == fibonacci2(k), \
                   (k, fibonacci2_gen(0, 1, k), fibonacci2(k))
            assert fibonacci2_gen(2, 1, k) == lucas2(k), \
                   (k, fibonacci2_gen(2, 1, k), lucas2(k))
        print('ok'); sys.stdout.flush()


        print('lucas()...', end=''); sys.stdout.flush()
        Lk_1 = -1
        Lk  = 2
        for k in range(5000):
            assert lucas(k) == Lk, (k, lucas(k))
            Lk_1, Lk = Lk, Lk + Lk_1
        Lk_1 = -1
        Lk  = 2
        for k in range(0, -1000, -1):
            assert lucas(k) == Lk, (k, lucas(k))
            Lk_1, Lk = Lk - Lk_1, Lk_1
        print('ok'); sys.stdout.flush()

        print('lucas2()...', end=''); sys.stdout.flush()
        assert lucas2(0) == (-1, 2), lucas2(0)
        assert lucas2(1) == (2, 1),  lucas2(1)
        assert lucas2(2) == (1, 3),  lucas2(2)
        assert lucas2(3) == (3, 4),  lucas2(3)
        Lk_1 = -1
        Lk  = 2
        for k in range(5000):
            assert lucas2(k) == (Lk_1, Lk), (k, lucas2(k))
            Lk_1, Lk = Lk, Lk + Lk_1
        Lk_1 = -1
        Lk  = 2
        for k in range(0, -1000, -1):
            assert lucas(k) == Lk, (k, lucas(k))
            Lk_1, Lk = Lk - Lk_1, Lk_1
        print('ok'); sys.stdout.flush()


        print('lucas_is()...', end=''); sys.stdout.flush()
        assert not lucas_is(-5)
        assert lucas_is(-4)
        assert not lucas_is(-3)
        assert not lucas_is(-2)
        assert lucas_is(-1)
        assert not lucas_is(0)
        assert lucas_is(1)
        assert lucas_is(2)
        assert lucas_is(3)
        assert lucas_is(4)
        assert not lucas_is(5)
        for k in range(-1000, 1000):
            assert lucas_is(lucas(k)), (k, lucas(k))
        for n in range(-16384, 16384):
            if lucas_is(n):
                assert -45 <= lucas_to_index(n) <= 46, (n, lucas_to_index(n))
            else:
                assert lucas_to_index(n) == None, (n, lucas_to_index(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(-(2**45), 2**45, 10000000000):
                if lucas_is(n):
                    assert -45 <= lucas_to_index(n) <= 46, (n, lucas_to_index(n))
                else:
                    assert lucas_to_index(n) == None, (n, lucas_to_index(n))
            for n in range(-2**32 - 32768, -2**32 + 32768):
                if lucas_is(n):
                    assert -45 <= lucas_to_index(n) <= 46, (n, lucas_to_index(n))
                else:
                    assert lucas_to_index(n) == None, (n, lucas_to_index(n))
            for n in range(2**32 - 32768, 2**32 + 32768):
                if lucas_is(n):
                    assert 0 <= lucas_to_index(n) <= 46, (n, lucas_to_index(n))
                else:
                    assert lucas_to_index(n) == None, (n, lucas_to_index(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('lucas_to_index()...', end=''); sys.stdout.flush()
        assert lucas_to_index(-5) == None, lucas_to_index(-5)
        assert lucas_to_index(-4) == -3,   lucas_to_index(-4)
        assert lucas_to_index(-3) == None, lucas_to_index(-3)
        assert lucas_to_index(-2) == None, lucas_to_index(-2)
        assert lucas_to_index(-1) == -1,   lucas_to_index(-1)
        assert not lucas_to_index(0) == 0, lucas_to_index(0)
        assert lucas_to_index(1) == 1,     lucas_to_index(1)
        assert lucas_to_index(2) == 0,     lucas_to_index(2)
        assert lucas_to_index(3) == 2,     lucas_to_index(3)
        assert lucas_to_index(4) == 3,     lucas_to_index(4)
        assert lucas_to_index(5) == None,  lucas_to_index(5)
        for k in range(-1000, 0):
            assert lucas_to_index(lucas(k)) == -(-1)**(k&1) * k, (k, lucas_to_index(lucas(k)))
        for k in range(3, 1000):
            assert lucas_to_index(lucas(k)) == k, (k, lucas_to_index(lucas(k)))
        for n in range(-16384, 16384):
            k = lucas_to_index(n)
            if k != None:
                assert lucas(k) == n, (n, k, lucas(k))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(-(2**45), 2**45, 1000000000):
                k = lucas_to_index(n)
                if k != None:
                    assert lucas(k) == n, (n, k, lucas(k))
            for n in range(-2**32 - 32768, -2**32 + 32768):
                k = lucas_to_index(n)
                if k != None:
                    assert lucas(k) == n, (n, k, lucas(k))
            for n in range(2**32 - 32768, 2**32 + 32768):
                k = lucas_to_index(n)
                if k != None:
                    assert lucas(k) == n, (n, k, lucas(k))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('prod()...', end=''); sys.stdout.flush()
        for n in range(100):
            f = lambda n: n
            assert prod(f, range(1, n + 1)) == math.factorial(n), \
                   (n, prod(f, range(1, n + 1)), math.factorial(n))
            assert prod(f, range(-n, n + 1)) == 0, (n, prod(f, range(-n, n + 1)))

            assert prod(f, range(1, 2*n, 2)) == natseq.prod(range(1, 2*n, 2)), \
                   (n, prod(f, range(1, 2*n, 2)), natseq.prod(range(1, 2*n, 2)))

            f = lambda n: n * n
            assert prod(f, range(1, n + 1)) == math.factorial(n)**2, \
                   (n, prod(f, range(1, n + 1)), math.factorial(n)**2)
            assert prod(f, range(-n, n + 1)) == 0, (n, prod(f, range(-n, n + 1)))

            assert prod(f, range(1, 2*n, 2)) == natseq.prod(range(1, 2*n, 2))**2, \
                   (n, prod(f, range(1, 2*n, 2)), natseq.prod(range(1, 2*n, 2))**2)
        print('ok'); sys.stdout.flush()


        print('sum()...', end=''); sys.stdout.flush()
        for n in range(100):
            f = lambda n: n
            assert sum(f, range(n + 1)) == n*(n + 1)//2, (n, sum(f, range(n + 1)), n*(n + 1)//2)
            assert sum(f, range(-n, n + 1)) == 0, (n, sum(f, range(-n, n + 1)))

            assert sum(f, range(1, 2*n, 2)) == n*n, (n, sum(f, range(1, 2*n, 2)), n*n)

            f = lambda n: n * n
            assert sum(f, range(n + 1)) == n*(n + 1)*(2*n + 1)//6,  \
                   (n, sum(f, range(n + 1)), n*(n + 1)*(2*n + 1)//6)
            assert sum(f, range(-n, n + 1)) == n*(n + 1)*(2*n + 1)//3,  \
                   (n, sum(f, range(-n, n + 1)), n*(n + 1)*(2*n + 1)//3)

            assert sum(f, range(1, 2*n, 2)) == n*(4*n*n - 1)//3, \
                   (n, sum(f, range(1, 2*n, 2)), n*(4*n*n - 1)//3)
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
