#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.finitec Calcul finitsimal : oprateurs aux diffrences finies
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Nombres1" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Nombres1</tt></a>\endhtmlonly

##\file
# Calcul finitsimal : oprateurs aux diffrences finies
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Nombres1" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Nombres1</tt></a>\endhtmlonly

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 4 fvrier 2006
####################################
from __future__ import division
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'finitec --- 2012 June 25'

import numbers, types



# ##############################
# Fonctions pour les fonctions #
################################
## Fonction c : c(x)
def c(x):
    """Fonction c : c(x),
    renvoie (-1)**d * 2**(2d)             si 0 <= m < 2,
            0                             si 2 <= m < 3,
            (-1)**(d + 1) * 2**(2d + 1)   si 3 <= m < 4
       avec d, m = divmod(x, 4)

    Pour les n naturels, correspond  [OEIS A146559]

    Pre: x: Number

    Result: Number

    O(f) = ..."""
    assert isinstance(x, numbers.Number), x

    d, m = divmod(x, 4)

    assert 0 <= m < 4

    sign = (1 if d%2 == 0
            else -1)

    return ((4**d)*sign if m < 2
            else (0 if m < 3
                  else (4**d)*(-2)*sign))


## Fonction identit : id(x)
def id(x):
    """Fonction identit : id(x),
    renvoie x

    Pre: x: Number

    Result: Number

    O(f) = 1"""
    assert isinstance(x, numbers.Number), x

    return x


## Fonction s : s(x)
def s(x):
    """Fonction s : s(x),
    renvoie 0                       si 0 <= m < 1,
            (-1)**d * 2**(2d)       si 1 <= m < 2,
            (-1)**d * 2**(2d + 1)   si 2 <= m < 4
       avec d, m = divmod(x, 4)

    Pour les n naturels, correspond  [OEIS A009545]

    Pre: x: Number

    Result: Number

    O(f) = ..."""
    assert isinstance(x, numbers.Number), x

    d, m = divmod(x, 4)

    assert 0 <= m < 4

    sign = (1 if d%2 == 0
            else -1)

    return (0 if m < 1
            else ((4**d)*sign if m < 2
                  else (4**d)*2*sign))


## Fonction signe : sign(x)
def sign(x):
    """Fonction signe : signe(x),
    renvoie -1   si x <  0,
             0   si x == 0,
             1   si x >  0

    Pre: x: Number

    Result: -1, 0 ou 1

    O(f) = 1"""
    assert isinstance(x, numbers.Number), x

    return (1 if x > 0
            else (-1 if x < 0
                   else 0))



# ###############################
# Fonctions pour les oprateurs #
#################################
## Oprateur de diffrence (progressive) : Delta<sub>alpha</sub> f
def diff(f, alpha=1):
    """Oprateur de diffrence (progressive) : Delta_alpha f,
    renvoie la fonction x -> f(x + alpha) - f(x)

    Pre: f: fonction : Number -> (Number ou None)
         alpha: Number

    Result: fonction : Number -> (Number ou None)

    O(f, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(alpha, numbers.Number), alpha

    return lambda x : diff_x(f, x, alpha=alpha)


## Delta<sub>alpha</sub> f(x) == f(x + alpha) - f(x)
def diff_x(f, x, alpha=1):
    """Diffrence (progressive) value en x :
      Delta_alpha f(x) == f(x + alpha) - f(x)

    Pre: f: fonction : Number -> (Number ou None)
         x: Number
         alpha: Number

    Result: Number ou None

    O(f, x, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(x, numbers.Number), x
    assert isinstance(alpha, numbers.Number), alpha

    fx = f(x)
    if fx != None:
        fxalpha = f(x + alpha)
        if fxalpha != None:
            return fxalpha - fx
    return None


## Oprateur de sommation (progressive) : sum<sub>alpha</sub> f
def sum(f, alpha=1):
    """Oprateur de sommation (progressive) : sum_alpha f,
    renvoie la fonction x -> sum_alpha f(x)

    Pre: f: fonction : Number -> (Number ou None)
         alpha: Number != 0

    Result: fonction : Number -> (Number ou None)

    O(f, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(alpha, numbers.Number), alpha
    assert alpha != 0, alpha

    return lambda x : sum_x(f, x, alpha=alpha)


## sum<sub>alpha</sub> f(x)
def sum_x(f, x, alpha=1):
    """Sommation (progressive) value en x : sum_alpha f(x) ==
      f(x%alpha) + f(x%alpha + alpha) + f(x%alpha + 2*alpha)
                 + ... + f(x - alpha) si alpha > 0 et x >= 0,
      -f(x) - ... - f(x%alpha - 3*alpha) - f(x%alpha - 2*alpha)
            - f(x%alpha - alpha) si alpha > 0 et x <= 0,
      -f(x%alpha - alpha) - f(x%alpha - 2*alpha) - f(x%alpha - 3*alpha)
                 - ... - f(x) si alpha < 0 et x >= 0,
      f(x - alpha) + ... + f(x%alpha + 2*alpha) + f(x%alpha + alpha)
                   + f(x%alpha) si alpha < 0 et x <= 0

    Pre: f: fonction : Number -> (Number ou None)
         x: Number
         alpha: Integral != 0

    Result: Number ou None

    O(f, x, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(x, numbers.Number), type(x)
    assert isinstance(alpha, numbers.Integral), type(alpha)
    assert alpha != 0, alpha

    if alpha > 0:
        if x > 0:
            s = 0
            x -= alpha
            while x >= 0:
                fx = f(x)
                if fx == None:
                    return None
                s += fx
                x -= alpha
            return s
        elif x < 0:
            s = 0
            while x < 0:
                fx = f(x)
                if fx == None:
                    return None
                s -= fx
                x += alpha
            return s
        else:
            return 0
    else:  # alpha < 0
        if x > 0:
            s = 0
            while x > 0:
                fx = f(x)
                if fx == None:
                    return None
                s -= fx
                x += alpha
            return s
        elif x < 0:
            s = 0
            x -= alpha
            while x <= 0:
                fx = f(x)
                if fx == None:
                    return None
                s += fx
                x -= alpha
            return s
        else:
            return 0


## Oprateur de rflexion (symtrie horizontale) : check f
def sym(f):
    """Oprateur de rflexion (symtrie horizontale) : check f,
    renvoie la fonction x -> f(-x)

    Pre: f: fonction : Number -> (Number ou None)

    Result: fonction : Number -> (Number ou None)

    O(f) = ..."""
    assert isinstance(f, types.FunctionType), type(f)

    return lambda x : trans_x(f, x)


## check f(x) == f(-x)
def sym_x(f, x):
    """Rflexion (symtrie horizontale) value en x : check f(x) == f(-x)

    Pre: f: fonction : Number -> (Number ou None)
         x: Number

    Result: Number ou None

    O(f, x) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(x, numbers.Number), x

    return f(-x)


## Oprateur de translation (horizontale) : E<sub>alpha</sub> f
def trans(f, alpha=1):
    """Oprateur de translation (horizontale) : E_alpha f,
    renvoie la fonction x -> f(x + alpha)

    Pre: f: fonction : Number -> (Number ou None)
         alpha: Number

    Result: fonction : Number -> (Number ou None)

    O(f, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(alpha, numbers.Number), alpha

    return lambda x : trans_x(f, x, alpha=alpha)


## E<sub>alpha</sub> f(x) == f(x + alpha)
def trans_x(f, x, alpha=1):
    """Translation (horizontale) value en x : E_alpha f(x) == f(x + alpha)

    Pre: f: fonction : Number -> (Number ou None)
         x: Number
         alpha: Number

    Result: Number ou None

    O(f, x, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(x, numbers.Number), x
    assert isinstance(alpha, numbers.Number), alpha

    return f(x + alpha)


## Oprateur de translation verticale : V<sub>alpha</sub> f
def trans_vert(f, alpha=1):
    """Oprateur de translation verticale : V_alpha f,
    renvoie la fonction x -> f(x) + alpha

    Pre: f: fonction : Number -> (Number ou None)
         alpha: Number

    Result: fonction : Number -> (Number ou None)

    O(f, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(alpha, numbers.Number), alpha

    return lambda x : trans_vert_x(f, x, alpha=alpha)


## V<sub>alpha</sub> f(x) == f(x) + alpha
def trans_vert_x(f, x, alpha=1):
    """Translation verticale value en x : V_alpha f(x) == f(x) + alpha

    Pre: f: fonction : Number -> (Number ou None)
         x: Number
         alpha: Number

    Result: Number ou None

    O(f, x, alpha) = ..."""
    assert isinstance(f, types.FunctionType), type(f)
    assert isinstance(x, numbers.Number), x
    assert isinstance(alpha, numbers.Number), alpha

    fx = f(x)
    return (fx + alpha if fx != None
            else None)



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        import DSPython.debug as debug
        import DSPython.numbernone as numbernone

        debug.test_begin(VERSION, __debug__)


        def _eq(x, y, prec=2**32):
            """Renvoie True si x == y (avec une prcision relative de 1/prec)
            False sinon"""
            return ((((x >= 0) and (y >= 0)) or ((x <= 0) and (y <= 0)))
                    and (abs(x - y) <= 1/prec) if (x != None) and (y != None)
                    else x == y)



        print('c()...', end=''); sys.stdout.flush()
        assert c(-3) == -0.25, c(-3)
        assert c(-2) == 0, c(-2)
        assert c(-1) == 0.5, c(-1)
        assert c(0) == 1, c(0)
        assert c(1) == 1, c(1)
        assert c(2) == 0, c(2)
        assert c(3) == -2, c(3)
        assert c(4) == -4, c(4)
        assert c(5) == -4, c(5)

        for n in range(-100, 100):
            assert(c(n + 4) == -4*c(n)), n
            assert(c(n + 3) == 4*c(n + 2) - 6*c(n + 1) + 4*c(n)), n
            assert(c(n + 2) == 2*c(n + 1) - 2*c(n)), n

            assert(c(n/10 + 4) == -4*c(n/10)), n/10
            assert(c(n/10 + 3) == 4*c(n/10 + 2) - 6*c(n/10 + 1) + 4*c(n/10)), n/10
            assert(c(n/10 + 2) == 2*c(n/10 + 1) - 2*c(n/10)), n/10

            assert(c(n) == ((1 + 1j)**n).real), n
        for n in range(-90, 100):
            assert(c(n) == ((1 + 1j)**(n - 2) - (1 - 1j)**(n - 2))*1j), \
                (n, c(n), ((1 + 1j)**(n - 2) - (1 - 1j)**(n - 2))*1j)
        print('ok'); sys.stdout.flush()


        print('id()...', end=''); sys.stdout.flush()
        for x in range(-100, 100):
            assert id(x) == x, (x, id(x))
            assert id(x/10) == x/10, (x, id(x/10))
        print('ok'); sys.stdout.flush()


        print('s()...', end=''); sys.stdout.flush()
        assert s(-3) == -0.25, s(-3)
        assert s(-2) == -0.5, s(-2)
        assert s(-1) == -0.5, s(-1)
        assert s(0) == 0, s(0)
        assert s(1) == 1, s(1)
        assert s(2) == 2, s(2)
        assert s(3) == 2, s(3)
        assert s(4) == 0, s(4)
        assert s(5) == -4, s(5)

        for n in range(-100, 100):
            assert(s(n + 4) == -4*s(n)), n
            assert(s(n + 3) == 4*s(n + 2) - 6*s(n + 1) + 4*s(n)), n
            assert(s(n + 2) == 2*s(n + 1) - 2*s(n)), n

            assert(s(n/10 + 4) == -4*s(n/10)), n/10
            assert(s(n/10 + 3) == 4*s(n/10 + 2) - 6*s(n/10 + 1) + 4*s(n/10)), n/10
            assert(s(n/10 + 2) == 2*s(n/10 + 1) - 2*s(n/10)), n/10

            assert(s(n) == ((1 + 1j)**n).imag), n
        for n in range(-90, 100):
            assert(s(n) == (1 + 1j)**(n - 2) + (1 - 1j)**(n - 2)), n

        import math

        for n in range(-30, 30):
            assert(_eq(s(n), 2**(n/2) * math.sin(math.pi*n/4))), \
                (n, s(n), 2**(n/2) * math.sin(math.pi*n/4))
        print('ok'); sys.stdout.flush()


        print('sign()...', end=''); sys.stdout.flush()
        for x in range(-100, 0):
            assert sign(x) < 0, (x, sign(x))
            assert sign(x/10) < 0, (x, sign(x/10))
        assert sign(0) == 0, sign(0)
        for x in range(1, 100):
            assert sign(x) > 0, (x, sign(x))
            assert sign(x/10) > 0, (x, sign(x/10))
        print('ok'); sys.stdout.flush()



        print('diff()...', end=''); sys.stdout.flush()
        for x in range(-500, 500):
            assert diff(lambda x: x)(x) == 1, (x, diff(lambda x: x)(x))
            assert diff(lambda x: x)(x/2) == 1, (x/2, diff(lambda x: x)(x/2))

            assert diff(lambda x: 3*x)(x) == 3, (x, diff(lambda x: 3*x)(x))
            assert diff(lambda x: 3*x)(x/2) == 3, (x/2, diff(lambda x: 3*x)(x/2))

            assert diff(lambda x: x*x)(x) == 2*x + 1, (x, diff(lambda x: x*x)(x), 2*x + 1)
            assert diff(lambda x: x*x)(x/2) == x + 1, (x/2, diff(lambda x: x*x)(x/2), x + 1)

            for k in range(1, 10):
                assert diff(lambda x: numbernone.falling_factorial_pow(x, k))(x) \
                       == k * numbernone.falling_factorial_pow(x, k - 1), \
                       (x, k, diff(lambda x: numbernone.falling_factorial_pow(x, k))(x),
                        k * numbernone.falling_factorial_pow(x, k - 1))

            for alpha in range(5):
                assert diff(lambda x: x, alpha)(x) == alpha, (x, alpha, diff(lambda x: x, alpha)(x))
                assert diff(lambda x: x, alpha)(x/2) == alpha, \
                       (x/2, alpha, diff(lambda x: x, alpha)(x/2))

                assert diff(lambda x: 3*x, alpha)(x) == 3*alpha, \
                       (x, alpha, diff(lambda x: 3*x, alpha)(x))
                assert diff(lambda x: 3*x, alpha)(x/2) == 3*alpha, \
                       (x/2, alpha, diff(lambda x: 3*x, alpha)(x/2))

            assert diff(lambda x: x*x, alpha)(x) == 2*x*alpha + alpha*alpha, \
                   (x, diff(lambda x: x*x, alpha)(x), 2*x*alpha + alpha*alpha)
            assert diff(lambda x: x*x, alpha)(x/2) == x*alpha + alpha*alpha, \
                   (x/2, diff(lambda x: x*x, alpha)(x/2), x*alpha + alpha*alpha)

        import decimal

        for x in range(-50 if debug.assertspeed >= debug.ASSERT_NORMAL else -25,
                        50 if debug.assertspeed >= debug.ASSERT_NORMAL else 25):
            for k in range(-5, 0):
                assert _eq(diff(lambda x:
                                    numbernone.falling_factorial_pow(x, k))(decimal.Decimal(x)),
                           numbernone.mul(k,
                                          numbernone.falling_factorial_pow(decimal.Decimal(x),
                                                                           k - 1))), \
                       (x, k,
                        diff(lambda x:
                                 numbernone.falling_factorial_pow(x, k))(decimal.Decimal(x)),
                        numbernone.mul(k, numbernone.falling_factorial_pow(decimal.Decimal(x),
                                                                           k - 1)))
        print('???', end='')
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('diff_x()...', end=''); sys.stdout.flush()
        for x in range(-500 if debug.assertspeed >= debug.ASSERT_NORMAL else -100,
                       500 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            assert diff_x(lambda x: x, x) == 1, (x, diff_x(lambda x: x, x))
            assert diff_x(lambda x: x, x/2) == 1, (x/2, diff_x(lambda x: x, x/2))

            assert diff_x(lambda x: 3*x, x) == 3, (x, diff_x(lambda x: 3*x, x))
            assert diff_x(lambda x: 3*x, x/2) == 3, (x/2, diff_x(lambda x: 3*x, x/2))

            assert diff_x(lambda x: x*x, x) == 2*x + 1, (x, diff_x(lambda x: x*x, x), 2*x + 1)
            assert diff_x(lambda x: x*x, x/2) == x + 1, \
                   (x/2, diff_x(lambda x: x*x, x/2), x + 1)

            for k in range(1, 10):
                assert diff_x(lambda x: numbernone.falling_factorial_pow(x, k), x) \
                       == k * numbernone.falling_factorial_pow(x, k - 1), \
                       (x, k, diff_x(lambda x: numbernone.falling_factorial_pow(x, k), x),
                        k * numbernone.falling_factorial_pow(x, k - 1))

            for alpha in range(5):
                assert diff_x(lambda x: x, x, alpha,) == alpha, \
                       (x, alpha, diff_x(lambda x: x, x, alpha))
                assert diff_x(lambda x: x, x/2, alpha) == alpha, \
                       (x/2, alpha, diff_x(lambda x: x, x/2, alpha))

                assert diff_x(lambda x: 3*x, x, alpha) == 3*alpha, \
                       (x, alpha, diff_x(lambda x: 3*x, x, alpha))
                assert diff_x(lambda x: 3*x, x/2, alpha) == 3*alpha, \
                       (x/2, alpha, diff_x(lambda x: 3*x, x/2, alpha))

            assert diff_x(lambda x: x*x, x, alpha) == 2*x*alpha + alpha*alpha, \
                   (x, diff_x(lambda x: x*x, x, alpha), 2*x*alpha + alpha*alpha)
            assert diff_x(lambda x: x*x, x/2, alpha) == x*alpha + alpha*alpha, \
                   (x/2, diff_x(lambda x: x*x, x/2, alpha), x*alpha + alpha*alpha)

        for x in range(-50, 50):
            for k in range(-5, 0):
                assert _eq(diff_x(lambda x: numbernone.falling_factorial_pow(x, k),
                                  decimal.Decimal(x)),
                           numbernone.mul(k,
                                          numbernone.falling_factorial_pow(decimal.Decimal(x),
                                                                           k - 1))),\
                       (x, k,
                        diff_x(lambda x: numbernone.falling_factorial_pow(x, k),
                               decimal.Decimal(x)),
                        numbernone.mul(k, numbernone.falling_factorial_pow(decimal.Decimal(x),
                                                                           k - 1)))
        print('???', end='')
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('sum()...', end=''); sys.stdout.flush()
        for x in range(-1000 if debug.assertspeed >= debug.ASSERT_NORMAL else -100,
                       1000 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            assert sum(lambda x: x) (x) == (x - 1)*x/2, (x, sum(lambda x: x)(x), (x - 1)*x/2)
            assert sum(lambda x: x + 1)(x) == x*(x + 1)/2, \
                   (x, sum(lambda x: x + 1)(x), x*(x + 1)/2)
            assert sum(lambda x: x, -1)(x) == -x*(x + 1)/2, \
                   (x, sum(lambda x: x, -1)(x), -x*(x + 1)/2)
        print('???', end='')
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('sum_x()...', end=''); sys.stdout.flush()
        for x in range(-1000 if debug.assertspeed >= debug.ASSERT_NORMAL else -100,
                       1000 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            assert sum_x(lambda x: x, x) == (x - 1)*x/2, (x, sum_x(lambda x: x, x), (x - 1)*x/2)
            assert sum_x(lambda x: x + 1, x) == x*(x + 1)/2, \
                   (x, sum_x(lambda x: x + 1, x), x*(x + 1)/2)
            assert sum_x(lambda x: x, x, -1) == -x*(x + 1)/2, \
                   (x, sum_x(lambda x: x, x, -1), -x*(x + 1)/2)
        print('???', end='')
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('sym()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('sym_x()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('trans()...', end=''); sys.stdout.flush()
        for x in range(-1000, 1000):
            assert trans(lambda x: x)(x) == x + 1, (x, trans(lambda x: x)(x), x + 1)
            assert trans(lambda x: x, -1)(x) == x - 1, (x, trans(lambda x: x, -1)(x), x - 1)
            assert trans(lambda x: x, 2.5)(x) == x + 2.5, (x, trans(lambda x: x, 2.5)(x), x + 2.5)
            assert trans(lambda x: x*x, 2.5)(x) == (x + 2.5)**2, \
                   (x, trans(lambda x: x*x, 2.5)(x), (x + 2.5)**2)
            assert trans(lambda x: 3*x, 2.5)(x) == 3*(x + 2.5), \
                   (x, trans(lambda x: 3*x, 2.5)(x), 3*(x + 2.5))
            assert trans(lambda x: 3*x, 2.5)(x/2) == 3*(x/2 + 2.5), \
                   (x, trans(lambda x: 3*x, 2.5)(x/2), 3*(x/2 + 2.5))
        print('ok'); sys.stdout.flush()


        print('trans_x()...', end=''); sys.stdout.flush()
        for x in range(-1000, 1000):
            assert trans_x(lambda x: x, x) == x + 1, (x, trans_x(lambda x: x, x), x + 1)
            assert trans_x(lambda x: x, x, -1) == x - 1, (x, trans_x(lambda x: x, x, -1), x - 1)
            assert trans_x(lambda x: x, x, 2.5) == x + 2.5, \
                   (x, trans_x(lambda x: x, x, 2.5), x + 2.5)
            assert trans_x(lambda x: x*x, x, 2.5) == (x + 2.5)**2, \
                   (x, trans_x(lambda x: x*x, x, 2.5), (x + 2.5)**2)
            assert trans_x(lambda x: 3*x, x, 2.5) == 3*(x + 2.5), \
                   (x, trans_x(lambda x: 3*x, x, 2.5), 3*(x + 2.5))
            assert trans_x(lambda x: 3*x, x/2, 2.5) == 3*(x/2 + 2.5), \
                   (x, trans_x(lambda x: 3*x, x/2, 2.5), 3*(x/2 + 2.5))
        print('ok'); sys.stdout.flush()


        print('trans_vert()...', end=''); sys.stdout.flush()
        for x in range(-1000, 1000):
            assert trans_vert(lambda x: x)(x) == x + 1, (x, trans_vert(lambda x: x)(x), x + 1)
            assert trans_vert(lambda x: x, -1)(x) == x - 1, \
                (x, trans_vert(lambda x: x, -1)(x), x - 1)
            assert trans_vert(lambda x: x, 2.5)(x) == x + 2.5, \
                (x, trans_vert(lambda x: x, 2.5)(x), x + 2.5)
            assert trans_vert(lambda x: x*x, 2.5)(x) == x*x + 2.5, \
                   (x, trans_vert(lambda x: x*x, 2.5)(x), x*x + 2.5)
            assert trans_vert(lambda x: 3*x, 2.5)(x) == 3*x + 2.5, \
                   (x, trans_vert(lambda x: 3*x, 2.5)(x), 3*x + 2.5)
            assert trans_vert(lambda x: 3*x, 2.5)(x/2) == 3*x/2 + 2.5, \
                   (x, trans_vert(lambda x: 3*x, 2.5)(x/2), 3*x/2 + 2.5)
        print('ok'); sys.stdout.flush()


        print('trans_vert_x()...', end=''); sys.stdout.flush()
        for x in range(-1000, 1000):
            assert trans_vert_x(lambda x: x, x) == x + 1, (x, trans_vert_x(lambda x: x, x), x + 1)
            assert trans_vert_x(lambda x: x, x, -1) == x - 1, \
                (x, trans_vert_x(lambda x: x, x, -1), x - 1)
            assert trans_vert_x(lambda x: x, x, 2.5) == x + 2.5, \
                   (x, trans_vert_x(lambda x: x, x, 2.5), x + 2.5)
            assert trans_vert_x(lambda x: x*x, x, 2.5) == x*x + 2.5, \
                   (x, trans_vert_x(lambda x: x*x, x, 2.5), x*x + 2.5)
            assert trans_vert_x(lambda x: 3*x, x, 2.5) == 3*x + 2.5, \
                   (x, trans_vert_x(lambda x: 3*x, x, 2.5), 3*x + 2.5)
            assert trans_vert_x(lambda x: 3*x, x/2, 2.5) == 3*x/2 + 2.5, \
                   (x, trans_vert_x(lambda x: 3*x, x/2, 2.5), 3*x/2 + 2.5)
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
