#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.factors Facteurs premiers

##\file
# Facteurs premiers

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 8 dcembre 2006
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'factors --- 2010 April 12'

import bisect, fractions, types

import DSPython
import DSPython.nat32 as nat32
import DSPython.natural as natural



# ###########
# Fonctions #
#############
##\brief Pour s = \htmlonly(&hellip; (P<sub>i</sub>, e<sub>i</sub>) &hellip;)\endhtmlonly
## o P<sub>i</sub> est le i<sup>e</sup> nombre premier
## renvoie \htmlonly&sum;<sub>i</sub> e<sub>i</sub>.2<sup>i - 1</sup>\endhtmlonly
def bef(s):
    """Pour s = [... (P_i, e_i) ...] o P_i est le ime nombre premier
    renvoie Sum_i e_i * 2**(i - 1)
    [OEIS A048675] (Binary Encoding of Factorizations)

    Pre: s: squence strictement ordonne de primaries

    Result: natural

    O(s) = ..."""
    assert primaries_is(s), s

    n = 0
    for p in s:
        i = bisect.bisect_left(nat32.PRIME_16_ARRAY, p[0])
        if i >= len(nat32.PRIME_16_ARRAY):
            raise NotImplementedError
        n += p[1] * (2**i)
    return n


## s et t sont premiers entre eux ?
def coprime_is(s, t):
    """Renvoie True si s et t sont premiers entre eux,
      False sinon

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries

    Result: boolean

    O(s, t) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t

    return gcd(s, t) == []


##\brief Distance de Dominici  partir des primaries
#
# (cf. <i lang="en">An Arithmetic Metric</i>,
# Diego <span style="font-variant:small-caps">Dominici</span>,
# <a href="http://arxiv.org/abs/0906.0632/"
#   target="_blank"><tt>http://arxiv.org/abs/0906.0632/</tt></a>)
def distance_dominici(s, t):
    """Renvoie la distance de Dominici entre les naturels
      ayant s et t pour liste de leurs primaries

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries

    Result: naturel
    O(s) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t

    d = 0
    for p in lcm(s, t):
        d += p[1]
    for p in gcd(s, t):
        d -= p[1]
    return d


## Nombre des diviseurs  partir des primaries (fonction \htmlonly &nu;\endhtmlonly)
def divisors_nb(s):
    """Renvoie le nombre des diviseurs du naturel
      ayant s pour liste des primaries
      == divisors_sum_pow(s, 0)

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    n = 1
    for i in s:
        n *= i[1] + 1
    return n


## Produit des diviseurs  partir des primaries (fonction \htmlonly &piv;\endhtmlonly)
def divisors_prod(s):
    """Renvoie le produit des diviseurs du naturel
      ayant s pour liste des primaries

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    nb = divisors_nb(s)
    if nb&1:  # nb impair (ssi le naturel correspondant  s est un carr)
        p = 1
        for i in s:
            assert i[1]%2 == 0, (i, s)

            p *= i[0]**(i[1]//2)
        return p**nb
    else:     # nb pair (ssi le naturel correspondant  s n'est pas un carr)
        return primaries_to_n(s)**(nb//2)


## Somme des diviseurs  partir des primaries (fonction \htmlonly &sigma;\endhtmlonly)
def divisors_sum(s):
    """Renvoie la somme des diviseurs du naturel
      ayant s pour liste des primaries
      == divisors_sum_pow(s, 1)

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    n = divisors_sum_odd(s)
    return (n if (len(s) == 0) or (s[0][0] != 2)     # n impair
            else n * natural.mersenne(s[0][1] + 1))  # n pair : M_(k+1) * divisors_sum_odd


## Somme des diviseurs pairs  partir des primaries
def divisors_sum_even(s):
    """Renvoie la somme des diviseurs pairs du naturel
      ayant s pour liste des primaries

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    if (len(s) == 0) or (s[0][0] != 2):  # nombre impair
        return 0
    else:                                # nombre pair
        n = 1
        for i in s[1:]:
            n *= (i[0]**(i[1] + 1) - 1) // (i[0] - 1)
        return (n * natural.mersenne(s[0][1])) << 1  # == 2 * M_k * divisors_sum_odd


## Somme des diviseurs impairs  partir des primaries
def divisors_sum_odd(s):
    """Renvoie la somme des diviseurs pairs du naturel
      ayant s pour liste des primaries

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    n = 1
    if (len(s) == 0) or (s[0][0] != 2):  # nombre impair
        for i in s:
            n *= (i[0]**(i[1] + 1) - 1) // (i[0] - 1)
    else:                                # nombre pair
        for i in s[1:]:
            n *= (i[0]**(i[1] + 1) - 1) // (i[0] - 1)
    return n


## Somme des diviseurs  partir des primaries (fonction \htmlonly &sigma;\endhtmlonly<sub>k</sub>)
def divisors_sum_pow(s, k):
    """Renvoie la somme des diviseurs ** k du naturel
      ayant s pour liste des primaries

    Pre: s: squence strictement ordonne de primaries
         k: naturel

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s
    assert DSPython.natural_is(k), k

    if k > 1:
        n = 1
        for i in s:
            n *= (i[0]**(k*(i[1] + 1)) - 1) // (i[0]**k - 1)
        return n
    elif k == 1:
        return divisors_sum(s)
    else:
        return divisors_nb(s)


##\brief Pour n =
## \htmlonly&sum;<sub>i=0</sub><sup>&infin;</sup> b<sub>i</sub>.2<sup>i</sup>\endhtmlonly
## renvoie les primaries de
## \htmlonly&prod;<sub>i=1</sub><sup>&infin;</sup>
## P<sub>i</sub><sup>b<sub>i-1</sub></sup>\endhtmlonly
## o P<sub>i</sub> est le i<sup>e</sup> nombre premier
def feb_primaries(n):
    """Pour n = Sum_(i=0)^infini b_i * 2**i
    renvoie les primaries de Prod_(i=1)^infini P_i**(b_(i-1))
    o P_i est le ime nombre premier
    [OEIS A019565]
    bef(feb_primaries(n)) == n

    Pre: n: natural

    Result: squence strictement ordonne de primaries (naturel premier, 1)

    O(n) = ..."""
    assert DSPython.natural_is(n), n

    i = 0
    s = []
    while (n > 0) and (i < len(nat32.PRIME_16_ARRAY)):
        if n&1:
            s.append((nat32.PRIME_16_ARRAY[i], 1))
        i += 1
        n >>= 1
    if n > 0:
        raise NotImplementedError
    return s


## PGCD de s et t (Plus Grand Commun Diviseur/ Greatest Common Divisor)
def gcd(s, t):
    """Renvoie le PGCD de s et t
      (Plus Grand Commun Diviseur/ Greatest Common Divisor)

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries

    Result: squence strictement ordonne de primaries

    O(s, t) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t

    if (len(s) > 0) and (len(t) > 0):
        s_i = 0
        t_i = 0
        l = []
        while (len(s) > s_i) and (len(t) > t_i):
            while s[s_i][0] < t[t_i][0]:
                s_i += 1
                if len(s) <= s_i:
                    return l
            while s[s_i][0] > t[t_i][0]:
                t_i += 1
                if len(t) <= t_i:
                    return l
            if s[s_i][0] != t[t_i][0]:
                continue
            l.append((s[s_i][0], min(s[s_i][1], t[t_i][1])))
            s_i += 1
            t_i += 1
        return l
    else:
        return []


##\brief Renvoie le "nombre de Gdel" de la squence s :
## \htmlonly
## pour s == (a<sub>1</sub>, a<sub>2</sub>, a<sub>3</sub>, &hellip;.., a<sub>k</sub>),
## renvoie &prod;<sub>i=1</sub><sup>k</sup> P<sub>i</sub><sup>a<sub>i</sub></sup>
## \endhtmlonly
def godelnumber(s):
    """Renvoie le "nombre de Gdel" de la squence s
    Pour s == (a_1, a_2, a_3, ..., a_k), renvoie prod_i=1^k (P_i)^(a_i)
    o P_i est le ime nombre premier

    Pre: s: squence de naturels

    Result: naturel >= 1

    O() = ..."""
    assert isinstance(s, list) or isinstance(s, tuple), type(s)

    n = 1
    if len(s) > len(nat32.PRIME_16_ARRAY):
        raise NotImplementedError
    for i in range(len(s)):
        assert DSPython.natural_is(s[i]), (i, type(s[i]), s[i])

        n *= nat32.PRIME_16_ARRAY[i]**s[i]
    return n


##\brief
## \htmlonly
## Pour n == &prod;<sub>i=1</sub><sup>k</sup> P<sub>i</sub><sup>a<sub>i</sub></sup>,
## renvoie [a<sub>1</sub>, a<sub>2</sub>, a<sub>3</sub>, &hellip;.., a<sub>k</sub>]
## \endhtmlonly
def godelnumber_to_list(n):
    """Pour n == prod_i=1^k (P_i)^(a_i) renvoie [a_1, a_2, a_3, ..., a_k]
    Si n == 1 alors renvoie []

    Pre: n: naturel >= 1

    Result: [] ou list de naturels dont le dernier lment est != 0

    O() = ..."""
    assert DSPython.natural_is(n), n
    assert n > 0, n

    if n == 1:
        return []
    l = [natural.scan1(n)]
    n >>= l[0]
    for p in nat32.PRIME_16_ARRAY[1:]:  # tant qu'on sait diviser n
        if n == 1:
            return l
        a = 0
        q, r = divmod(n, p)
        while r == 0:  # tant que divisible
            a += 1
            n = q
            q, r = divmod(n, p)
        l.append(a)
    if n == 1:
        return l
    raise NotImplementedError  # plus de nombre premier dans la table


## PPCM de s et t (Plus Petit Commun Multiple/ Least Common Multiple)
def lcm(s, t):
    """Renvoie PPCM de s et t
      (Plus Petit Commun Multiple/ Least Common Multiple)

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries

    Result: squence strictement ordonne de primaries

    O(s, t) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t

    if (len(s) > 0) and (len(t) > 0):
        s_i = 0
        t_i = 0
        l = []
        while (len(s) > s_i) and (len(t) > t_i):
            if s[s_i][0] < t[t_i][0]:
                l.append((s[s_i][0], s[s_i][1]))
                s_i += 1
            elif s[s_i][0] > t[t_i][0]:
                l.append((t[t_i][0], t[t_i][1]))
                t_i += 1
            else:
                l.append((s[s_i][0], max(s[s_i][1], t[t_i][1])))
                s_i += 1
                t_i += 1
        if len(s) > s_i:
            return l + s[s_i:]
        return (l + t[t_i:] if len(t) > t_i
                else l)
    elif len(s) > 0:                   # t == []
        return s
    else:                              # s == []
        return t


## Fonction de Mbius  partir des primaries (fonction \htmlonly &mu;\endhtmlonly)
def mobius(s):
    """Renvoie la fonction de Mbius du naturel
      ayant s pour liste des primaries

    Pre: s: squence strictement ordonne de primaries

    Result: -1, 0 ou 1

    O(s) = ..."""
    assert primaries_is(s), s

    k = 0
    for i in s:
        if i[1] > 1:
            return 0
        k += 1
    return (1 if k&1 == 0
            else -1)


## Produit de s et t
def mul(s, t):
    """Renvoie le produit s et t

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries

    Result: squence strictement ordonne de primaries

    O(s, t) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t

    if (len(s) > 0) and (len(t) > 0):
        s_i = 0
        t_i = 0
        l = []
        while (len(s) > s_i) and (len(t) > t_i):
            while s[s_i][0] < t[t_i][0]:
                l.append((s[s_i][0], s[s_i][1]))
                s_i += 1
                if len(s) <= s_i:
                    return l + t[t_i:]
            while s[s_i][0] > t[t_i][0]:
                l.append((t[t_i][0], t[t_i][1]))
                t_i += 1
                if len(t) <= t_i:
                    return l + s[s_i:]
            if s[s_i][0] == t[t_i][0]:
                l.append((s[s_i][0], s[s_i][1] + t[t_i][1]))
                s_i += 1
                t_i += 1
        if len(s) > s_i:
            return l + s[s_i:]
        return (l + t[t_i:] if len(t) > t_i
                else l)
    elif len(s) > 0:                   # t == []
        return s
    else:                              # s == []
        return t


##\brief Nombre de dcompositions (en tenant compte de l'ordre) en somme de 4 carrs d'entiers
## (fonction GR)
def nb_in_integers_4sqr(s):
    """Renvoie le nombre de dcompositions (en tenant compte de l'ordre)
      en somme de 4 carrs d'entiers du naturel
      ayant s pour liste des primaries

    Pre: s: squence strictement ordonne de primaries

    Result: naturel multiple de 8

    O(s) = ..."""
    assert primaries_is(s), s

    return ((divisors_sum_odd(s) * 3) << 3 if (len(s) > 0) and (s[0][0] == 2)  # nombre pair
            else divisors_sum_odd(s) << 3)                                     # nombre impair


## Produit des facteurs non communs de s et t (Not common Factors Product)
def nfp(s, t):
    """Renvoie le produit des facteurs non communs de s et t
      (Not common Factors Product)

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries

    Result: squence strictement ordonne de primaries

    O(s, t) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t

    if (len(s) > 0) and (len(t) > 0):
        s_i = 0
        t_i = 0
        l = []
        while (len(s) > s_i) and (len(t) > t_i):
            while s[s_i][0] < t[t_i][0]:
                l.append((s[s_i][0], s[s_i][1]))
                s_i += 1
                if len(s) <= s_i:
                    return l + t[t_i:]
            while s[s_i][0] > t[t_i][0]:
                l.append((t[t_i][0], t[t_i][1]))
                t_i += 1
                if len(t) <= t_i:
                    return l + s[s_i:]
            if s[s_i][0] == t[t_i][0]:
                if s[s_i][1] > t[t_i][1]:
                    l.append((s[s_i][0], s[s_i][1] - t[t_i][1]))
                elif s[s_i][1] < t[t_i][1]:
                    l.append((s[s_i][0], t[t_i][1] - s[s_i][1]))
                s_i += 1
                t_i += 1
        if len(s) > s_i:
            return l + s[s_i:]
        return (l + t[t_i:] if len(t) > t_i
                else l)
    elif len(s) > 0:                   # t == []
        return s
    else:                              # s == []
        return t


## Nontotient  partir des primaries
def nontotient(s):
    """Renvoie le nombre de nontotatives du naturel
      ayant s pour liste des primaries
      (Si s == [] alors renvoie 0)

    Pre: s: squence strictement ordonne de primaries

    Result: naturel

    O(n) = ..."""
    assert primaries_is(s), s

    return primaries_to_n(s) - totient(s)


##\brief
## \htmlonly
## Pour s = ((p<sub>1</sub>, &alpha;<sub>1</sub>), (p<sub>2</sub>, &alpha;<sub>2</sub>),
## (p<sub>3</sub>, &alpha;<sub>3</sub>), &hellip;, (p<sub>k</sub>, &alpha;<sub>k</sub>))
## et t = ((q<sub>1</sub>, &beta;<sub>1</sub>), (q<sub>2</sub>, &beta;<sub>2</sub>),
## (q<sub>3</sub>, &beta;<sub>3</sub>), &hellip;, (q<sub>l</sub>, &beta;<sub>l</sub>)),<br>
## renvoie [(r<sub>1</sub>, &gamma;<sub>1</sub>), (r<sub>2</sub>, &gamma;<sub>2</sub>),
## (r<sub>3</sub>, &gamma;<sub>3</sub>), &hellip;, (r<sub>m</sub>, &gamma;<sub>m</sub>)]<br>
## tel que pour les p<sub>i</sub> = q<sub>j</sub>
## on ai r<sub>n</sub> = p<sub>i</sub>
## et &gamma;<sub>n</sub> = f(&alpha;<sub>i</sub>, &beta;<sub>j</sub>).<br>
## Pour les p<sub>i</sub> &ne; q<sub>j</sub>
## on prend &alpha;<sub>i</sub> = 0 ou &beta;<sub>j</sub> = 0.
## On s'arrte ds il n'y a plus de primary.
## \endhtmlonly
def ope_exp(s, t, f):
    """Renvoie []

    Pre: s: squence strictement ordonne de primaries
         t: squence strictement ordonne de primaries
         f: fonction : (naturel >= 1, naturel >= 1) -> naturel

    Result: squence strictement ordonne de primaries

    O(s, t) = ..."""
    assert primaries_is(s), s
    assert primaries_is(t), t
    assert isinstance(f, types.FunctionType), type(f)

    s_i = 0  # position dans s
    t_i = 0  # position dans t
    l = []  # rsultat
    while (len(s) > s_i) and (len(t) > t_i):  # il reste des p_i et des q_j
        if s[s_i][0] == t[t_i][0]:   # p_i = q_j
            gamma = f(s[s_i][1], t[t_i][1])
            if gamma > 0:
                l.append((s[s_i][0], gamma))
            s_i += 1
            t_i += 1
        elif s[s_i][0] > t[t_i][0]:  # p_i > q_j
            gamma = f(0, t[t_i][1])
            if gamma > 0:
                l.append((t[t_i][0], gamma))
            t_i += 1
        else:                        # p_i < q_j
            gamma = f(s[s_i][1], 0)
            if gamma > 0:
                l.append((s[s_i][0], gamma))
            s_i += 1
    while len(s) > s_i:  # il ne reste que des p_i
        gamma = f(s[s_i][1], 0)
        if gamma > 0:
            l.append((s[s_i][0], gamma))
        s_i += 1
    while len(t) > t_i:  # il ne reste que des q_j
        gamma = f(0, t[t_i][1])
        if gamma > 0:
            l.append((t[t_i][0], gamma))
        t_i += 1

    assert primaries_is(l), l

    return l


##\brief Liste des primaries de n,\n
## \htmlonly
## c.--d. [(p<sub>1</sub>, &alpha;<sub>1</sub>), (p<sub>2</sub>, &alpha;<sub>2</sub>),
## (p<sub>3</sub>, &alpha;<sub>3</sub>), &hellip;, (p<sub>k</sub>, &alpha;<sub>k</sub>)]
## tel que n = &prod;<sub>i=1</sub><sup>k</sup> p<sub>i</sub><sup>&alpha;<sub>i</sub></sup>
## avec p<sub>i</sub> premier,
## p<sub>i</sub> &lt; p<sub>i + 1</sub>
## et &alpha;<sub>i</sub> &ge; 1
## \endhtmlonly
def primaries(n):
    """Renvoie la liste des primaries (dans l'ordre strictement croissant) de n
      c.--d. les couples (facteur premier, exposant) dcomposant n
      (Si n == 1 alors renvoie []).
    Le nombre de primaries (la longueur de la liste renvoye) correspond
      au nombre de facteurs premiers distincts de n
      (fonction omega)

    Pre: n: naturel >= 1

    Result: squence strictement ordonne (sur le premier lment) de couples
              (naturel premier, naturel >= 1)

    O(n) = ..."""
    assert DSPython.natural_is(n), n
    assert n > 0, n

    # Facteurs 2
    nb = natural.scan1(n)
    if nb > 0:
        n >>= nb
        l = [(2, nb)]
    else:
        l = []

    # Facteurs premiers tenant sur 16 bits
    for i in range(1, len(nat32.PRIME_16_ARRAY)):
        nb = 0
        d = nat32.PRIME_16_ARRAY[i]
        q, r = divmod(n, d)
        while r == 0:
            nb += 1
            n = q
            q, r = divmod(n, d)
        if nb > 0:
            l.append((d, nb))
        if n == 1:
            return l

    assert d%nat32.PRIMORIAL32_TUPLE[4] == 1, \
           (d, nat32.PRIMORIAL32_TUPLE[4], d%nat32.PRIMORIAL32_TUPLE[4])

    # Facteurs premiers ne tenant pas sur 16 bits
    while True:
        for m in nat32.MODS_PRIMORIAL_4_DIFF_TUPLE:
            # Parcourt les restes modulo 210 premiers avec 210
            nb = 0
            q, r = divmod(n, d)
            while r == 0:
                nb += 1
                n = q
                q, r = divmod(n, d)
            if nb > 0:
                l.append((d, nb))
                if n == 1:
                    return l
            d += m


## Renvoie True si p est une squence strictement ordonne de primaries, False sinon
def primaries_is(s):
    """Renvoie True si p est une squence strictement ordonne de primaries,
    False sinon

    Pre: s: quelconque

    Result: boolean"""
    if isinstance(s, list) or isinstance(s, tuple):
        prev_n = -1
        prev_e = -1
        for i in s:
            if ((not isinstance(i, tuple)) or (len(i) != 2)
                or (not DSPython.natural_is(i[0])) or (not DSPython.natural_is(i[1]))
                or (i[1] == 0) or (prev_n >= i[0]) or (not natural.prime_is(i[0]))):
                return False
            prev_n = i[0]
            prev_e = i[1]
        return True
    else:
        return False


## Renvoie True si s reprsente un nombre premier, False sinon
def primaries_prime_is(s):
    """Renvoie True si s reprsente un nombre premier, False sinon

    Pre: s: squence strictement ordonne de primaries

    Result: boolean"""
    assert primaries_is(s), s

    return (len(s) == 1) and (s[0][1] == 1)


## String reprsentant les primaries
def primaries_str(s, times='.', exp='^'):
    """Renvoie un string reprsentant les primaries
      (Si s est vide alors renvoie '1')

    Pre: s: squence strictement ordonne de primaries
         times: string reprsentant l'oprateur de multiplication
         exp: string reprsentant l'oprateur d'exponentiation

    Result: string

    O(s) = ..."""
    assert primaries_is(s), s

    if len(s) > 0:
        l = []
        for i in s:
            l.append('{0}{1}{2}'.format(i[0], exp, i[1]) if i[1] > 1
                     else str(i[0]))
        return times.join(l)
    else:
        return '1'


## Produit des primaries
def primaries_to_n(s):
    """Renvoie le naturel ayant s pour liste des primaries
      (Si s est vide alors renvoie 1)

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    p = 1
    for i in s:
        p *= i[0]**i[1]
    return p


## Liste des primes  partir des primaries
def primaries_to_primes(s):
    """Renvoie la liste des primes  partir de la liste des primaries
      (Si s est vide alors renvoie [])

    Pre: s: squence strictement ordonne de primaries

    Result: squence ordonne de primes

    O(s) = ..."""
    assert primaries_is(s), s

    r = []
    for c in s:
        r += [c[0]] * c[1]
    return r


## Liste des primes de n
def primes(n):
    """Renvoie la liste des primes (dans l'ordre croissant) de n
      c.--d. les facteurs premiers de n
      (Si n == 1 alors renvoie []).
    Le nombre de primes (la longueur de la liste renvoye) correspond
      au nombre de facteurs premiers (non ncessairement distincts) de n
      (fonction Omega)

    Pre: n: naturel >= 1

    Result: squence ordonne de naturels premiers

    O(n) = ..."""
    assert DSPython.natural_is(n), n
    assert n > 0, n

    # Facteurs 2
    nb = natural.scan1(n)
    n >>= nb
    l = [2] * nb

    # Facteurs premiers tenant sur 16 bits
    for i in range(1, len(nat32.PRIME_16_ARRAY)):
        nb = 0
        d = nat32.PRIME_16_ARRAY[i]
        q, r = divmod(n, d)
        while r == 0:
            nb += 1
            n = q
            q, r = divmod(n, d)
        if nb > 0:
            l += [d] * nb
        if n == 1:
            return l

    assert d%nat32.PRIMORIAL32_TUPLE[4] == 1, \
           (d, nat32.PRIMORIAL32_TUPLE[4], d%nat32.PRIMORIAL32_TUPLE[4])

    # Facteurs premiers ne tenant pas sur 16 bits
    while True:
        for m in nat32.MODS_PRIMORIAL_4_DIFF_TUPLE:
            # Parcourt les restes modulo 210 premiers avec 210
            nb = 0
            q, r = divmod(n, d)
            while r == 0:
                nb += 1
                n = q
                q, r = divmod(n, d)
            if nb > 0:
                l += [d] * nb
                if n == 1:
                    return l
            d += m


## Renvoie True si p est une squence ordonne de primes, False sinon
def primes_is(s):
    """Renvoie True si p est une squence ordonne de primes,
    False sinon

    Pre: s: quelconque

    Result: boolean"""
    if isinstance(s, list) or isinstance(s, tuple):
        prev_n = 1
        for n in s:
            if (not DSPython.natural_is(n)) or (prev_n > n) or (not natural.prime_is(n)):
                return False
            prev_n = n
        return True
    else:
        return False


## String reprsentant les primes
def primes_str(s, times='.'):
    """Renvoie un string reprsentant les primes
      (Si s est vide alors renvoie '1')

    Pre: s: squence ordonne de primes
         times: string reprsentant l'oprateur de multiplication

    Result: string

    O(s) = ..."""
    assert primes_is(s), s

    if len(s) > 0:
        l = []
        for i in s:
            l.append(str(i))
        return times.join(l)
    else:
        return '1'


## Produit des primes
def primes_to_n(s):
    """Renvoie le naturel ayant s pour liste des primes
      (Si s est vide alors renvoie 1)

    Pre: s: squence ordonne de primes

    Result: naturel >= 1

    O(s) = ..."""
    assert primes_is(s), s

    p = 1
    for i in s:
        p *= i
    return p


## Liste des primaries  partir des primes
def primes_to_primaries(s):
    """Renvoie la liste des primaries  partir de la liste des primes
      (Si s est vide alors renvoie [])

    Pre: s: squence ordonne de primes

    Result: squence strictement ordonne de primaries

    O(s) = ..."""
    assert primes_is(s), s

    if len(s) > 0:
        r = []
        prev = None
        nb = 1
        for n in s:
            if n == prev:
                nb += 1
            else:
                if prev != None:
                    r.append((prev, nb))
                nb = 1
                prev = n
        r.append((prev, nb))
        return r
    else:
        return []


## Somme des diviseurs  partir des primaries (fonction s)
def properdivisors_sum(s):
    """Renvoie la somme des diviseurs propres du naturel
      ayant s pour liste des primaries
      == divisors_sum(s) - primaries_to_n(s)

    Pre: s: squence strictement ordonne de primaries

    Result: naturel >= 1

    O(s) = ..."""
    assert primaries_is(s), s

    p = 1
    n = 1
    for i in s:
        t = i[0]**i[1]
        p *= t
        n *= (t*i[0] - 1) // (i[0] - 1)
    return n - p


## Produit des facteurs premiers distincts  partir des primaries (fonction radical)
def rad(s):
    """Pour s = [... (P_i, e_i) ...] o P_i est le ime nombre premier
    renvoie [... (P_i, 1) ...]
    Renvoie le produit des facteurs premiers distincts
      du naturel ayant s pour liste des primaries
    [OEIS A007947] Largest square-free number dividing n (the square-free kernel of n)

    Pre: s: squence strictement ordonne de primaries

    Result: squence strictement ordonne de primaries (naturel premier, 1)

    O(s) = ..."""
    assert primaries_is(s), s

    l = []
    for p in s:
        l.append((p[0], 1))
    return l


## Renvoie True si p reprsente un nombre sans carr, False sinon
def squarefree_is(s):
    """Renvoie True si p reprsente un nombre sans carr [OEIS A005117],
    False sinon

    Pre: s: squence strictement ordonne de primaries

    Result: boolean"""
    assert primaries_is(s), s

    for p in s:
        if p[1] > 1:
            return False
    return True


## Totient  partir des primaries
def totient(s):
    """Renvoie le nombre de totatives du naturel
      ayant s pour liste des primaries
      (Si s == [] alors renvoie 1)

    Pre: s: squence strictement ordonne de primaries

    Result: naturel

    O(n) = ..."""
    assert primaries_is(s), s

    nb = 1
    for i in s:
        nb *= (i[0] - 1) * i[0]**(i[1] - 1)
    return nb



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        import DSPython.debug as debug

        import DSPython.natseq as natseq

        debug.test_begin(VERSION, __debug__)

        print('bef()...', end=''); sys.stdout.flush()
        assert bef([]) == 0, bef([])
        assert bef([(2, 1)]) == 1, bef([(2, 1)])
        assert bef([(2, 3)]) == 3, bef([(2, 3)])
        assert bef([(5, 1)]) == 4, bef([(5, 1)])
        assert bef([(5, 3)]) == 12, bef([(5, 3)])
        assert bef([(2, 10), (5, 3)]) == 22, bef([(2, 10), (5, 3)])
        assert bef([(2, 10), (5, 3), (11, 2)]) == 54, bef([(2, 10), (5, 3), (11, 2)])
        for i in range(100):
            assert bef([(nat32.PRIME_16_ARRAY[i], 1)]) == 2**i, \
                   (i, nat32.PRIME_16_ARRAY[i], bef([(nat32.PRIME_16_ARRAY[i], 1)]), 2**i)
        print('ok'); sys.stdout.flush()


        print('coprime_is()...', end=''); sys.stdout.flush()
        assert coprime_is([], []), coprime_is([], [])
        for m in range(1, 50 if debug.assertspeed >= debug.ASSERT_NORMAL else 30):
            f = primaries(m)
            assert coprime_is(f, []), (m, f, coprime_is(primaries(m), []))
            for p in nat32.PRIME_16_ARRAY[:1000 if debug.assertspeed >= debug.ASSERT_NORMAL else
                                          100]:
                if m%p == 0:
                    assert not coprime_is(f, [(p, 1)]), (m, f, p, coprime_is(f, [(p, 1)]))
                else:
                    assert coprime_is(f, [(p, 1)]), (m, f, p, coprime_is(f, [(p, 1)]))
            assert coprime_is(f, primaries(m + 1)), m
        for m in range(1, 50 if debug.assertspeed >= debug.ASSERT_NORMAL else 30):
            f_m = primaries(m)
            for n in range(1, 10):
                f_n = primaries(n)
                assert coprime_is(f_m, f_n) == coprime_is(f_n, f_m), \
                       (m, n, coprime_is(f_m, f_n), gcd(f_n, f_m))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for m in range(2**32 - 1, 2**32 + 1):
                assert coprime_is(primaries(m), primaries(m + 1)), m
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('distance_dominici()...', end=''); sys.stdout.flush()
        f = primaries(11)
        g = primaries(12)
        assert distance_dominici(f, g) == 4, distance_dominici(f, g)
        for a in range(1, 100):
            f = primaries(a)
            assert distance_dominici(f, f) == 0, distance_dominici(f, f)
            for b in range(1, 50):
                g = primaries(b)
                assert distance_dominici(f, g) == distance_dominici(g, f), \
                       (distance_dominici(f, g), distance_dominici(g, f))
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('divisors_nb()...', end=''); sys.stdout.flush()
        assert divisors_nb([]) == 1,         divisors_nb([])
        assert divisors_nb([(2, 1)]) == 2,   divisors_nb([(2, 1)])
        assert divisors_nb([(2, 2)]) == 3,   divisors_nb([(2, 2)])
        assert divisors_nb([(2, 1), (3, 1)]) == 4,  divisors_nb([(2, 1), (3, 1)])
        assert divisors_nb([(2, 5), (3, 2)]) == 18, divisors_nb([(2, 5), (3, 2)])

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors(n)
            assert divisors_nb(f) == len(d), (n, divisors_nb(f), len(d), f, d)
        print('ok'); sys.stdout.flush()


        print('divisors_prod()...', end=''); sys.stdout.flush()
        assert divisors_prod([]) == 1,         divisors_prod([])
        assert divisors_prod([(2, 1)]) == 2,   divisors_prod([(2, 1)])
        assert divisors_prod([(2, 2)]) == 8,   divisors_prod([(2, 2)])
        assert divisors_prod([(2, 1), (3, 1)]) == 36,    divisors_prod([(2, 1), (3, 1)])
        assert divisors_prod([(2, 5), (3, 2)]) == 13631146639813244878848, \
               divisors_prod([(2, 5), (3, 2)])

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors(n)
            assert divisors_prod(f) == natseq.prod(d), (n, divisors_prod(f), natseq.prod(d), f, d)
        print('ok'); sys.stdout.flush()


        print('divisors_sum()...', end=''); sys.stdout.flush()
        assert divisors_sum([]) == 1,       divisors_sum([])
        assert divisors_sum([(2, 1)]) == 3, divisors_sum([(2, 1)])
        assert divisors_sum([(2, 2)]) == 7, divisors_sum([(2, 2)])
        assert divisors_sum([(2, 1), (3, 1)]) == 12,    divisors_sum([(2, 1), (3, 1)])
        assert divisors_sum([(2, 5), (3, 2)]) == 63*13, divisors_sum([(2, 5), (3, 2)])

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors(n)
            assert divisors_sum(f) == natseq.sum(d), (n, divisors_sum(f), natseq.sum(d), f, d)
        print('ok'); sys.stdout.flush()


        print('divisors_sum_odd()...', end=''); sys.stdout.flush()
        assert divisors_sum_odd([]) == 1,       divisors_sum_odd([])
        assert divisors_sum_odd([(2, 1)]) == 1, divisors_sum_odd([(2, 1)])
        assert divisors_sum_odd([(2, 2)]) == 1, divisors_sum_odd([(2, 2)])
        assert divisors_sum_odd([(2, 1), (3, 1)]) == 4,  divisors_sum_odd([(2, 1), (3, 1)])
        assert divisors_sum_odd([(2, 5), (3, 2)]) == 13, divisors_sum_odd([(2, 5), (3, 2)])

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors_cond(n, lambda d: d&1 != 0)
            assert divisors_sum_odd(f) == natseq.sum(d), \
                   (n, divisors_sum_odd(f), natseq.sum(d), f, d)
        print('ok'); sys.stdout.flush()


        print('divisors_sum_even()...', end=''); sys.stdout.flush()
        assert divisors_sum_even([]) == 0,       divisors_sum_even([])
        assert divisors_sum_even([(2, 1)]) == 2, divisors_sum_even([(2, 1)])
        assert divisors_sum_even([(2, 2)]) == 6, divisors_sum_even([(2, 2)])
        assert divisors_sum_even([(2, 1), (3, 1)]) == 8,     divisors_sum_even([(2, 1), (3, 1)])
        assert divisors_sum_even([(2, 5), (3, 2)]) == 62*13, divisors_sum_even([(2, 5), (3, 2)])

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors_cond(n, lambda d: d&1 == 0)
            assert divisors_sum_even(f) == natseq.sum(d), \
                   (n, divisors_sum_even(f), natseq.sum(d), f, d)
            assert divisors_sum_even(f) + divisors_sum_odd(f) == divisors_sum(f), \
                   (n, d, divisors_sum_even(f), divisors_sum_odd(f), divisors_sum(f), f)
        print('ok'); sys.stdout.flush()


        print('divisors_sum_pow()...', end=''); sys.stdout.flush()
        assert divisors_sum_pow([], 0) == 1,       divisors_sum_pow([], 0)
        assert divisors_sum_pow([(2, 1)], 0) == 2, divisors_sum_pow([(2, 1)], 0)
        assert divisors_sum_pow([(2, 2)], 0) == 3, divisors_sum_pow([(2, 2)], 0)
        assert divisors_sum_pow([(2, 1), (3, 1)], 0) == 4,  divisors_sum_pow([(2, 1), (3, 1)], 0)
        assert divisors_sum_pow([(2, 5), (3, 2)], 0) == 18, divisors_sum_pow([(2, 5), (3, 2)], 0)

        assert divisors_sum_pow([], 1) == 1,       divisors_sum_pow([], 1)
        assert divisors_sum_pow([(2, 1)], 1) == 3, divisors_sum_pow([(2, 1)], 1)
        assert divisors_sum_pow([(2, 2)], 1) == 7, divisors_sum_pow([(2, 2)], 1)
        assert divisors_sum_pow([(2, 1), (3, 1)], 1) == 12,    divisors_sum_pow([(2, 1), (3, 1)], 1)
        assert divisors_sum_pow([(2, 5), (3, 2)], 1) == 63*13, divisors_sum_pow([(2, 5), (3, 2)], 1)

        assert divisors_sum_pow([], 2) == 1,                divisors_sum_pow([], 2)
        assert divisors_sum_pow([(2, 1)], 2) == 5,          divisors_sum_pow([(2, 1)], 2)
        assert divisors_sum_pow([(2, 2)], 2) == 21,         divisors_sum_pow([(2, 2)], 2)
        assert divisors_sum_pow([(2, 1), (3, 1)], 2) == 50, divisors_sum_pow([(2, 1), (3, 1)], 2)
        assert divisors_sum_pow([(2, 5), (3, 2)], 2) == 124215, \
               divisors_sum_pow([(2, 5), (3, 2)], 2)

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors(n)
            assert divisors_sum_pow(f, 0) == len(d), \
                   (n, divisors_sum_pow(f, 0), len(d), f, d)
            assert divisors_sum_pow(f, 1) == natseq.sum(d), \
                   (n, divisors_sum_pow(f, 1), natseq.sum(d), f, d)
            for k in range(10):
                assert divisors_sum_pow(f, k) == natseq.sum_pow(d, k), \
                       (n, k, divisors_sum_pow(f, k), natseq.sum_pow(d, k), f, d)
        print('ok'); sys.stdout.flush()


        print('feb_primaries()...', end=''); sys.stdout.flush()
        assert feb_primaries(0) == [], feb_primaries(0)
        assert feb_primaries(1) == [(2, 1)], feb_primaries(1)
        assert feb_primaries(2) == [(3, 1)], feb_primaries(2)
        assert feb_primaries(3) == [(2, 1), (3, 1)], feb_primaries(3)
        assert feb_primaries(4) == [(5, 1)], feb_primaries(4)
        assert feb_primaries(5) == [(2, 1), (5, 1)], feb_primaries(5)
        assert feb_primaries(6) == [(3, 1), (5, 1)], feb_primaries(6)
        assert feb_primaries(7) == [(2, 1), (3, 1), (5, 1)], feb_primaries(7)
        for i in range(100):
            assert feb_primaries(2**i) == [(nat32.PRIME_16_ARRAY[i], 1)], (i, feb_primaries(2**i))
            for j in range(100):
                if i != j:
                    assert feb_primaries(2**i + 2**j) == \
                           [(nat32.PRIME_16_ARRAY[min(i, j)], 1),
                            (nat32.PRIME_16_ARRAY[max(i, j)], 1)], \
                           (i, j, feb_primaries(2**i + 2**j))
        for n in range(1000):
            for p in feb_primaries(n):
                assert nat32.prime_is(p[0]), (n, p)
                assert p[1] == 1, (n, p)
            assert bef(feb_primaries(n)) == n , (n, bef(feb_primaries(n)), feb_primaries(n))
        print('ok'); sys.stdout.flush()


        print('gcd()...', end=''); sys.stdout.flush()
        assert gcd([], []) == [], gcd([], [])
        for m in range(1, 1000 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            f = primaries(m)
            assert gcd(f, []) == [], (m, gcd(f, []), f)
            assert gcd([], f) == [], (m, gcd([], f), f)
            assert gcd(f, f)  == f,  (m, gcd(f, f), f)
            for p in nat32.PRIME_16_ARRAY[:200]:
                if m%p == 0:
                    assert gcd(f, [(p, 1)]) == [(p, 1)], (m, p, gcd(f, [(p, 1)]), f)
                else:
                    assert gcd(f, [(p, 1)]) == [], (m, p, gcd(f, [(p, 1)]), f)
        for m in range(1, 100):
            m_f = primaries(m)
            for n in range(1, 100):
                n_f = primaries(n)
                d_f = primaries(fractions.gcd(m, n))
                assert gcd(m_f, n_f) == d_f, (m, n, gcd(m_f, n_f),
                                              m_f, n_f, fractions.gcd(m, n), d_f)
                assert gcd(n_f, m_f) == d_f, (m, n, gcd(n_f, m_f),
                                              m_f, n_f, fractions.gcd(m, n), d_f)
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('godelnumber()...', end=''); sys.stdout.flush()
        assert godelnumber([]) == 1, godelnumber([])
        assert godelnumber(()) == 1, godelnumber(())

        for a in range(10):
            assert godelnumber([a])   == 2**a, (a, godelnumber([a]), 2**a)
            assert godelnumber((a, )) == 2**a, (a, godelnumber((a, )), 2**a)
        for a in range(10):
            for b in range(10):
                assert godelnumber((a, b)) == 2**a * 3**b, (a, b, godelnumber((a, b)), 2**a * 3**b)
        p = 1
        for n in range(10):
            assert godelnumber((1, )*n) == p, (n, godelnumber((1, )*n), p)
            p *= nat32.PRIME_16_ARRAY[n]
            for a in range(10):
                assert godelnumber((0, )*n + (a, )) == nat32.PRIME_16_ARRAY[n]**a, \
                       (n, a, godelnumber((0, )*n + (a, )),
                        nat32.PRIME_16_ARRAY[n]**a, (0, )*n + (a, ))
        print('ok'); sys.stdout.flush()


        print('godelnumber_to_list()...', end=''); sys.stdout.flush()
        assert godelnumber_to_list(1) == [],        godelnumber_to_list(1)
        assert godelnumber_to_list(2) == [1],       godelnumber_to_list(2)
        assert godelnumber_to_list(3) == [0, 1],    godelnumber_to_list(3)
        assert godelnumber_to_list(4) == [2],       godelnumber_to_list(4)
        assert godelnumber_to_list(5) == [0, 0, 1], godelnumber_to_list(5)
        for n in range(1, 1000):
            for k in range(10):
                assert godelnumber(godelnumber_to_list(n) + [0]*k) == n, \
                       (n, k, godelnumber_to_list(n) + [0]*k,
                        godelnumber(godelnumber_to_list(n) + [0]*k))
        print('ok'); sys.stdout.flush()


        print('lcm()...', end=''); sys.stdout.flush()
        assert lcm([], []) == [], lcm([], [])
        for m in range(1, 1000 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            f = primaries(m)
            assert lcm(f, []) == f, (m, lcm(f, []), f)
            assert lcm([], f) == f, (m, lcm([], f), f)
            assert lcm(f, f)  == f, (m, lcm(f, f), f)
            for p in nat32.PRIME_16_ARRAY[:30]:
                if m%p == 0:
                    assert lcm(f, [(p, 1)]) == f, (m, p, lcm(f, [(p, 1)]), f)
                else:
                    assert lcm(f, [(p, 1)]) == primaries(m * p), (m, p, lcm(f, [(p, 1)]), f)
        for m in range(1, 100):
            m_f = primaries(m)
            for n in range(1, 100):
                n_f = primaries(n)
                d_f = primaries(nat32.lcm(m, n))
                assert lcm(m_f, n_f) == d_f, (m, n, lcm(m_f, n_f), m_f, n_f, nat32.lcm(m, n), d_f)
                assert lcm(n_f, m_f) == d_f, (m, n, lcm(n_f, m_f), m_f, n_f, nat32.lcm(m, n), d_f)
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('mobius()...', end=''); sys.stdout.flush()
        assert mobius([]) == 1,               mobius([])
        assert mobius([(2, 1)]) == -1,        mobius([(2, 1)])
        assert mobius([(3, 1)]) == -1,        mobius([(3, 1)])
        assert mobius([(2, 2)]) == 0,         mobius([(2, 2)])
        assert mobius([(5, 1)]) == -1,        mobius([(5, 1)])
        assert mobius([(2, 1), (5, 1)]) == 1, mobius([(2, 1), (5, 1)])
        for p in nat32.PRIME_16_ARRAY[2:]:
            assert mobius([(p, 1)]) == -1,                 (p, mobius([(p, 1)]))
            assert mobius([(p, 2)]) == 0,                  (p, mobius([(p, 2)]))
            assert mobius([(2, 1), (p, 1)]) == 1,          (p, mobius([(2, 1), (p, 1)]))
            assert mobius([(2, 1), (3, 1), (p, 1)]) == -1, (p, mobius([(2, 1), (3, 1), (p, 1)]))
        print('ok'); sys.stdout.flush()


        print('mul()...', end=''); sys.stdout.flush()
        assert mul([], []) == [], mul([], [])
        for n in range(1, 1000):
             f = primaries(m)
             assert mul(f, []) == f, (m, mul(f, []), f)
             assert mul([], f) == f, (m, mul([], f), f)
             assert mul(f, f)  == primaries(m*m), (m, mul(f, f), primaries(m*m))
        for m in range(1, 100):
            m_f = primaries(m)
            for n in range(1, 100):
                n_f = primaries(n)
                d_f = primaries(m*n)
                assert mul(m_f, n_f) == d_f, (m, n, mul(m_f, n_f), m_f, n_f, m*n, d_f)
                assert mul(n_f, m_f) == d_f, (m, n, mul(n_f, m_f), m_f, n_f, m*n, d_f)
        print('ok'); sys.stdout.flush()


        print('nb_in_integers_4sqr()...', end=''); sys.stdout.flush()
        assert nb_in_integers_4sqr([]) == 8,        nb_in_integers_4sqr([])
        assert nb_in_integers_4sqr([(2, 1)]) == 24, nb_in_integers_4sqr([(2, 1)])
        assert nb_in_integers_4sqr([(3, 1)]) == 32, nb_in_integers_4sqr([(3, 1)])
        assert nb_in_integers_4sqr([(2, 2)]) == 24, nb_in_integers_4sqr([(2, 2)])
        assert nb_in_integers_4sqr([(5, 1)]) == 48, nb_in_integers_4sqr([(5, 1)])
        for n in range(1, 1000):
            f = primaries(n)
            if n&1 == 0:
                assert nb_in_integers_4sqr(f) == 24 * divisors_sum(f[1:]), \
                       (n, nb_in_integers_4sqr(f), divisors_sum(f[1:]), f)
            else:
                assert nb_in_integers_4sqr(f) == 8 * divisors_sum(f), \
                       (n, nb_in_integers_4sqr(f), divisors_sum(f), f)
        print('ok'); sys.stdout.flush()


        print('nfp()...', end=''); sys.stdout.flush()
        assert nfp([], []) == [], nfp([], [])
        for n in range(1, 1000 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            f = primaries(m)
            assert nfp(f, []) == f,  (m, nfp(f, []), f)
            assert nfp([], f) == f,  (m, nfp([], f), f)
            assert nfp(f, f)  == [], (m, nfp(f, f), f)
            for p in nat32.PRIME_16_ARRAY[:50]:
                if m%p == 0:
                    assert nfp(f, [(p, 1)]) == primaries(m//p), (m, p, nfp(f, [(p, 1)]), f)
                else:
                    assert nfp(f, [(p, 1)]) == primaries(m*p), (m, p, nfp(f, [(p, 1)]), f)
        for m in range(1, 100):
            m_f = primaries(m)
            for n in range(1, 100):
                n_f = primaries(n)
                d_f = primaries(nat32.nfp(m, n))
                assert nfp(m_f, n_f) == d_f, (m, n, nfp(m_f, n_f), m_f, n_f, nat32.nfp(m, n), d_f)
                assert nfp(n_f, m_f) == d_f, (m, n, nfp(n_f, m_f), m_f, n_f, nat32.nfp(m, n), d_f)
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('nontotient()...', end=''); sys.stdout.flush()
        assert nontotient([]) == 0,                nontotient([])
        assert nontotient([(2, 1)]) == 1,          nontotient([(2, 1)])
        assert nontotient([(3, 1)]) == 1,          nontotient([(3, 1)])
        assert nontotient([(2, 2)]) == 2,          nontotient([(2, 2)])
        assert nontotient([(5, 1)]) == 1,          nontotient([(5, 1)])
        assert nontotient([(2, 1), (3, 1)]) == 4,  nontotient([(2, 1), (3, 1)])
        assert nontotient([(2, 3), (3, 1)]) == 16, nontotient([(2, 3), (3, 1)])
        for n in range(1, 500 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            assert nontotient(primaries(n)) == len(natural.nontotatives(n)), \
                   (n, nontotient(primaries(n)), len(natural.nontotatives(n)),
                    natural.nontotatives(n))
        for p in nat32.PRIME_16_ARRAY:
            assert nontotient([(p, 1)]) == 1, (p, nontotient([(p, 1)]))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('ope_exp()...', end=''); sys.stdout.flush()
        assert ope_exp([], [], lambda a, b: a + b) == [], ope_exp([], [], lambda a, b: a + b)
        for m in range(1, 200):
            m_f = primaries(m)
            for n in range(1, 200 if debug.assertspeed >= debug.ASSERT_NORMAL else 20):
                n_f = primaries(n)
                r = ope_exp(m_f, n_f, lambda a, b: a + b)
                assert r == mul(m_f, n_f), (m, n, r, mul(m_f, n_f))
                if m%n == 0:
                    r = ope_exp(m_f, n_f, lambda a, b: a - b)
                    assert r == primaries(m//n), (m, n, r, primaries(m//n))
                r = ope_exp(m_f, n_f, lambda a, b: min(a, b))
                assert r == gcd(m_f, n_f), (m, n, r, gcd(m_f, n_f))
                r = ope_exp(m_f, n_f, lambda a, b: max(a, b))
                assert r == lcm(m_f, n_f), (m, n, r, lcm(m_f, n_f))
                r = ope_exp(m_f, n_f, lambda a, b: abs(a - b))
                assert r == nfp(m_f, n_f), (m, n, r, nfp(m_f, n_f))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('primaries()...', end=''); sys.stdout.flush()
        assert primaries(1) == [],               primaries(1)
        assert primaries(2) == [(2, 1)],         primaries(2)
        assert primaries(3) == [(3, 1)],         primaries(3)
        assert primaries(4) == [(2, 2)],         primaries(4)
        assert primaries(5) == [(5, 1)],         primaries(5)
        assert primaries(6) == [(2, 1), (3, 1)], primaries(6)

        q = nat32.PRIME_16_ARRAY[100]
        for p in nat32.PRIME_16_ARRAY[:100]:
            for k in range(1, 10):
                assert primaries(p**k) == [(p, k)], (p, k, primaries(p**k))
                assert primaries(p**k * q**(k + 1)) == [(p, k), (q, k + 1)], \
                       (p, k, primaries(p**k * q**(k + 1)))

        for n in range(1, 1000):
            f = primaries(n)
            for p in f:
                assert nat32.prime_is(p[0]), (n, p)
                assert DSPython.natural_is(p[1]), (n, p)
                assert p[1] >= 1, (n, p[1])
        print('ok'); sys.stdout.flush()


        print('primaries_is()...', end=''); sys.stdout.flush()
        assert primaries_is([])
        assert primaries_is(())
        assert primaries_is([(2, 1)])
        assert not primaries_is([(2, 0)])
        assert not primaries_is([(1, 1)])
        assert primaries_is([(2, 1), (3, 1)])
        assert not primaries_is([(2, 1), (2, 1)])
        assert primaries_is([(2, 4), (3, 1)])
        assert primaries_is([(2, 24), (3, 19), (17, 3)])
        for n in range(1, 1000):
            assert primaries_is(primaries(n)), (n, primaries(n))
        print('ok'); sys.stdout.flush()


        print('primaries_prime_is()...', end=''); sys.stdout.flush()
        for p in nat32.PRIME_16_ARRAY[:1000]:
            f = primaries(p)
            assert primaries_prime_is(f), (p, f)
        for n in range(1, 100):
            f = primaries(n)
            assert primaries_prime_is(f) == nat32.prime_is(n), \
                   (n, primaries_prime_is(f), nat32.prime_is(n))
        print('ok'); sys.stdout.flush()


        print('primaries_str()...', end=''); sys.stdout.flush()
        assert primaries_str([]) == '1',         primaries_str([])
        assert primaries_str([(3, 1)]) == '3',   primaries_str([(3, 1)])
        assert primaries_str([(3, 2)]) == '3^2', primaries_str([(3, 2)])
        assert primaries_str([(3, 1), (5, 7)]) == '3.5^7',   primaries_str([(3, 1), (5, 7)])
        assert primaries_str([(3, 2), (5, 1)]) == '3^2.5',   primaries_str([(3, 2), (5, 1)])
        assert primaries_str([(3, 2), (5, 7)]) == '3^2.5^7', primaries_str([(3, 2), (5, 7)])

        assert primaries_str([(3, 2), (5, 7)], times=' x ', exp='**') == '3**2 x 5**7', \
               primaries_str([(3, 2), (5, 7)])
        print('ok'); sys.stdout.flush()


        print('primaries_to_n()...', end=''); sys.stdout.flush()
        assert primaries_to_n([]) == 1,                primaries([])
        assert primaries_to_n([(2, 1)]) == 2,          primaries([(2, 1)])
        assert primaries_to_n([(2, 2)]) == 4,          primaries([(2, 2)])
        assert primaries_to_n([(2, 3), (5, 1)]) == 40, primaries([(2, 3), (5, 1)])

        for n in range(1, 1000):
            assert primaries_to_n(primaries(n)) == n, \
                   (n, primaries_to_n(primaries(n)), primaries(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(2**16 - 50, 2**16 + 100):
                assert primaries_to_n(primaries(n)) == n, \
                       (n, primaries_to_n(primaries(n)), primaries(n))
            for n in range(2**24 - 5, 2**24 + 10):
                assert primaries_to_n(primaries(n)) == n, \
                       (n, primaries_to_n(primaries(n)), primaries(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('primaries_to_primes()...', end=''); sys.stdout.flush()
        assert primaries_to_primes([]) == [],           primaries_to_primes([])
        assert primaries_to_primes([(2, 1)]) == [2],    primaries_to_primes([(2, 1)])
        assert primaries_to_primes([(2, 2)]) == [2, 2], primaries_to_primes([(2, 2)])
        assert primaries_to_primes([(2, 2), (3, 1)]) == [2, 2, 3], \
               primaries_to_primes([(2, 2), (3, 1)])
        for n in range(1, 1000):
            assert primaries_to_primes(primaries(n)) == primes(n), \
                   (n, primaries_to_primes(primaries(n)), primes(n), primaries(n))
        print('ok'); sys.stdout.flush()


        print('primes()...', end=''); sys.stdout.flush()
        assert primes(1) == [],     primes(1)
        assert primes(2) == [2],    primes(2)
        assert primes(3) == [3],    primes(3)
        assert primes(4) == [2, 2], primes(4)
        assert primes(5) == [5],    primes(5)
        assert primes(6) == [2, 3], primes(6)

        q = nat32.PRIME_16_ARRAY[100]
        for p in nat32.PRIME_16_ARRAY[:100]:
            for k in range(1, 10):
                assert primes(p**k) == [p]*k, (p, k, primes(p**k))
                assert primes(p**k * q**(k + 1)) == [p]*k + [q]*(k + 1), \
                       (p, k, primes(p**k * q**(k + 1)))

        for n in range(1, 1000):
            f = primes(n)
            for p in f:
                assert nat32.prime_is(p), (n, p)

            fp = primaries(n)
            nb = 0
            for p in fp:
                nb += p[1]
            assert len(f) == nb, (n, len(f), nb)
        print('ok'); sys.stdout.flush()


        print('primes_is()...', end=''); sys.stdout.flush()
        assert primes_is([])
        assert primes_is(())
        assert primes_is([2])
        assert not primes_is([0])
        assert not primes_is([1])
        assert primes_is([2, 3])
        assert primes_is([2, 2])
        assert primes_is([2, 3])
        assert primes_is([2, 3, 3])
        assert not primes_is([2, 3, 3, 6])
        for n in range(1, 1000):
            assert primes_is(primes(n)), (n, primes(n))
        print('ok'); sys.stdout.flush()


        print('primes_str()...', end=''); sys.stdout.flush()
        assert primes_str([]) == '1',       primes_str([])
        assert primes_str([2]) == '2',      primes_str([2])
        assert primes_str([2, 2]) == '2.2', primes_str([2, 2])
        assert primes_str([2, 3]) == '2.3', primes_str([2, 3])

        assert primes_str([], times=' x ') == '1',         primes_str([], times=' x ')
        assert primes_str([2], times=' x ') == '2',        primes_str([2], times=' x ')
        assert primes_str([2, 2], times=' x ') == '2 x 2', primes_str([2, 2], times=' x ')
        assert primes_str([2, 3], times=' x ') == '2 x 3', primes_str([2, 3], times=' x ')
        print('ok'); sys.stdout.flush()


        print('primes_to_n()...', end=''); sys.stdout.flush()
        assert primes_to_n([]) == 1,            primaries([])
        assert primes_to_n([2]) == 2,           primaries([0])
        assert primes_to_n([2, 2]) == 4,        primaries([1])
        assert primes_to_n([2, 2, 2, 5]) == 40, primaries([2])

        for n in range(1, 1000):
            assert primes_to_n(primes(n)) == n, (n, primes_to_n(primes(n)), primes(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(2**16 - 50, 2**16 + 100):
                assert primes_to_n(primes(n)) == n, (n, primes_to_n(primes(n)), primes(n))
            for n in range(2**24 - 5, 2**24 + 10):
                assert primes_to_n(primes(n)) == n, (n, primes_to_n(primes(n)), primes(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('primes_to_primaries()...', end=''); sys.stdout.flush()
        assert primes_to_primaries([]) == [],                      primes_to_primaries([])
        assert primes_to_primaries([2]) == [(2, 1)],               primes_to_primaries([2])
        assert primes_to_primaries([2, 2]) == [(2, 2)],            primes_to_primaries([2, 2])
        assert primes_to_primaries([2, 2, 3]) == [(2, 2), (3, 1)], primes_to_primaries([2, 2, 3])
        for n in range(1, 1000):
            assert primes_to_primaries(primes(n)) == primaries(n), \
                   (n, primes_to_primaries(primes(n)), primaries(n), primes(n))
        print('ok'); sys.stdout.flush()


        print('properdivisors_sum()...', end=''); sys.stdout.flush()
        assert properdivisors_sum([]) == 0,         properdivisors_sum([])
        assert properdivisors_sum([(2, 1)]) == 1,   properdivisors_sum([(2, 1)])
        assert properdivisors_sum([(2, 2)]) == 3,   properdivisors_sum([(2, 2)])
        assert properdivisors_sum([(2, 1), (3, 1)]) == 6, \
               properdivisors_sum([(2, 1), (3, 1)])
        assert properdivisors_sum([(2, 5), (3, 2)]) == 63*13 - 32*9, \
               properdivisors_sum([(2, 5), (3, 2)])

        for n in range(1, 1000):
            f = primaries(n)
            d = natural.divisors(n)[:-1]
            assert properdivisors_sum(f) == natseq.sum(d), \
                   (n, properdivisors_sum(f), natseq.sum(d), f, d)
        print('ok'); sys.stdout.flush()


        print('rad()...', end=''); sys.stdout.flush()
        assert rad([]) == [], rad([])
        for p in nat32.PRIME_16_ARRAY[:100]:
            for k in range(1, 10):
                assert rad([(p, k)]) == [(p, 1)], rad([(p, k)])
        for n in range(1, 100):
            f = primaries(n)
            r = rad(f)
            assert primaries_is(r), (n, r)
            assert len(f) == len(r), (n, len(f), len(r), f, r)
            assert squarefree_is(r), (n, r, f)
        print('ok'); sys.stdout.flush()


        print('squarefree_is()...', end=''); sys.stdout.flush()
        assert squarefree_is([])
        assert squarefree_is([(2, 1)])
        assert squarefree_is([(3, 1)])
        assert squarefree_is([(5, 1)])
        assert not squarefree_is([(2, 2)])
        assert not squarefree_is([(2, 33)])
        assert not squarefree_is([(5, 2)])
        assert not squarefree_is([(5, 33)])
        for n in range(100):
            assert squarefree_is(feb_primaries(n)), (n, feb_primaries(n))
            assert squarefree_is([(nat32.PRIME_16_ARRAY[n], 1)]), (n, nat32.PRIME_16_ARRAY[n])
            for k in range(2, 10):
                assert not squarefree_is([(nat32.PRIME_16_ARRAY[n], k)]), \
                       (n, k, nat32.PRIME_16_ARRAY[n])
        print('ok'); sys.stdout.flush()


        print('totient()...', end=''); sys.stdout.flush()
        assert totient([]) == 1,               totient([])
        assert totient([(2, 1)]) == 1,         totient([(2, 1)])
        assert totient([(3, 1)]) == 2,         totient([(3, 1)])
        assert totient([(2, 2)]) == 2,         totient([(2, 2)])
        assert totient([(5, 1)]) == 4,         totient([(5, 1)])
        assert totient([(2, 1), (3, 1)]) == 2, totient([(2, 1), (3, 1)])
        assert totient([(2, 3), (3, 1)]) == 8, totient([(2, 3), (3, 1)])
        for n in range(1, 500 if debug.assertspeed >= debug.ASSERT_NORMAL else 100):
            f = primaries(n)
            t = totient(f)
            assert t== len(natural.totatives(n)), \
                   (n, t, len(natural.totatives(n)), natural.totatives(n), f)
            assert t + nontotient(f) == n, (n, t, nontotient(n), f)
        for p in nat32.PRIME_16_ARRAY:
            for k in range(1, 10):
                assert totient([(p, k)]) == (p - 1) * p**(k - 1), (p, k, totient([(p, k)]))
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
