#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.combinator Combinateurs
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Theo" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Theo</tt></a>\endhtmlonly\n
# et \htmlonly <a href="http://fr.wikipedia.org/wiki/Logique_combinatoire" target="_blank">
#   <tt>http://fr.wikipedia.org/wiki/Logique_combinatoire</tt></a>\endhtmlonly
# pour une prsentation de la logique combinatoire.

##\file
# Combinateurs
#
# Cf. \htmlonly <a href="http://www.opimedia.be/Bruno_Marchal/index.htm#Theo" target="_blank">
#   <tt>http://www.opimedia.be/Bruno_Marchal/index.htm#Theo</tt></a>\endhtmlonly\n
# et \htmlonly <a href="http://fr.wikipedia.org/wiki/Logique_combinatoire" target="_blank">
#   <tt>http://fr.wikipedia.org/wiki/Logique_combinatoire</tt></a>\endhtmlonly
# pour une prsentation de la logique combinatoire.

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 1er juillet 2007
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'combinator --- 2010 April 12'

import collections, sys, types

import DSPython



# ###########
# Fonctions #
#############
## Fonction pour la dynamique du combinateur atomique B : Bxyz... --> x(yz)...
def func_B(a, s, nb):
    """Fonction pour la dynamique du combinateur atomique B : Bxyz... --> x(yz)...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si len(s) => 3 et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(x_1, x_3, Combinator(x_2, x_3), x_4, ... , x_k)
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ..., x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = ..."""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return ((Combinator(s[0],
                       Combinator(s[1], s[2]),
                       *s[3:]),
             1) if (len(s) >= 3) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour la dynamique du combinateur atomique C : Cxyz... --> xzy...
def func_C(a, s, nb):
    """Fonction pour la dynamique du combinateur atomique C : Cxyz... --> xzy...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si len(s) => 3 et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(x_1, x_3, Combinator(x_2, x_3), x_4, ... , x_k)
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ..., x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = ..."""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return ((Combinator(s[0], s[2], s[1], *s[3:]),
             1) if (len(s) >= 3) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour une "dynamique constante" : a... --> a...
def func_const(a, s, nb):
    """Fonction pour une "dynamique constante" : a... --> a...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    renvoie (Combinator(a, x_1, x_2, x_3, x_4, ..., x_k),
             0) quelque soit nb
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = 1"""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return (Combinator(a, *s),
            0)


## Fonction pour la dynamique du combinateur atomique I : Ix... --> x...
def func_I(a, s, nb):
    """Fonction pour la dynamique du combinateur atomique I : Ix... --> x...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si len(s) => 1 et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(x_1, x_2, x_3, x_4, ..., x_k),
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ... ,x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = ..."""
    assert isinstance(a, Atom), (type(a), a)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert isinstance(s, collections.Iterable), (type(s), s)

    return ((Combinator(*s),
             1) if (len(s) >= 1) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour la dynamique du combinateur atomique K : Kxy... --> x...
def func_K(a, s, nb):
    """Fonction pour la dynamique du combinateur atomique K : Kxy... --> x...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si len(s) => 2 et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(x_1, x_3, x_4, ..., x_k),
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ... ,x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = ..."""
    assert isinstance(a, Atom), (type(a), a)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert isinstance(s, collections.Iterable), (type(s), s)

    return ((Combinator(s[0], *s[2:]),
             1) if (len(s) >= 2) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour la dynamique du combinateur atomique S : Sxyz... --> xz(yz)...
def func_S(a, s, nb):
    """Fonction pour la dynamique du combinateur atomique S : Sxyz... --> xz(yz)...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si len(s) => 3 et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(x_1, x_3, Combinator(x_2, x_3), x_4, ... , x_k)
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ..., x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = ..."""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return ((Combinator(s[0],
                        s[2],
                        Combinator(s[1], s[2]),
                        *s[3:]),
             1) if (len(s) >= 3) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour la dynamique du combinateur atomique W : Wxy... --> xyy...
def func_W(a, s, nb):
    """Fonction pour la dynamique du combinateur atomique W : Wxy... --> xyy...
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si len(s) => 2 et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(x_1, x_2, x_2, x_3, x_4, ..., x_k),
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ... ,x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = ..."""
    assert isinstance(a, Atom), (type(a), a)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert isinstance(s, collections.Iterable), (type(s), s)

    return ((Combinator(s[0], s[1], s[1], *s[2:]),
             1) if (len(s) >= 2) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour le remplacement de la "mta-variable" x
def func_x(a, s, nb):
    """Fonction pour le remplacement de la "mta-variable" x
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si var_x != None et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(var_x, x_1, x_2, x_3, x_4, ..., x_k),
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ... ,x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = 1"""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return ((Combinator(var_x, *s),
             1) if (var_x != None) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour le remplacement de la "mta-variable" y
def func_y(a, s, nb):
    """Fonction pour le remplacement de la "mta-variable" y
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si var_y != None et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(var_y, x_1, x_2, x_3, x_4, ..., x_k),
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ... ,x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = 1"""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return ((Combinator(var_y, *s),
             1) if (var_y != None) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))


## Fonction pour le remplacement de la "mta-variable" z
def func_z(a, s, nb):
    """Fonction pour le remplacement de la "mta-variable" z
    Pour s = [x_1, x_2, x_3, x_4, ..., x_k]
    si var_z != None et ((nb == None) ou (nb > 0))
      alors renvoie (Combinator(var_z, x_1, x_2, x_3, x_4, ..., x_k),
                     1)
      sinon renvoie (Combinator(a, x_1, x_2, x_3, x_4, ... ,x_k),
                     0)
    (Seul a est valu, une seule fois, et pas les x_i)

    Pre: a: Atom
         s: Iterable de Combinator
         nb: None ou naturel

    Result: (Combinator, naturel)

    O() = 1"""
    assert isinstance(a, Atom), (type(a), a)
    assert isinstance(s, collections.Iterable), (type(s), s)
    if __debug__:
        for x in s:
            assert isinstance(x, Combinator), (type(x), x, s)
    assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

    return ((Combinator(var_z, *s),
             1) if (var_z != None) and ((nb == None) or (nb > 0))
            else (Combinator(a, *s),
                  0))



# #########
# Classes #
###########
## "Atome" pour les combinateurs
class Atom(collections.Hashable, collections.Callable):
    """"Atome" pour les combinateurs"""

    ## Initialise l'atome
    def __init__(self, s, f=func_const):
        """Initialise l'atome

        Pre: s: string non vide compose de caractres alphanumriques ou '_',
             f: fonction : (Atom, Combinator, None ou naturel a) -> (Combinator, naturel b)
                  tel a == None ou a >= b

        O() = 1"""
        assert isinstance(s, str), (type(s), s)
        assert s != '', s
        if __debug__:
            for c in s:
                assert c.isalnum() or (c == '_'), (c, s)
        assert isinstance(f, types.FunctionType), type(f)

        ## Fonction dcrivant la dynamique de l'atome
        self._f = f

        ## String reprsentant l'atome
        self._s = s


    ## Renvoie (l'valuation du combinateur atomique suivi des combinateurs de args,
    #           nombre d'tapes effectues)
    def __call__(self, *args, **keywords):
        """Renvoie (l'valuation du combinateur atomique suivi des combinateurs de args,
                 nombre d'tapes effectues)
        En fait renvoie self.f()(self, args, nb) rpte jusqu' puis nb
        Si nb == None alors value tant que le combinateur atomique du dbut ragit
        (! certains combinateurs peuvent ne jamais s'arrter)
        sinon value au plus nb tapes

        Pre: args: Iterable de Combinator
             keywords: nb=None ou naturel

        Result: (Combinator, naturel)

        O() = ..."""
        assert isinstance(args, collections.Iterable), (type(args), args)
        if __debug__:
            for x in args:
                assert isinstance(x, Combinator), (type(x), x, args)
        assert len(keywords) <= 1, (len(keywords), keywords)
        assert (len(keywords) == 0) or ('nb' in keywords), keywords

        nb = (keywords['nb'] if 'nb' in keywords
              else None)

        assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

        x, nb_eval = self.f()(self, args, nb)
        if nb_eval == 0:  # L'atome n'a rien fait
            return (x, 0)
        if nb != None:
            assert nb >= nb_eval, (nb, nb_eval)

            nb -= nb_eval

        while (nb == None) or (nb > 0):
            x, nb_tmp = x[0].f()(x[0], x[1:], nb=nb)
            if nb_tmp == 0:  # L'atome n'a rien fait
                return (x, nb_eval)
            else:            # L'atome a ragit
                if nb != None:
                    assert nb >= nb_tmp, (nb, nb_tmp)

                    nb -= nb_tmp
                nb_eval += nb_tmp
        return (x, nb_eval)


    ## cmp(self, other)
    def __cmp__(self, other):
        """Renvoie
        <0 si la reprsentation (string) de self < la reprsentation (string) de other,
        0 si la reprsentation (string) de self == la reprsentation (string) de other,
        >0 si la reprsentation (string) de self > la reprsentation (string) de other

        Result: Integral

        Exception: TypeError si other n'est pas un Atom

        O() = ..."""
        if not isinstance(other, Atom):
            raise TypeError

        return (cmp(self._s, other._s) if sys.version_info[0] < 3
                else (self._s > other._s) - (self._s < other._s))


    ## self == other ?
    def __eq__(self, other):
        """Renvoie True si other est un Atom qui a mme reprsentation (string) que self,
        False sinon
        (Ne se proccupe pas des fonctions associes aux dynamiques des atomes)

        Result: bool

        O() = ..."""
        return isinstance(other, Atom) and (self._s == other._s)


    ## self >= other ?
    def __ge__(self, other):
        """Renvoie True si la reprsentation (string) de self >= la reprsentation (string) de other
        False sinon

        Result: bool

        Exception: TypeError si other n'est pas un Atom

        O() = ..."""
        if not isinstance(other, Atom):
            raise TypeError

        return self._s >= other._s


    ## self > other ?
    def __gt__(self, other):
        """Renvoie True si la reprsentation (string) de self > la reprsentation (string) de other
        False sinon

        Result: bool

        Exception: TypeError si other n'est pas un Atom

        O() = ..."""
        if not isinstance(other, Atom):
            raise TypeError

        return self._s > other._s


    ## Hashcode de self
    def __hash__(self):
        """Renvoie le hashcode de self

        Result: naturel

        O() = ..."""
        return hash(self._s)


    ## self <= other ?
    def __le__(self, other):
        """Renvoie True si la reprsentation (string) de self <= la reprsentation (string) de other
        False sinon

        Result: bool

        Exception: TypeError si other n'est pas un Atom

        O() = ..."""
        if not isinstance(other, Atom):
            raise TypeError

        return self._s <= other._s


    ## self < other ?
    def __lt__(self, other):
        """Renvoie True si la reprsentation (string) de self < la reprsentation (string) de other
        False sinon

        Result: bool

        Exception: TypeError si other n'est pas un Atom

        O() = ..."""
        if not isinstance(other, Atom):
            raise TypeError

        return self._s < other._s


    ## self != other ?
    def __ne__(self, other):
        """Renvoie True si other n'est pas un Atom
        ou n'a pas la mme reprsentation (string) que self,
        False sinon
        (Ne se proccupe pas des fonctions associes aux dynamiques des atomes)

        Result: bool

        O() = ..."""
        return not isinstance(other, Atom) or (self._s != other._s)


    ## Renvoie l'atome sous forme de string "Atom('...')"
    def __repr__(self):
        """Renvoie l'atome sous forme de string "Atom('...')"

        Result: string "Atom('...')"

        O() = ..."""
        return ''.join(("Atom('", self._s, "')"))


    ## Renvoie l'atome sous forme de string
    def __str__(self):
        """Renvoie l'atome sous forme de string

        Result: string

        O() = ..."""
        return self._s


    ## Renvoie la fonction correspondant  la dynamique de l'atome
    def f(self):
        """Renvoie la fonction correspondant  la dynamique de l'atome

        Result: fonction : (Atom, Combinator, None ou naturel) -> (Combinator, naturel)

        O() = 1"""
        return self._f


    ## Renvoie le nombre d'argument ncessaires pour dclencher l'atome
    def nb_args(self, max=100):
        """Renvoie le nombre d'argument ncessaires pour dclencher l'atome
        Si max==None alors essaie avec de plus en plus d'arguments,
                           sans s'arrter jusqu' ce que l'atome dclenche
                           (ou sans jamais s'arrter si l'atome ne dclenche jamais)
        Si max est un naturel alors essaie avec au plus max arguments.
                                    Si l'atome n'a pas dclench alors renvoie None

        Pre: max: None ou naturel

        Result: naturel ou None

        O() = 1"""
        assert (max == None) or DSPython.natural_is(max), (type(max), max)

        nb = 0
        l = []
        while (max == None) or (max >= nb):
            if self(*l, nb=1)[1] > 0:
                return nb
            l.append(_COMB_CONST)
            nb += 1
        return None



## Combinateur
# (Sequence non vide dont le premier lment est un Atom et les suivants des Combinator)
class Combinator(collections.Sequence, collections.Hashable, collections.Callable):
    """Combinateur
    (Sequence non vide dont le premier lment est un Atom et les suivants des Combinator)"""

    ## Initialise le Combinator
    def __init__(self, *seq):
        """Initialise le Combinator

        Pre: seq: Sequence non vide de Combinator, Atom,
                  bool, naturel (selon le systme spcifi par natural_sys)
                  ou string reprsentant un combinateur

        Exception: TypeError si l contient des string incorrectes

        O() = ..."""
        assert isinstance(seq, collections.Sequence), (type(seq), seq)
        assert len(seq) > 0

        # Premier lment
        x = seq[0]
        if not (isinstance(x, Combinator) or isinstance(x, Atom)):
            if isinstance(x, bool):
                x = (K if x
                     else KI)
            elif DSPython.natural_is(x):
                x = Combinator.n_to(x)
            elif isinstance(x, str):
                x = Combinator.str_to(x)
                if x == None:
                    raise TypeError

            assert isinstance(x, Combinator) or isinstance(x, Atom), (type(x), x, seq)

        if isinstance(x, Combinator) and x.atomic_is():
            x = x[0]
        l = ([x] if isinstance(x, Atom)
             else list(x))

        # lments suivants
        for x in seq[1:]:
            if isinstance(x, bool):
                x = (K if x
                     else KI)
            elif isinstance(x, str):
                x = Combinator.str_to(x)
                if x == None:
                    raise TypeError

            assert isinstance(x, Combinator) or isinstance(x, Atom), (type(x), x, seq)

            l.append(Combinator(x) if isinstance(x, Atom)
                     else x)

        self._seq = tuple(l)


    ## self & other
    def __and__(self, other):
        """self & other (combinateur boolen et)
        En fait renvoie Combinator(Band, self, other)

        Pre: other: Combinator ou bool

        Result: Combinator

        Exception: TypeError si other n'est pas un Combinator ou un bool

        O() = ..."""
        if not (isinstance(other, Combinator) or isinstance(other, bool)):
            raise TypeError

        return Combinator(Band, self, other)


    ## Renvoie (le combinateur valu, le nombre d'tapes ralises)
    def __call__(self, nb=None):
        """Renvoie (le combinateur valu,
                    le nombre d'tapes ralises)
        Si nb == None alors tente d'valuer jusqu'au bout
        (! certains combinateurs peuvent ne jamais s'arrter)
        sinon value au plus nb tapes

        Pre: nb: None ou naturel

        Result: (Combinator, naturel)

        O() = ..."""
        assert (nb == None) or DSPython.natural_is(nb), (type(nb), nb)

        x, nb_eval = self[0](*self[1:], nb=nb)  # valuation du combinateur atomique du dbut
        if nb != None:
            assert nb >= nb_eval, (nb, nb_eval)

            nb -= nb_eval
            if nb <= 0:
                return (x, nb_eval)

        # Parcours de la liste de Combinator  partir du deuxime lment
        # (le premier est un Atom qui ne ragit plus)
        l = [x[0]]
        for i in range(1, len(x)):
            assert isinstance(x[i], Combinator), (type(x[i]), x[i])

            t, nb_tmp = x[i](nb=nb)  # valuation de l'lment intrieur x[i]
            l.append(t)
            nb_eval += nb_tmp
            if nb != None:
                assert nb >= nb_tmp, (nb, nb_tmp)

                nb -= nb_tmp
                if nb <= 0:
                    i += 1
                    return ((Combinator(*(tuple(l) + x[i:])), nb_eval) if len(x) > i
                            else (Combinator(*l), nb_eval))
        return (Combinator(*l), nb_eval)


    ## cmp(self, other)
    def __cmp__(self, other):
        """Renvoie <0 si self < other
                    0 si self == other
                   >0 si self > other
        Si len(self) < len(other) alors renvoie <0
        Si len(self) > len(other) alors renvoie >0
        Si len(self) == len(other) alors c'est les premiers lments diffrents qui dterminent

        Result: Integral

        Exception: TypeError si other n'est pas un Combinator

        O() = ..."""
        if not isinstance(other, Combinator):
            raise TypeError

        i = cmp(len(self), len(other))
        return (i if i != 0
                else cmp(self._seq, other._seq))


    ## item in self ?
    def __contains__(self, item):
        """Renvoie True si le combinateur self contient le combinateur atomique item,
        False sinon

        Result: bool

        Exception: TypeError si item n'est pas un Atom ou un Combinator

        O() = ..."""
        if not (isinstance(item, Atom) or isinstance(item, Combinator)):
            raise TypeError
        if isinstance(item, Combinator) and (not item.atomic_is()):
            return NotImplemented

        if isinstance(item, Combinator):
            item = item[0]

        if self[0] == item:
            return True
        else:
            for x in self[1:]:
                if item in x:
                    return True
            return False


    ## self == other ?
    def __eq__(self, other):
        """Renvoie True si other est un Combinator gal  self,
        False sinon
        (Ne se proccupe pas de l'ventuelle quivalence (thorique) entre les combinateurs)

        Result: bool

        O() = ..."""
        return (isinstance(other, Combinator)
                and (self.atomic_is() == other.atomic_is())
                and (self[0] == other[0])
                and (self.atomic_is()
                     or (self[1:] == other[1:])))


    ## self[key]
    def __getitem__(self, key):
        """self[key]

        O() = ..."""
        return self._seq.__getitem__(key)


    ## Hashcode de self
    def __hash__(self):
        """Renvoie le hashcode de self

        Result: naturel

        O() = ..."""
        return hash(self._seq)


    ## Nombre naturel correspondant  self
    def __int__(self):
        """Nombre naturel correspondant  self
        (Selon le systme spcifi par natural_sys : BARENDREGT ou CHURCH)

        Pre: self: Combinator correspondant  un nombre naturel dans le systme spcifi

        Result: naturel

        O() = ..."""
        assert (natural_sys == BARENDREGT) or (natural_sys == CHURCH), natural_sys

        if natural_sys == BARENDREGT:
            x = I
            succ = NBsucc
        else:
            x = KI
            succ = NCsucc
        n = 0
        while self != x:
            n += 1
            x = Combinator(succ, x)

            assert len(self) >= len(x), (len(self), len(x), self, x)# sinon prcondition non remplie

        return n


    ## Renvoie la longueur du combinateur
    def __len__(self):
        """Renvoie la longueur du combinateur

        Result: naturel > 0

        O() = ..."""
        return len(self._seq)


    ## self != other ?
    def __ne__(self, other):
        """Renvoie True si other n'est pas un Combinator
        ou n'est pas la mme Combinator que self,
        False sinon
        (Ne se proccupe pas de l'ventuelle quivalence (thorique) entre les combinateurs)
        O() = ..."""
        return not self.__eq__(other)


    ## self == K ?
    def __nonzero__(self):
        """Renvoie True si self == K,
        False sinon
        (Ne se proccupe pas de l'ventuelle quivalence (thorique) entre les combinateurs)

        Result: bool

        O() = ..."""
        return self == K


    ## self | other
    def __or__(self, other):
        """self | other (combinateur boolen ou inclusif)
        En fait renvoie Combinator(Bor, self, other)

        Pre: other: Combinator ou bool

        Result: Combinator

        Exception: TypeError si other n'est pas un Combinator ou un bool

        O() = ..."""
        if not (isinstance(other, Combinator) or isinstance(other, bool)):
            raise TypeError

        return Combinator(Bor, self, other)


    ## Renvoie le combinateur sous forme de string "Combinator('...')"
    def __repr__(self):
        """Renvoie le combinateur sous forme de string "Combinator('...')"

        Result: string "Combinator('...')"

        O() = 1"""
        return ''.join(("Combinator('", str(self), "')"))


    ## Renvoie le combinateur sous forme de string
    def __str__(self, allparent=False, left='(', right=')', space=' '):
        """Renvoie le combinateur sous forme de string.
        Si allparent alors avec toutes les parenthses,
        sinon vite les parenthses superflues.
        left reprsente la parenthse gauche et right la droite.
        Les "morceaux" du combinateur sont spars par space

        Pre: allparent: valeur boolenne
             left: string
             right: string
             space: string

        Result: string

        O() = ..."""
        assert isinstance(left, str), (type(left), left)
        assert isinstance(right, str), (type(right), right)
        assert isinstance(space, str), (type(space), space)
        assert isinstance(self[0], Atom)

        if self.atomic_is():  # Atom
            return str(self[0])
        else:                 # (Combinator, Combinator)
            l = [left*(len(self) - 1) + str(self[0]) if allparent
                 else str(self[0])]
            for x in self[1:]:
                assert isinstance(x, Combinator)

                s = x.__str__(allparent=allparent, left=left, right=right, space=space)
                if (not allparent) and (not x.atomic_is()):
                    s = left + s
                if allparent or (not x.atomic_is()):
                    s += right
                l.append(''.join(s))
            return space.join(l)


    ## self ^ other
    def __xor__(self, other):
        """self ^ other (combinateur boolen ou exclusif)
        En fait renvoie Combinator(Bxor, self, other)

        Pre: other: Combinator ou bool

        Result: Combinator

        Exception: TypeError si other n'est pas un Combinator ou un bool

        O() = ..."""
        if not (isinstance(other, Combinator) or isinstance(other, bool)):
            raise TypeError

        return NotImplemented


    ## Renvoie True si le combinateur est atomique, False sinon
    def atomic_is(self):
        """Renvoie True si le combinateur est atomique,
        False sinon

        Result: bool

        O() = ..."""
        assert isinstance(self[0], Atom), (type(self[0]), self[0])

        return len(self) == 1


    ## Non self
    def bnot(self):
        """Non self (combinateur ngation boolenne)
        En fait renvoie Combinator(Bnot, self)

        Result: Combinator

        O() = ..."""
        return Combinator(Bnot, self)


    ## Renvoie le combinateur dans lequel chaque Atom atom est remplac par new
    def change_atom(self, atom, new):
        """Renvoie le combinateur dans lequel chaque Atom atom est remplac par new

        Pre: atom: Atom
             new: Combinator ou Atom

        Result: Combinator

        O() = ..."""
        assert isinstance(atom, Atom), type(atom)
        assert isinstance(new, Combinator) or isinstance(new, Atom), type(new)

        l = [new if self[0] == atom
             else self[0]]
        for x in self[1:]:
            l.append(x.change_atom(atom, new))
        return Combinator(*l)


    ## Renvoie la profondeur du combinateur
    def depth(self):
        """Renvoie la profondeur du combinateur
        == (profondeur de l'arbre binaire correspondant) - 1.
        depth(combinateur atomique) == 0
        depth(combinateur (a b)) == max(depth(a), depth(b)) + 1

        Result: naturel

        O() = ..."""
        n = 0
        for x in self[1:]:
            n = max(n, x.depth()) + 1
        return n


    ## Renvoie le combinateur correspondant au nombre naturel n
    # selon le systme spcifi par natural_sys
    @staticmethod
    def n_to(n):
        """ Renvoie le combinateur correspondant au nombre naturel n
        (Selon le systme spcifi par natural_sys : BARENDREGT ou CHURCH)

        Pre: n: naturel

        Result: Combinator

        O() = ..."""
        assert (natural_sys == BARENDREGT) or (natural_sys == CHURCH), natural_sys
        assert DSPython.natural_is(n), type(n)

        return (Combinator.n_to_Barendregt(n) if natural_sys == BARENDREGT
                else Combinator.n_to_Church(n))


    ## Renvoie le combinateur correspondant au nombre naturel n de Barendregt
    @staticmethod
    def n_to_Barendregt(n):
        """Renvoie le combinateur correspondant au nombre naturel n de Barendregt

        Pre: n: naturel

        Result: Combinator

        O() = ..."""
        assert DSPython.natural_is(n), type(n)

        c = I
        for i in range(n):
            c = Combinator(NBsucc, c)
        return c


    ## Renvoie le combinateur correspondant au nombre naturel n de Church
    @staticmethod
    def n_to_Church(n):
        """Renvoie le combinateur correspondant au nombre naturel n de Church

        Pre: n: naturel

        Result: Combinator

        O() = ..."""
        assert DSPython.natural_is(n), type(n)

        c = KI
        for i in range(n):
            c = Combinator(NCsucc, c)
        return c


    ## Renvoie le nombre de combinateurs atomiques composant le combinateur (sa taille)
    def nb_atom(self):
        """Renvoie le nombre de combinateurs atomiques composant le combinateur (sa taille)
        == nombre de feuilles de l'arbre binaire correspondant.
        nb_atom(combinateur atomique) == 1
        nb_atom(combinateur (a b)) == nb_atom(a) + nb_atom(b)

        Result: naturel

        O() = ..."""
        nb = 1
        for x in self[1:]:
            nb += x.nb_atom()
        return nb


    ## Combinateur stable
    def stable_is(self):
        """Renvoie True si le combinateur est stable (il ne dclenche plus),
        False sinon

        Result: bool

        O() = ..."""
        return self(nb=1)[1] == 0


    ## Renvoie le combinateur reprsent par le string s
    @staticmethod
    def str_to(s):
        """Renvoie le combinateur reprsent par le string s
        (Si s n'est pas une rprsentation correcte alors renvoie None)
        Un string correcte est un string non vide
        comprenant des string appartenant  str_combs et des '(' et ')',
        le tout bien agenc

        Pre: s: string

        Result: Combinator ou None

        O() = ..."""
        assert isinstance(s, str), (type(s), s)

        if s == '':
            raise None

        l = []  # liste des Combinator dj traits
        i = 0  # position courante dans le string s
        k = 0  # position dans le string s du dpart de la sous-chane courante
        while i < len(s):
            if (s[i] == ' ') or (s[i] == '('):            # caractre de sparation ' ' ou '('
                if i > k:  # si il y a quelque chose  ajout avant de passer au '(...)'
                    if s[k:i] in str_combs:
                        x = str_combs[s[k:i]]

                        assert x != None
                    else:
                        return None
                    l.append(Combinator(x))
                k = i + 1

                if s[i] == '(':  # ouverture d'une '(...)'
                    nb = 1
                    while k < len(s):  # parcourt jusqu' trouv le ')' qui correspond  cet '('
                        if s[k] == '(':
                            nb += 1
                        elif s[k] == ')':
                            nb -= 1
                            if nb == 0:
                                break
                        k += 1
                    if nb > 0:  # pas trouv le ')' correspondant
                        return None

                    comb = Combinator.str_to(s[i + 1:k])  # convertit la sous-chane '(...)'
                    if comb != None:
                        l.append(comb)
                    i = k
                    k += 1
            elif (s[i] != '_') and (not s[i].isalnum()):  # caractre non valide (ou mal plac)
                return None
            i += 1

        if len(s) > k:
            if s[k:] in str_combs:
                x = str_combs[s[k:]]

                assert x != None
            else:
                return None
            l.append(Combinator(x))

        return (Combinator(*l) if l != [] else
                None)



# ############
# Constantes #
##############
## Constante pour utiliser les oprations sur les nombres naturels de Barendregt
BARENDREGT = 1

## Constante pour utiliser les oprations sur les nombres naturels de Church
CHURCH = 2


## Constante d'un atome de "dynamique constante", utilise en interne
_ATOM_CONST = Atom('c')


## Atome K   (constante,  ne pas modifier)
Atom_K = Atom('K', func_K)

## Atome S   (constante,  ne pas modifier)
Atom_S = Atom('S', func_S)


## Atome utilis par la "mta-variable" x   (constante,  ne pas modifier)
Atom_Vx = Atom('x', func_x)

## Atome utilis par la "mta-variable" y   (constante,  ne pas modifier)
Atom_Vy = Atom('y', func_y)

## Atome utilis par la "mta-variable" z   (constante,  ne pas modifier)
Atom_Vz = Atom('z', func_z)



# Combinateurs atomiques

## Constante d'un combinateur de "dynamique constante", utilise en interne
_COMB_CONST = Combinator(_ATOM_CONST)


## Combinateur K = combinateur vrai   (constante,  ne pas modifier)
K = Combinator(Atom_K)
assert isinstance(K, Combinator)
assert len(K) == 1, (len(K), K)
assert K.atomic_is()
assert isinstance(K[0], Atom)


## Combinateur S   (constante,  ne pas modifier)
S = Combinator(Atom_S)
assert isinstance(S, Combinator)
assert len(S) == 1, (len(S), S)
assert S.atomic_is()
assert isinstance(S[0], Atom)



# Combinateurs standards

## Combinateur KK   (constante,  ne pas modifier)
KK = Combinator(K, K)
assert isinstance(KK, Combinator)
assert len(KK) == 2, (len(KK), KK)
assert not KK.atomic_is()
assert isinstance(KK[0], Atom)
assert isinstance(KK[1], Combinator)


## Combinateur KS   (constante,  ne pas modifier)
KS = Combinator(K, S)
assert isinstance(KS, Combinator)
assert len(KS) == 2, (len(KS), KS)
assert not KS.atomic_is()
assert isinstance(KS[0], Atom)
assert isinstance(KS[1], Combinator)


## Combinateur SS   (constante,  ne pas modifier)
SS = Combinator(S, S)


## Combinateur SK   (constante,  ne pas modifier)
SK = Combinator(S, K)


## Combinateur B = S(KS)K = combinateur produit pour les nombres naturels de Church
# (constante,  ne pas modifier)
B = Combinator(S, KS, K)


## Combinateur C = S[S(KB)S](KK)   (constante,  ne pas modifier)
C = Combinator(S,
               Combinator(S,
                          Combinator(K, B),
                          S),
               KK)

## Combinateur I = SKK = combinateur nombre naturel 0 de Barendregt
# (constante,  ne pas modifier)
I = Combinator(SK, K)


## Combinateur O = SI   (constante,  ne pas modifier)
O = Combinator(S, I)


## Combinateur iota \htmlonly &iota;\endhtmlonly = S[O(KS)](KK)   (constante,  ne pas modifier)
iota = Combinator(S,
                  Combinator(O, KS),
                  KK)


## Combinateur KI = combinateur faux = combinateur nombre naturel 0 de Church
# (constante,  ne pas modifier)
KI = Combinator(K, I)


## Combinateur M = OI   (constante,  ne pas modifier)
M = Combinator(O, I)


## Combinateur L = SB(KM)   (constante,  ne pas modifier)
L = Combinator(S,
               B,
               Combinator(K, M))


## Combinateur Omega \htmlonly &Omega;\endhtmlonly = MM   (constante,  ne pas modifier)
Omega = Combinator(M, M)


## Combinateur R =  S(KB){S(KO)K}   (constante,  ne pas modifier)
R = Combinator(S,
               Combinator(K, B),
               Combinator(S,
                          Combinator(K, O),
                          K))


## Combinateur T = S(K0)K   (constante,  ne pas modifier)
T = Combinator(S,
               Combinator(K, O),
               K)


## Combinateur U = S(KO)M   (constante,  ne pas modifier)
U = Combinator(S,
               Combinator(K, O),
               M)


## Combinateur V = S(KC)T   (constante,  ne pas modifier)
V = Combinator(S,
               Combinator(K, C),
               T)


## Combinateur W = SS(KI)   (constante,  ne pas modifier)
W = Combinator(SS,
               Combinator(K, I))


## Combinateur Y = S(KM)[SB(KM)]   (constante,  ne pas modifier)
Y = Combinator(S,
               Combinator(K, M),
               Combinator(S,
                          B,
                          Combinator(K, M)))



# Combinateurs boolens

## Combinateur not (ngation boolenne) = S{O[K(KI)]}(KK)   (constante,  ne pas modifier)
Bnot = Combinator(S,
                  Combinator(O,
                             Combinator(K, KI)),
                  KK)

## Autre combinateur not (ngation boolenne) = S{O[O(KI)]}(KK)   (constante,  ne pas modifier)
Bnot_my = Combinator(S,
                     Combinator(O,
                                Combinator(O, KI)),
                     KK)

## Combinateur and (et  boolen, conjonction) = S(K{O[K(KI)]})   (constante,  ne pas modifier)
Band = Combinator(S,
                  Combinator(K,
                             Combinator(O,
                                        Combinator(K, KI))))

## Combinateur or (ou boolen, disjonction) = O(KK)   (constante,  ne pas modifier)
Bor = Combinator(O, KK)

## Combinateur implication boolenne = S{K[O(KK)]}   (constante,  ne pas modifier)
Bimp = Combinator(S,
                  Combinator(K,
                             Combinator(O, KK)))

## Combinateur ngation de la rciproque boolenne = O[K(KI)]   (constante,  ne pas modifier)
Bnotrec = Combinator(O,
                     Combinator(K, KI))



# Combinateurs nombres naturels de Barendregt

## Combinateur est gal  0 ? pour nombres naturels de Barendregt = TK
# (constante,  ne pas modifier)
NBzero_is = Combinator(T, K)

## Combinateur successeur pour nombres naturels de Barendregt = V(KI)
# (constante,  ne pas modifier)
NBsucc = Combinator(V, KI)

## Combinateur prdcesseur pour nombres naturels de Barendregt = T(KI)
# (constante,  ne pas modifier)
NBprev = Combinator(T, KI)

## Combinateur prdcesseur pour nombres naturels de Barendregt = ???
# (constante,  ne pas modifier)
NBadd = Combinator(Y,
                   Combinator(B,
                              Combinator(S,
                                         Combinator(B,
                                                    S,
                                                    Combinator(C, NBzero_is))),
                              Combinator(B,
                                         Combinator(B,
                                                    Combinator(B, NBsucc)),
                                         Combinator(C,
                                                    Combinator(B, C,
                                                               Combinator(B, B)),
                                                    NBprev))))



# Combinateurs nombres naturels de Church

## Combinateur successeur pour les nombres naturels de Church = SB   (constante,  ne pas modifier)
NCsucc = Combinator(S, B)

## Combinateur addition pour les nombres naturels de Church = BS(BB)
# (constante,  ne pas modifier)
NCadd = Combinator(B,
                   S,
                   Combinator(B, B))


## Combinateur atomiques utilis comme "mta-variable" x
Vx = Combinator(Atom_Vx)

## Combinateur atomiques utilis comme "mta-variable" y
Vy = Combinator(Atom_Vy)

## Combinateur atomiques utilis comme "mta-variable" z
Vz = Combinator(Atom_Vz)



# ###########
# Variables #
#############
## Spcifie le type de nombres naturels grs par les oprations arithmtiques de la classe
# == BARENDREGT ou CHURCH
natural_sys = CHURCH


## Dictionnaire des strings reconnues comme combinateurs par str_to, associes  leur Combinator
str_combs = {'K' : K,
             'S' : S,
             'x' : Vx,
             'y' : Vy,
             'z' : Vz}


## None ou Combinator qui remplacera la "mta-variable" Vx
var_x = None

## None ou Combinator qui remplacera la "mta-variable" Vy
var_y = None

## None ou Combinator qui remplacera la "mta-variable" Vz
var_z = None



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        global natural_sys

        import sys

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        import DSPython.debug as debug

        debug.test_begin(VERSION, __debug__)

        BASICS = (K, S, KK, KS, SS, SK)
        STDS = (B, C, I, iota, KI, L, M, O, R, T, U, V, W, Y)
        BOOLS = (Bnot, Bnot_my, Band, Bor, Bimp, Bnotrec)
        NATURALS_FCT = (NBzero_is, NBsucc, NBprev, NBadd)
        NATURALS_FCT_CHURCH = (NCsucc, NCadd)

        STABLES = BASICS + STDS + BOOLS + (NCsucc, )
        UNSTABLES = (Omega, ) + NATURALS_FCT + (NCadd, )

        ALLS = STABLES + UNSTABLES
        VARS = (Vx, Vy, Vz)

        ALLS_VARS = ALLS + VARS



        print('func_B()...', end=''); sys.stdout.flush()
        assert func_B(Atom_K, (), 0) == (Combinator(Atom_K), 0)
        assert func_B(Atom_K, (), 1) == (Combinator(Atom_K), 0)
        assert func_B(Atom_K, (), 2) == (Combinator(Atom_K), 0)
        assert func_B(Atom_K, (), None) == (Combinator(Atom_K), 0)

        assert func_B(Atom_S, (), 0) == (Combinator(Atom_S), 0)
        assert func_B(Atom_S, (), 1) == (Combinator(Atom_S), 0)
        assert func_B(Atom_S, (), 2) == (Combinator(Atom_S), 0)
        assert func_B(Atom_S, (), None) == (Combinator(Atom_S), 0)

        assert func_B(Atom_K, (Vx, ), 0) == (Combinator(Atom_K, Vx), 0)
        assert func_B(Atom_K, (Vx, ), 1) == (Combinator(Atom_K, Vx), 0)
        assert func_B(Atom_K, (Vx, ), 2) == (Combinator(Atom_K, Vx), 0)
        assert func_B(Atom_K, (Vx, ), None) == (Combinator(Atom_K, Vx), 0)

        assert func_B(Atom_S, (Vx, ), 0) == (Combinator(Atom_S, Vx), 0)
        assert func_B(Atom_S, (Vx, ), 1) == (Combinator(Atom_S, Vx), 0)
        assert func_B(Atom_S, (Vx, ), 2) == (Combinator(Atom_S, Vx), 0)
        assert func_B(Atom_S, (Vx, ), None) == (Combinator(Atom_S, Vx), 0)

        assert func_B(Atom_K, (Vx, Vy), 0) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_B(Atom_K, (Vx, Vy), 1) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_B(Atom_K, (Vx, Vy), 2) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_B(Atom_K, (Vx, Vy), None) == (Combinator(Atom_K, Vx, Vy), 0)

        assert func_B(Atom_S, (Vx, Vy), 0) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_B(Atom_S, (Vx, Vy), 1) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_B(Atom_S, (Vx, Vy), 2) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_B(Atom_S, (Vx, Vy), None) == (Combinator(Atom_S, Vx, Vy), 0)

        for i in range(3, len(ALLS_VARS)):
            assert func_B(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_B(Atom_K, ALLS_VARS[:i], 1) == (Combinator(ALLS_VARS[0],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_B(Atom_K, ALLS_VARS[:i], 2) == (Combinator(ALLS_VARS[0],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_B(Atom_K, ALLS_VARS[:i], None) == (Combinator(ALLS_VARS[0],
                                                                      Combinator(ALLS_VARS[1],
                                                                                 ALLS_VARS[2]),
                                                                      *ALLS_VARS[3:i]), 1)

            assert func_B(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_B(Atom_S, ALLS_VARS[:i], 1) == (Combinator(ALLS_VARS[0],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_B(Atom_S, ALLS_VARS[:i], 2) == (Combinator(ALLS_VARS[0],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_B(Atom_S, ALLS_VARS[:i], None) == (Combinator(ALLS_VARS[0],
                                                                      Combinator(ALLS_VARS[1],
                                                                                 ALLS_VARS[2]),
                                                                      *ALLS_VARS[3:i]), 1)
        print('ok'); sys.stdout.flush()


        print('func_C()...', end=''); sys.stdout.flush()
        assert func_C(Atom_K, (), 0) == (Combinator(Atom_K), 0)
        assert func_C(Atom_K, (), 1) == (Combinator(Atom_K), 0)
        assert func_C(Atom_K, (), 2) == (Combinator(Atom_K), 0)
        assert func_C(Atom_K, (), None) == (Combinator(Atom_K), 0)

        assert func_C(Atom_S, (), 0) == (Combinator(Atom_S), 0)
        assert func_C(Atom_S, (), 1) == (Combinator(Atom_S), 0)
        assert func_C(Atom_S, (), 2) == (Combinator(Atom_S), 0)
        assert func_C(Atom_S, (), None) == (Combinator(Atom_S), 0)

        assert func_C(Atom_K, (Vx, ), 0) == (Combinator(Atom_K, Vx), 0)
        assert func_C(Atom_K, (Vx, ), 1) == (Combinator(Atom_K, Vx), 0)
        assert func_C(Atom_K, (Vx, ), 2) == (Combinator(Atom_K, Vx), 0)
        assert func_C(Atom_K, (Vx, ), None) == (Combinator(Atom_K, Vx), 0)

        assert func_C(Atom_S, (Vx, ), 0) == (Combinator(Atom_S, Vx), 0)
        assert func_C(Atom_S, (Vx, ), 1) == (Combinator(Atom_S, Vx), 0)
        assert func_C(Atom_S, (Vx, ), 2) == (Combinator(Atom_S, Vx), 0)
        assert func_C(Atom_S, (Vx, ), None) == (Combinator(Atom_S, Vx), 0)

        assert func_C(Atom_K, (Vx, Vy), 0) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_C(Atom_K, (Vx, Vy), 1) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_C(Atom_K, (Vx, Vy), 2) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_C(Atom_K, (Vx, Vy), None) == (Combinator(Atom_K, Vx, Vy), 0)

        assert func_C(Atom_S, (Vx, Vy), 0) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_C(Atom_S, (Vx, Vy), 1) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_C(Atom_S, (Vx, Vy), 2) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_C(Atom_S, (Vx, Vy), None) == (Combinator(Atom_S, Vx, Vy), 0)

        for i in range(3, len(ALLS_VARS)):
            assert func_C(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_C(Atom_K, ALLS_VARS[:i], 1) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   ALLS_VARS[1],
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_C(Atom_K, ALLS_VARS[:i], 2) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   ALLS_VARS[1],
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_C(Atom_K, ALLS_VARS[:i], None) == (Combinator(ALLS_VARS[0],
                                                                      ALLS_VARS[2],
                                                                      ALLS_VARS[1],
                                                                      *ALLS_VARS[3:i]), 1)

            assert func_C(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_C(Atom_S, ALLS_VARS[:i], 1) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   ALLS_VARS[1],
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_C(Atom_S, ALLS_VARS[:i], 2) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   ALLS_VARS[1],
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_C(Atom_S, ALLS_VARS[:i], None) == (Combinator(ALLS_VARS[0],
                                                                      ALLS_VARS[2],
                                                                      ALLS_VARS[1],
                                                                      *ALLS_VARS[3:i]), 1)
        print('ok'); sys.stdout.flush()


        print('func_const()...', end=''); sys.stdout.flush()
        for i in range(len(ALLS_VARS)):
            assert func_const(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_const(Atom_K, ALLS_VARS[:i], 1) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_const(Atom_K, ALLS_VARS[:i], 2) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_const(Atom_K, ALLS_VARS[:i], None) \
                   == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)

            assert func_const(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_const(Atom_S, ALLS_VARS[:i], 1) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_const(Atom_S, ALLS_VARS[:i], 2) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_const(Atom_S, ALLS_VARS[:i], None) \
                   == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
        print('ok'); sys.stdout.flush()


        print('func_I()...', end=''); sys.stdout.flush()
        assert func_I(Atom_K, (), 0) == (Combinator(Atom_K), 0)
        assert func_I(Atom_K, (), 1) == (Combinator(Atom_K), 0)
        assert func_I(Atom_K, (), 2) == (Combinator(Atom_K), 0)
        assert func_I(Atom_K, (), None) == (Combinator(Atom_K), 0)

        assert func_I(Atom_S, (), 0) == (Combinator(Atom_S), 0)
        assert func_I(Atom_S, (), 1) == (Combinator(Atom_S), 0)
        assert func_I(Atom_S, (), 2) == (Combinator(Atom_S), 0)
        assert func_I(Atom_S, (), None) == (Combinator(Atom_S), 0)

        assert func_I(Atom_K, (Vx, ), 0) == (Combinator(Atom_K, Vx), 0)
        assert func_I(Atom_K, (Vx, ), 1) == (Vx, 1)
        assert func_I(Atom_K, (Vx, ), 2) == (Vx, 1)
        assert func_I(Atom_K, (Vx, ), None) == (Vx, 1)

        assert func_I(Atom_S, (Vx, ), 0) == (Combinator(Atom_S, Vx), 0)
        assert func_I(Atom_S, (Vx, ), 1) == (Vx, 1)
        assert func_I(Atom_S, (Vx, ), 2) == (Vx, 1)
        assert func_I(Atom_S, (Vx, ), None) == (Vx, 1)

        for i in range(1, len(ALLS_VARS)):
            assert func_I(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_I(Atom_K, ALLS_VARS[:i], 1) == (Combinator(*ALLS_VARS[:i]), 1)
            assert func_I(Atom_K, ALLS_VARS[:i], 2) == (Combinator(*ALLS_VARS[:i]), 1)
            assert func_I(Atom_K, ALLS_VARS[:i], None) == (Combinator(*ALLS_VARS[:i]), 1)

            assert func_I(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_I(Atom_S, ALLS_VARS[:i], 1) == (Combinator(*ALLS_VARS[:i]), 1)
            assert func_I(Atom_S, ALLS_VARS[:i], 2) == (Combinator(*ALLS_VARS[:i]), 1)
            assert func_I(Atom_S, ALLS_VARS[:i], None) == (Combinator(*ALLS_VARS[:i]), 1)
        print('ok'); sys.stdout.flush()


        print('func_K()...', end=''); sys.stdout.flush()
        assert func_K(Atom_K, (), 0) == (Combinator(Atom_K), 0)
        assert func_K(Atom_K, (), 1) == (Combinator(Atom_K), 0)
        assert func_K(Atom_K, (), 2) == (Combinator(Atom_K), 0)
        assert func_K(Atom_K, (), None) == (Combinator(Atom_K), 0)

        assert func_K(Atom_S, (), 0) == (Combinator(Atom_S), 0)
        assert func_K(Atom_S, (), 1) == (Combinator(Atom_S), 0)
        assert func_K(Atom_S, (), 2) == (Combinator(Atom_S), 0)
        assert func_K(Atom_S, (), None) == (Combinator(Atom_S), 0)

        assert func_K(Atom_K, (Vx, ), 0) == (Combinator(Atom_K, Vx), 0)
        assert func_K(Atom_K, (Vx, ), 1) == (Combinator(Atom_K, Vx), 0)
        assert func_K(Atom_K, (Vx, ), 2) == (Combinator(Atom_K, Vx), 0)
        assert func_K(Atom_K, (Vx, ), None) == (Combinator(Atom_K, Vx), 0)

        assert func_K(Atom_S, (Vx, ), 0) == (Combinator(Atom_S, Vx), 0)
        assert func_K(Atom_S, (Vx, ), 1) == (Combinator(Atom_S, Vx), 0)
        assert func_K(Atom_S, (Vx, ), 2) == (Combinator(Atom_S, Vx), 0)
        assert func_K(Atom_S, (Vx, ), None) == (Combinator(Atom_S, Vx), 0)

        for i in range(2, len(ALLS_VARS)):
            assert func_K(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_K(Atom_K, ALLS_VARS[:i], 1) \
                   == (Combinator(ALLS_VARS[0], *ALLS_VARS[2:i]), 1)
            assert func_K(Atom_K, ALLS_VARS[:i], 2) \
                   == (Combinator(ALLS_VARS[0], *ALLS_VARS[2:i]), 1)
            assert func_K(Atom_K, ALLS_VARS[:i], None) \
                   == (Combinator(ALLS_VARS[0], *ALLS_VARS[2:i]), 1)

            assert func_K(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_K(Atom_S, ALLS_VARS[:i], 1) \
                   == (Combinator(ALLS_VARS[0], *ALLS_VARS[2:i]), 1)
            assert func_K(Atom_S, ALLS_VARS[:i], 2) \
                   == (Combinator(ALLS_VARS[0], *ALLS_VARS[2:i]), 1)
            assert func_K(Atom_S, ALLS_VARS[:i], None) \
                   == (Combinator(ALLS_VARS[0], *ALLS_VARS[2:i]), 1)
        print('ok'); sys.stdout.flush()


        print('func_S()...', end=''); sys.stdout.flush()
        assert func_S(Atom_K, (), 0) == (Combinator(Atom_K), 0)
        assert func_S(Atom_K, (), 1) == (Combinator(Atom_K), 0)
        assert func_S(Atom_K, (), 2) == (Combinator(Atom_K), 0)
        assert func_S(Atom_K, (), None) == (Combinator(Atom_K), 0)

        assert func_S(Atom_S, (), 0) == (Combinator(Atom_S), 0)
        assert func_S(Atom_S, (), 1) == (Combinator(Atom_S), 0)
        assert func_S(Atom_S, (), 2) == (Combinator(Atom_S), 0)
        assert func_S(Atom_S, (), None) == (Combinator(Atom_S), 0)

        assert func_S(Atom_K, (Vx, ), 0) == (Combinator(Atom_K, Vx), 0)
        assert func_S(Atom_K, (Vx, ), 1) == (Combinator(Atom_K, Vx), 0)
        assert func_S(Atom_K, (Vx, ), 2) == (Combinator(Atom_K, Vx), 0)
        assert func_S(Atom_K, (Vx, ), None) == (Combinator(Atom_K, Vx), 0)

        assert func_S(Atom_S, (Vx, ), 0) == (Combinator(Atom_S, Vx), 0)
        assert func_S(Atom_S, (Vx, ), 1) == (Combinator(Atom_S, Vx), 0)
        assert func_S(Atom_S, (Vx, ), 2) == (Combinator(Atom_S, Vx), 0)
        assert func_S(Atom_S, (Vx, ), None) == (Combinator(Atom_S, Vx), 0)

        assert func_S(Atom_K, (Vx, Vy), 0) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_S(Atom_K, (Vx, Vy), 1) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_S(Atom_K, (Vx, Vy), 2) == (Combinator(Atom_K, Vx, Vy), 0)
        assert func_S(Atom_K, (Vx, Vy), None) == (Combinator(Atom_K, Vx, Vy), 0)

        assert func_S(Atom_S, (Vx, Vy), 0) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_S(Atom_S, (Vx, Vy), 1) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_S(Atom_S, (Vx, Vy), 2) == (Combinator(Atom_S, Vx, Vy), 0)
        assert func_S(Atom_S, (Vx, Vy), None) == (Combinator(Atom_S, Vx, Vy), 0)

        for i in range(3, len(ALLS_VARS)):
            assert func_S(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_S(Atom_K, ALLS_VARS[:i], 1) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_S(Atom_K, ALLS_VARS[:i], 2) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_S(Atom_K, ALLS_VARS[:i], None) == (Combinator(ALLS_VARS[0],
                                                                      ALLS_VARS[2],
                                                                      Combinator(ALLS_VARS[1],
                                                                                 ALLS_VARS[2]),
                                                                      *ALLS_VARS[3:i]), 1)

            assert func_S(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_S(Atom_S, ALLS_VARS[:i], 1) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_S(Atom_S, ALLS_VARS[:i], 2) == (Combinator(ALLS_VARS[0],
                                                                   ALLS_VARS[2],
                                                                   Combinator(ALLS_VARS[1],
                                                                              ALLS_VARS[2]),
                                                                   *ALLS_VARS[3:i]), 1)
            assert func_S(Atom_S, ALLS_VARS[:i], None) == (Combinator(ALLS_VARS[0],
                                                                      ALLS_VARS[2],
                                                                      Combinator(ALLS_VARS[1],
                                                                                 ALLS_VARS[2]),
                                                                      *ALLS_VARS[3:i]), 1)
        print('ok'); sys.stdout.flush()


        print('func_W()...', end=''); sys.stdout.flush()
        assert func_W(Atom_K, (), 0) == (Combinator(Atom_K), 0)
        assert func_W(Atom_K, (), 1) == (Combinator(Atom_K), 0)
        assert func_W(Atom_K, (), 2) == (Combinator(Atom_K), 0)
        assert func_W(Atom_K, (), None) == (Combinator(Atom_K), 0)

        assert func_W(Atom_S, (), 0) == (Combinator(Atom_S), 0)
        assert func_W(Atom_S, (), 1) == (Combinator(Atom_S), 0)
        assert func_W(Atom_S, (), 2) == (Combinator(Atom_S), 0)
        assert func_W(Atom_S, (), None) == (Combinator(Atom_S), 0)

        assert func_W(Atom_K, (Vx, ), 0) == (Combinator(Atom_K, Vx), 0)
        assert func_W(Atom_K, (Vx, ), 1) == (Combinator(Atom_K, Vx), 0)
        assert func_W(Atom_K, (Vx, ), 2) == (Combinator(Atom_K, Vx), 0)
        assert func_W(Atom_K, (Vx, ), None) == (Combinator(Atom_K, Vx), 0)

        assert func_W(Atom_S, (Vx, ), 0) == (Combinator(Atom_S, Vx), 0)
        assert func_W(Atom_S, (Vx, ), 1) == (Combinator(Atom_S, Vx), 0)
        assert func_W(Atom_S, (Vx, ), 2) == (Combinator(Atom_S, Vx), 0)
        assert func_W(Atom_S, (Vx, ), None) == (Combinator(Atom_S, Vx), 0)

        for i in range(2, len(ALLS_VARS)):
            assert func_W(Atom_K, ALLS_VARS[:i], 0) == (Combinator(Atom_K, *ALLS_VARS[:i]), 0)
            assert func_W(Atom_K, ALLS_VARS[:i], 1) \
                   == (Combinator(ALLS_VARS[0], ALLS_VARS[1], ALLS_VARS[1], *ALLS_VARS[2:i]), 1)
            assert func_W(Atom_K, ALLS_VARS[:i], 2) \
                   == (Combinator(ALLS_VARS[0], ALLS_VARS[1], ALLS_VARS[1], *ALLS_VARS[2:i]), 1)
            assert func_W(Atom_K, ALLS_VARS[:i], None) \
                   == (Combinator(ALLS_VARS[0], ALLS_VARS[1], ALLS_VARS[1], *ALLS_VARS[2:i]), 1)

            assert func_W(Atom_S, ALLS_VARS[:i], 0) == (Combinator(Atom_S, *ALLS_VARS[:i]), 0)
            assert func_W(Atom_S, ALLS_VARS[:i], 1) \
                   == (Combinator(ALLS_VARS[0], ALLS_VARS[1], ALLS_VARS[1], *ALLS_VARS[2:i]), 1)
            assert func_W(Atom_S, ALLS_VARS[:i], 2) \
                   == (Combinator(ALLS_VARS[0], ALLS_VARS[1], ALLS_VARS[1], *ALLS_VARS[2:i]), 1)
            assert func_W(Atom_S, ALLS_VARS[:i], None) \
                   == (Combinator(ALLS_VARS[0], ALLS_VARS[1], ALLS_VARS[1], *ALLS_VARS[2:i]), 1)
        print('ok'); sys.stdout.flush()


        print('func_x()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('func_y()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('func_z()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()



        print()
        print()
        print("Atom_K: '{0}'".format(Atom_K)); sys.stdout.flush()
        print("Atom_S: '{0}'".format(Atom_S)); sys.stdout.flush()
        print("Atom_Vx: '{0}'".format(Atom_Vx)); sys.stdout.flush()
        print("Atom_Vy: '{0}'".format(Atom_Vy)); sys.stdout.flush()
        print("Atom_Vz: '{0}'".format(Atom_Vz)); sys.stdout.flush()


        print()
        print('Atom()...', end=''); sys.stdout.flush()
        assert Atom_K._f == func_K, Atom_K._f
        assert Atom_K._s == 'K', Atom_K._s

        assert Atom_S._f == func_S, Atom_S._f
        assert Atom_S._s == 'S', Atom_S._s

        assert Atom('xyz_ABC')._f == func_const, Atom('xyz_ABC')._f
        assert Atom('xyz_ABC')._s == 'xyz_ABC', Atom('xyz_ABC')._s

        assert Atom('xyz_ABC', func_K)._f == func_K, Atom('xyz_ABC', func_K)._f
        assert Atom('xyz_ABC', func_K)._s == 'xyz_ABC', Atom('xyz_ABC', func_K)._s
        print('ok'); sys.stdout.flush()


        print('Atom.__call__()...', end=''); sys.stdout.flush()
        assert Atom_K() == (K, 0), Atom_K(())
        assert Atom_K(Vx) == (Combinator(K, Vx), 0), Atom_K(Vx)
        assert Atom_K(Vx, Vy) == (Vx, 1), Atom_K(Vx, Vy)
        assert Atom_K(Vx, Vy, Vz) == (Combinator(Vx, Vz), 1), Atom_K(Vx, Vy, Vz)
        assert Atom_K(Vx, Vy, Vz, nb=0) == (Combinator(K, Vx, Vy, Vz), 0), Atom_K(Vx, Vy, Vz, nb=0)

        assert Atom_K(B) == (Combinator(K, B), 0), Atom_K(B)
        assert Atom_K(B, C) == (B, 1), Atom_K(B, C)

        assert Atom_S() == (S, 0), Atom_S()
        assert Atom_S(Vx) == (Combinator(S, Vx), 0), Atom_S(Vx)
        assert Atom_S(Vx, Vy) == (Combinator(S, Vx, Vy), 0), Atom_S(Vx, Vy)
        assert Atom_S(Vx, Vy, Vz) == (Combinator(Vx,
                                                 Vz,
                                                 Combinator(Vy, Vz)), 1), Atom_S(Vx, Vy, Vz)
        assert Atom_S(Vx, Vy, Vz, Vx) == (Combinator(Vx,
                                                     Vz,
                                                     Combinator(Vy, Vz),
                                                     Vx), 1), Atom_S(Vx, Vy, Vz, Vx)
        assert Atom_S(Vx, Vy, Vz, Vx, nb=0) == (Combinator(S, Vx, Vy, Vz, Vx), 0), \
               Atom_S(Vx, Vy, Vz, Vx, nb=0)

        assert Atom_S(B) == (Combinator(S, B), 0), Atom_S(B)
        assert Atom_S(B, C) == (Combinator(S, B, C), 0), Atom_S(B, C)
        assert Atom_S(B, C, I, nb=1) == (Combinator(B,
                                                    I,
                                                    Combinator(C, I)), 1), \
                                                    Atom_S(B, C, I, nb=1)

        assert Atom('_const')() == (Combinator(Atom('_const')), 0), (x, Atom('_const')())

        for x in STDS[:-1 if debug.assertspeed > debug.ASSERT_NORMAL else 7]:
            assert Atom('_const')(x) == (Combinator(Atom('_const'), x), 0), \
                   (x, Atom('_const')(x))

            assert Atom_K(x) == (Combinator(K, x), 0), (x, Atom_K(x))
            assert Atom_K(x, nb=None) == (Combinator(K, x), 0), (x, Atom_K(x, nb=None))
            assert Atom_K(x, nb=0) == (Combinator(K, x), 0), (x, Atom_K(x, nb=0))

            assert Atom_S(x)[0] == Combinator(S, x), (x, Atom_S(x))

            for y in STDS[:-1 if debug.assertspeed > debug.ASSERT_NORMAL else 7]:
                assert Atom('_const')(x, y) == (Combinator(Atom('_const'), x, y), 0),\
                       (x, Atom('_const')(x, y))

                assert Atom_K(x, y) == (x, 1), (x, y, Atom_K(x, y))
                assert Atom_K(x, y, nb=None) == (x, 1), (x, y, Atom_K(x, y, nb=None))
                assert Atom_K(x, y, nb=0) == (Combinator(K, x, y), 0), (x, Atom_K(x, y, nb=0))

                assert Atom_S(x, y)[0] == Combinator(S, x, y), (x, y, Atom_S(x, y))

                for z in STDS[:-1 if debug.assertspeed > debug.ASSERT_NORMAL else 5]:
                    assert Atom_K(x, y, z, nb=0) == (Combinator(K, x, y, z), 0), \
                           (x, y, z, Atom_K(x, y, z, nb=0))
                    assert Atom_K(x, y, z, nb=1) == (Combinator(x, z), 1), \
                           (x, y, z, Atom_K(x, y, z, nb=1))
                    assert Atom_K(x, y, z, nb=2)[0] == Combinator(x, z)(nb=1)[0], \
                           (x, y, z, Atom_K(x, y, z, nb=2), Combinator(x, z)(nb=1))

                    assert Atom_S(x, y, z, nb=0) == (Combinator(S, x, y, z), 0), \
                           (x, y, z, Atom_S(x, y, z, nb=0))
                    assert Atom_S(x, y, z, nb=1) == (Combinator(x, z, Combinator(y, z)), 1), \
                           (x, y, z, Atom_S(x, y, z, nb=1))
                    assert Atom_S(x, y, z, nb=2)[0] == Combinator(x, z, Combinator(y, z))(nb=1)[0],\
                           (x, y, z, Atom_S(x, y, z, nb=2))

                    assert Atom_S(x, y, z, Vx, nb=1) \
                           == (Combinator(x, z, Combinator(y, z), Vx), 1), \
                           (x, y, z, Atom_S(x, y, z, Vx, nb=1))

            assert Atom_K(Vx, x, Vy, nb=3)[0] == Combinator(Vx, Vy)(nb=2)[0], \
                   (Vx, x, Vy, Atom_K(Vx, x, Vy, nb=3), Combinator(Vx, Vy)(nb=2))
            assert Atom_K(Vx, x, Vy, nb=None)[0] == Combinator(Vx, Vy)(nb=None)[0], \
                   (Vx, x, Vy, Atom_K(Vx, x, Vy, nb=None), Combinator(Vx, Vy)(nb=None))
            assert Atom_K(Vx, x, Vy)[0] == Combinator(Vx, Vy)()[0], \
                   (Vx, x, Vy, Atom_K(Vx, x, Vy), Combinator(Vx, Vy)())

        if debug.assertspeed <= debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('Atom.__cmp__()...', end=''); sys.stdout.flush()
        if sys.version_info[0] >= 3:  # La fonction built-in cmp() n'existe plus sous Python 3
            def mycmp(a, b):
                return a.__cmp__(b)
        else:
            def mycmp(a, b):
                return cmp(a, b)

        assert mycmp(Atom_K, Atom_K) == 0, mycmp(Atom_K, Atom_K)
        assert mycmp(Atom_S, Atom_S) == 0, mycmp(Atom_S, Atom_S)
        assert mycmp(Atom_K, Atom_S) < 0, mycmp(Atom_K, Atom_S)
        assert mycmp(Atom_S, Atom_K) > 0, mycmp(Atom_S, Atom_K)

        assert mycmp(Atom_K, Atom('K')) == 0, mycmp(Atom_K, Atom('K'))
        assert mycmp(Atom_K, Atom('K', func_S)) == 0, mycmp(Atom_K, Atom('K', func_S))

        assert mycmp(Atom_S, Atom('K')) > 0, mycmp(Atom_S, Atom('K'))
        assert mycmp(Atom_S, Atom('K', func_S)) > 0, mycmp(Atom_S, Atom('K', func_S))

        assert mycmp(Atom_S, Atom('S')) == 0, mycmp(Atom_S, Atom('S'))
        assert mycmp(Atom_S, Atom('S', func_K)) == 0, mycmp(Atom_S, Atom('S', func_K))

        assert mycmp(Atom_K, Atom('S')) < 0, mycmp(Atom_K, Atom('S'))
        assert mycmp(Atom_K, Atom('S', func_K)) < 0, mycmp(Atom_K, Atom('S', func_K))

        assert mycmp(Atom_K, Atom('k')) < 0, mycmp(Atom_K, Atom('k'))
        assert mycmp(Atom_K, Atom('K_')) < 0, mycmp(Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__eq__()...', end=''); sys.stdout.flush()
        assert Atom_K == Atom_K, Atom_K
        assert Atom_S == Atom_S, Atom_S
        assert not(Atom_K == Atom_S), (Atom_K, Atom_S)
        assert not(Atom_S == Atom_K), (Atom_S, Atom_K)
        assert not(Atom_K == 'K')
        assert not(Atom_S == 'S')

        assert Atom_K == Atom('K'), (Atom_K, Atom('K'))
        assert Atom_K == Atom('K', func_S), (Atom_K, Atom('K', func_S))

        assert Atom_S == Atom('S'), (Atom_K, Atom('S'))
        assert Atom_S == Atom('S', func_K), (Atom_S, Atom('S', func_K))

        assert not(Atom_K == Atom('k')), (Atom_K, Atom('k'))
        assert not(Atom_K == Atom('K_')), (Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__ge__()...', end=''); sys.stdout.flush()
        assert Atom_K >= Atom_K, Atom_K
        assert Atom_S >= Atom_S, Atom_S
        assert not(Atom_K >= Atom_S), (Atom_K, Atom_S)
        assert Atom_S >= Atom_K, (Atom_S, Atom_K)

        assert Atom_K >= Atom('K'), (Atom_K, Atom('K'))
        assert Atom_K >= Atom('K', func_S), (Atom_K, Atom('K', func_S))

        assert Atom_S >= Atom('K'), (Atom_S, Atom('K'))
        assert Atom_S >= Atom('K', func_S), (Atom_S, Atom('K', func_S))

        assert Atom_S >= Atom('S'), (Atom_S, Atom('S'))
        assert Atom_S >= Atom('S', func_K), (Atom_S, Atom('S', func_K))

        assert not(Atom_K >= Atom('S')), (Atom_K, Atom('S'))
        assert not(Atom_K >= Atom('S', func_K)), (Atom_K, Atom('S', func_K))

        assert not(Atom_K >= Atom('k')), (Atom_K, Atom('k'))
        assert not(Atom_K >= Atom('K_')), (Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__gt__()...', end=''); sys.stdout.flush()
        assert not(Atom_K > Atom_K), Atom_K
        assert not(Atom_S > Atom_S), Atom_S
        assert not(Atom_K > Atom_S), (Atom_K, Atom_S)
        assert Atom_S > Atom_K, (Atom_S, Atom_K)

        assert not(Atom_K > Atom('K')), (Atom_K, Atom('K'))
        assert not(Atom_K > Atom('K', func_S)), (Atom_K, Atom('K', func_S))

        assert Atom_S > Atom('K'), (Atom_S, Atom('K'))
        assert Atom_S > Atom('K', func_S), (Atom_S, Atom('K', func_S))

        assert not(Atom_S > Atom('S')), (Atom_S, Atom('S'))
        assert not(Atom_S > Atom('S', func_K)), (Atom_S, Atom('S', func_K))

        assert not(Atom_K > Atom('S')), (Atom_K, Atom('S'))
        assert not(Atom_K > Atom('S', func_K)), (Atom_K, Atom('S', func_K))

        assert not(Atom_K > Atom('k')), (Atom_K, Atom('k'))
        assert not(Atom_K > Atom('K_')), (Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__hash__()...', end=''); sys.stdout.flush()
        assert hash(Atom_K) == hash(Atom_K)
        assert hash(Atom_S) == hash(Atom_S)
        assert hash(Atom_K) != hash(Atom_S)
        assert hash(Atom_S) != hash(Atom_K)

        assert hash(Atom_K) == hash(Atom('K'))
        assert hash(Atom_K) == hash(Atom('K', func_S))

        assert hash(Atom_S) == hash(Atom('S'))
        assert hash(Atom_S) == hash(Atom('S', func_K))
        assert hash(Atom_S) != hash(Atom('K'))

        assert hash(Atom_S) != hash(Atom('s'))
        assert hash(Atom_S) != hash(Atom('S_'))

        for x in (Atom_K, Atom_S, Atom_Vx, Atom_Vy, Atom_Vz):
            for y in (Atom_K, Atom_S, Atom_Vx, Atom_Vy, Atom_Vz):
                if x == y:
                    assert hash(x) == hash(y), (hash(x), hash(y), x, y)
                else:
                    assert hash(x) != hash(y), (hash(x), hash(y), x, y)
        print('ok'); sys.stdout.flush()


        print('Atom.__le__()...', end=''); sys.stdout.flush()
        assert Atom_K <= Atom_K, Atom_K
        assert Atom_S <= Atom_S, Atom_S
        assert Atom_K <= Atom_S, (Atom_K, Atom_S)
        assert not(Atom_S <= Atom_K), (Atom_S, Atom_K)

        assert Atom_K <= Atom('K'), (Atom_K, Atom('K'))
        assert Atom_K <= Atom('K', func_S), (Atom_K, Atom('K', func_S))

        assert not(Atom_S <= Atom('K')), (Atom_S, Atom('K'))
        assert not(Atom_S <= Atom('K', func_S)), (Atom_S, Atom('K', func_S))

        assert Atom_S <= Atom('S'), (Atom_S, Atom('S'))
        assert Atom_S <= Atom('S', func_K), (Atom_S, Atom('S', func_K))

        assert Atom_K <= Atom('S'), (Atom_K, Atom('S'))
        assert Atom_K <= Atom('S', func_K), (Atom_K, Atom('S', func_K))

        assert Atom_K <= Atom('k'), (Atom_K, Atom('k'))
        assert Atom_K <= Atom('K_'), (Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__lt__()...', end=''); sys.stdout.flush()
        assert not(Atom_K < Atom_K), Atom_K
        assert not(Atom_S < Atom_S), Atom_S
        assert Atom_K < Atom_S, (Atom_K, Atom_S)
        assert not(Atom_S < Atom_K), (Atom_S, Atom_K)

        assert not(Atom_K < Atom('K')), (Atom_K, Atom('K'))
        assert not(Atom_K < Atom('K', func_S)), (Atom_K, Atom('K', func_S))

        assert not(Atom_S < Atom('K')), (Atom_S, Atom('K'))
        assert not(Atom_S < Atom('K', func_S)), (Atom_S, Atom('K', func_S))

        assert not(Atom_S < Atom('S')), (Atom_S, Atom('S'))
        assert not(Atom_S < Atom('S', func_K)), (Atom_S, Atom('S', func_K))

        assert Atom_K < Atom('S'), (Atom_K, Atom('S'))
        assert Atom_K < Atom('S', func_K), (Atom_K, Atom('S', func_K))

        assert Atom_K < Atom('k'), (Atom_K, Atom('k'))
        assert Atom_K < Atom('K_'), (Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__ne__()...', end=''); sys.stdout.flush()
        assert not(Atom_K != Atom_K), Atom_K
        assert not(Atom_S != Atom_S), Atom_S
        assert Atom_K != Atom_S, (Atom_K, Atom_S)
        assert Atom_S != Atom_K, (Atom_S, Atom_K)
        assert Atom_K != 'K'
        assert Atom_S != 'S'

        assert not(Atom_K != Atom('K')), (Atom_K, Atom('K'))
        assert not(Atom_K != Atom('K', func_S)), (Atom_K, Atom('K', func_S))

        assert not(Atom_S != Atom('S')), (Atom_K, Atom('S'))
        assert not(Atom_S != Atom('S', func_K)), (Atom_S, Atom('S', func_K))

        assert Atom_K != Atom('k'), (Atom_K, Atom('k'))
        assert Atom_K != Atom('K_'), (Atom_K, Atom('K_'))
        print('ok'); sys.stdout.flush()


        print('Atom.__repr__()...', end=''); sys.stdout.flush()
        assert repr(Atom_K) == "Atom('K')", repr(Atom_K)
        assert repr(Atom_S) == "Atom('S')", repr(Atom_S)

        assert repr(Atom('K')) == "Atom('K')", repr(Atom('K'))
        assert repr(Atom('K', func_S)) == "Atom('K')", repr(Atom('K', func_S))

        assert repr(Atom('S')) == "Atom('S')", repr(Atom('S'))
        assert repr(Atom('S', func_K)) == "Atom('S')", repr(Atom('S', func_K))

        assert repr(Atom('xyz_ABC')) == "Atom('xyz_ABC')", repr(Atom('xyz_ABC'))
        print('ok'); sys.stdout.flush()


        print('Atom.__str__()...', end=''); sys.stdout.flush()
        assert str(Atom_K) == 'K', str(Atom_K)
        assert str(Atom_S) == 'S', str(Atom_S)

        assert str(Atom('K')) == 'K', str(Atom('K'))
        assert str(Atom('K', func_S)) == 'K', str(Atom('K', func_S))

        assert str(Atom('S')) == 'S', str(Atom('S'))
        assert str(Atom('S', func_K)) == 'S', str(Atom('S', func_K))

        assert str(Atom('xyz_ABC')) == 'xyz_ABC', str(Atom('xyz_ABC'))
        print('ok'); sys.stdout.flush()


        print('Atom.f()...', end=''); sys.stdout.flush()
        assert Atom_K.f() == func_K, (Atom_K.f(), func_K)
        assert Atom_S.f() == func_S, (Atom_S.f(), func_S)

        assert Atom('K').f() == func_const, (Atom('K').f(), func_const)
        assert Atom('xyz_ABC').f() == func_const, (Atom('xyz_ABC').f(), func_const)

        assert Atom('xyz_ABC', func_S).f() == func_S, (Atom('xyz_ABC', func_S).f(), func_S)
        print('ok'); sys.stdout.flush()


        print('Atom.nb_args()...', end=''); sys.stdout.flush()
        assert Atom_K.nb_args() == 2, Atom_K.nb_args()
        assert Atom_S.nb_args() == 3, Atom_S.nb_args()

        assert Atom('z', func_const).nb_args() == None, Atom('z', func_const).nb_args()

        assert Atom('z', func_B).nb_args() == 3, Atom('z', func_B).nb_args()
        assert Atom('z', func_B).nb_args(max=2) == None, Atom('z', func_B).nb_args(max=2)

        assert Atom('z', func_C).nb_args() == 3, Atom('z', func_C).nb_args()
        assert Atom('z', func_C).nb_args(max=2) == None, Atom('z', func_C).nb_args(max=2)

        assert Atom('z', func_I).nb_args() == 1, Atom('z', func_I).nb_args()
        assert Atom('z', func_I).nb_args(max=0) == None, Atom('z', func_I).nb_args(max=2)

        assert Atom('z', func_K).nb_args() == 2, Atom('z', func_K).nb_args()
        assert Atom('z', func_K).nb_args(max=1) == None, Atom('z', func_K).nb_args(max=2)

        assert Atom('z', func_S).nb_args() == 3, Atom('z', func_S).nb_args()
        assert Atom('z', func_S).nb_args(max=2) == None, Atom('z', func_S).nb_args(max=2)

        assert Atom('z', func_W).nb_args() == 2, Atom('z', func_W).nb_args()
        assert Atom('z', func_W).nb_args(max=1) == None, Atom('z', func_W).nb_args(max=2)
        print('ok'); sys.stdout.flush()



        print()
        print()
        print("BARENDREGT: '{0}'".format(BARENDREGT)); sys.stdout.flush()
        print("CHURCH: '{0}'".format(CHURCH)); sys.stdout.flush()

        print()
        print("K: '{0}'".format(K)); sys.stdout.flush()
        print("S: '{0}'".format(S)); sys.stdout.flush()
        print("KK: '{0}'".format(KK)); sys.stdout.flush()
        print("KS: '{0}'".format(KS)); sys.stdout.flush()
        print("SS: '{0}'".format(SS)); sys.stdout.flush()
        print("SK: '{0}'".format(SK)); sys.stdout.flush()

        print("B: '{0}'".format(B)); sys.stdout.flush()
        print("C: '{0}'".format(C)); sys.stdout.flush()
        print("I: '{0}'".format(I)); sys.stdout.flush()
        print("iota: '{0}'".format(iota)); sys.stdout.flush()
        print("KI: '{0}'".format(KI)); sys.stdout.flush()
        print("L: '{0}'".format(L)); sys.stdout.flush()
        print("M: '{0}'".format(M)); sys.stdout.flush()
        print("O: '{0}'".format(O)); sys.stdout.flush()
        print("Omega: '{0}'".format(Omega)); sys.stdout.flush()
        print("R: '{0}'".format(R)); sys.stdout.flush()
        print("T: '{0}'".format(T)); sys.stdout.flush()
        print("U: '{0}'".format(U)); sys.stdout.flush()
        print("V: '{0}'".format(V)); sys.stdout.flush()
        print("W: '{0}'".format(W)); sys.stdout.flush()
        print("Y: '{0}'".format(Y)); sys.stdout.flush()

        print("Bnot: '{0}'".format(Bnot)); sys.stdout.flush()
        print("Bnot_my: '{0}'".format(Bnot_my)); sys.stdout.flush()
        print("Band: '{0}'".format(Band)); sys.stdout.flush()
        print("Bor: '{0}'".format(Bor)); sys.stdout.flush()
        print("Bimp: '{0}'".format(Bimp)); sys.stdout.flush()
        print("Bnotrec: '{0}'".format(Bnotrec)); sys.stdout.flush()

        print("NBzero_is: '{0}'".format(NBzero_is)); sys.stdout.flush()
        print("NBsucc: '{0}'".format(NBsucc)); sys.stdout.flush()
        print("NBprev: '{0}'".format(NBprev)); sys.stdout.flush()
        print("NBadd: '{0}'".format(NBadd)); sys.stdout.flush()

        print("NCsucc: '{0}'".format(NCsucc)); sys.stdout.flush()
        print("NCadd: '{0}'".format(NCadd)); sys.stdout.flush()

        print("Vx: '{0}'".format(Vx)); sys.stdout.flush()
        print("Vy: '{0}'".format(Vy)); sys.stdout.flush()
        print("Vz: '{0}'".format(Vz)); sys.stdout.flush()


        assert str(K) == 'K', str(K)
        assert str(S) == 'S', str(S)
        assert str(KK) == 'K K', str(KK)
        assert str(KS) == 'K S', str(KS)
        assert str(SS) == 'S S', str(SS)
        assert str(SK) == 'S K', str(SK)

        assert str(B) == 'S (K S) K', str(B)
        assert str(C) == 'S (S (K (S (K S) K)) S) (K K)', str(C)
        assert str(I) == 'S K K', str(I)
        assert str(iota) == 'S (S (S K K) (K S)) (K K)', str(iota)
        assert str(KI) == 'K (S K K)', str(KI)
        assert str(L) == 'S (S (K S) K) (K (S (S K K) (S K K)))', str(L)
        assert str(M) == 'S (S K K) (S K K)', str(M)
        assert str(O) == 'S (S K K)', str(O)
        assert str(Omega) == 'S (S K K) (S K K) (S (S K K) (S K K))', str(Omega)
        assert str(R) == 'S (K (S (K S) K)) (S (K (S (S K K))) K)', str(R)
        assert str(T) == 'S (K (S (S K K))) K', str(T)
        assert str(U) == 'S (K (S (S K K))) (S (S K K) (S K K))', str(U)
        assert str(V) == 'S (K (S (S (K (S (K S) K)) S) (K K))) (S (K (S (S K K))) K)', str(V)
        assert str(W) == 'S S (K (S K K))', str(W)
        assert str(Y) == 'S (K (S (S K K) (S K K))) (S (S (K S) K) (K (S (S K K) (S K K))))', str(Y)

        assert str(Bnot) == 'S (S (S K K) (K (K (S K K)))) (K K)', str(Bnot)
        assert str(Bnot_my) == 'S (S (S K K) (S (S K K) (K (S K K)))) (K K)', str(Bnot_my)
        assert str(Band) == 'S (K (S (S K K) (K (K (S K K)))))', str(Band)
        assert str(Bor) == 'S (S K K) (K K)', str(Bor)
        assert str(Bimp) == 'S (K (S (S K K) (K K)))', str(Bimp)
        assert str(Bnotrec) == 'S (S K K) (K (K (S K K)))', str(Bnotrec)

        assert str(NBzero_is) == 'S (K (S (S K K))) K K', str(NBzero_is)
        assert str(NBsucc) == \
               'S (K (S (S (K (S (K S) K)) S) (K K))) (S (K (S (S K K))) K) (K (S K K))', \
               str(NBsucc)
        assert str(NBprev) == 'S (K (S (S K K))) K (K (S K K))', str(NBprev)
##         assert str(NBadd) == '???', str(NBadd)

        assert str(NCsucc) == 'S (S (K S) K)', str(NCsucc)
        assert str(NCadd) == 'S (K S) K S (S (K S) K (S (K S) K))', str(NCadd)

        assert str(Vx) == 'x', str(Vx)
        assert str(Vy) == 'y', str(Vy)
        assert str(Vz) == 'z', str(Vz)

        print()
        print('natural_sys: {0}'.format(natural_sys))
        assert (natural_sys == BARENDREGT) or (natural_sys == CHURCH), \
               natural_sys

        x = sorted(str_combs.keys())
        print('str_combs.keys(): {0}'.format(x))
        for x in str_combs:
            assert x == str(str_combs[x]), (x, str(str_combs[x]))


        print()
        print('Combinator()...', end=''); sys.stdout.flush()
        assert Combinator(Atom_K) == K, Combinator(Atom_K)
        assert Combinator(Atom_S) == S, Combinator(Atom_S)

        assert Combinator(K) == K, Combinator(K)
        assert Combinator(S) == S, Combinator(S)
        assert Combinator(KS) == KS, Combinator(KS)
        for x in ALLS:
            assert Combinator(x) == x, (x, Combinator(x))

        assert Combinator(Atom_K, Atom_S) == KS, Combinator(Atom_K, Atom_S)
        assert Combinator(S, K) == SK, Combinator(S, K)

        assert Combinator(False) == KI, Combinator(False)
        assert Combinator(True) == K, Combinator(True)

        natural_sys = BARENDREGT
        assert Combinator(False) == KI, Combinator(False)
        assert Combinator(True) == K, Combinator(True)

        assert Combinator(0) == I, Combinator(0)
        assert Combinator(1) == Combinator(NBsucc, I), Combinator(1)
        assert Combinator(2) == Combinator(NBsucc, Combinator(NBsucc, I)), Combinator(2)
        assert Combinator(3) == Combinator(NBsucc, Combinator(NBsucc, Combinator(NBsucc, I))), \
               Combinator(3)

        natural_sys = CHURCH
        assert Combinator(0) == KI, Combinator(0)
        assert Combinator(1) == Combinator(NCsucc, KI), Combinator(1)
        assert Combinator(2) == Combinator(NCsucc, Combinator(NCsucc, KI)), Combinator(2)
        assert Combinator(3) == Combinator(NCsucc, Combinator(NCsucc, Combinator(NCsucc, KI))), \
               Combinator(3)

        assert Combinator('K') == K, Combinator('K')
        assert Combinator('S') == S, Combinator('S')

        assert Combinator('K K') == KK, Combinator('K K')
        assert Combinator('K S') == KS, Combinator('K S')
        assert Combinator('S K') == SK, Combinator('S K')
        assert Combinator('S S') == SS, Combinator('S S')

        assert Combinator('S K K') == I, Combinator('S K K')

        assert Combinator('S (S (K (S (K S) K)) S) (K K)') == C, \
               (Combinator('S (S (K (S (K S) K)) S) (K K)'), C)

        assert Combinator(' S (K (S(S K K)(S K K))) (S (S(K S) K)(K (S (S K K)(S K K)) )) ') == Y, \
               (Combinator(' S (K (S(S K K)(S K K))) (S (S(K S) K)(K (S (S K K)(S K K)) )) '), Y)

        for x in (ALLS if debug.assertspeed >= debug.ASSERT_NORMAL else BASICS):
            assert Combinator(str(x)) == x, (x, Combinator(str(x)))
            assert Combinator('({0})'.format(x)) == x, (x, '({0})'.format(x))
            for y in (ALLS if debug.assertspeed > debug.ASSERT_NORMAL else BASICS):
                assert Combinator('{0}({1})'.format(x, y)) == Combinator(str(x),
                                                                         Combinator(str(y))), \
                       (x, y, Combinator('{0}({1})'.format(x, y)))
                assert Combinator('( (({0}) )  ({1}))'.format(x, y)) \
                       == Combinator(str(x), Combinator(str(y))), \
                       (x, y, Combinator('( (({0}) )  ({1}))'.format(x, y)))

        # La construction en une fois par une liste
        #   doit tre gale  la "construction parenthse  gauche"
        for n in range(1, len(ALLS)):
            x = Combinator(*ALLS[:n])
            y = Combinator(ALLS[0])
            for i in range(1, n):
                y = Combinator(y, ALLS[i])
            assert x == y
        if debug.assertspeed <= debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__and__()...', end=''); sys.stdout.flush()
        assert (K  & K)()[0]  == K,  (K  & K,  (K  & K)())
        assert (K  & KI)()[0] == KI, (K  & KI, (K  & KI)())
        assert (KI & K)()[0]  == KI, (KI & K,  (KI & K)())
        assert (KI & KI)()[0] == KI, (KI & KI, (KI & KI)())

        assert (K  & True)()[0]  == K,  (K  & True,  (K  & True)())
        assert (K  & False)()[0] == KI, (K  & False, (K  & False)())
        assert (KI & True)()[0]  == KI, (KI & True,  (KI & True)())
        assert (KI & False)()[0] == KI, (KI & False, (KI & False)())

        for x in ALLS_VARS:
            for y in ALLS_VARS + (True, False):
                assert x & y == Combinator(Band, x, y), (x, y, x & y, Combinator(Band, x, y))
        print('ok'); sys.stdout.flush()


        print('Combinator.__call__()...', end=''); sys.stdout.flush()
        assert K() == (K, 0), K()
        assert S() == (S, 0), S()
        assert KK() == (KK, 0), KK()
        assert KS() == (KS, 0), KS()
        assert SS() == (SS, 0), SS()
        assert SK() == (SK, 0), SK()
        for x in STABLES:
            assert x(nb=0) == (x, 0), (x, x(nb=0))
            assert x() == (x, 0), (x, x())
        for x in UNSTABLES:
            assert x(nb=0) == (x, 0), (x, x(nb=0))
            assert x(nb=1)[0] != x, (x, x(nb=1)[0])
            assert x(nb=1)[1] == 1, (x, x(nb=1)[1])

        assert Omega(nb=1) == (Combinator(I, M, Combinator(I, M)), 1), Omega(nb=1)

        assert Combinator(S, Combinator(B, B, S), KK)()[0] == C, \
               Combinator(S, Combinator(B, B, S), KK)()

        assert Combinator(W, S, K)()[0] == I, Combinator(W, S, K)()

        assert Combinator(S, I, I)()[0] == M, Combinator(S, I, I)()
        assert Combinator(W, I)()[0] == M, Combinator(W, I)()

        assert Combinator(B, Combinator(S, I), K)()[0] == T, Combinator(B, Combinator(S, I), K)()
        assert Combinator(B, O, K)()[0] == T, Combinator(B, O, K)()

        for x in STABLES:
            assert x() == (x, 0), (x, x())

            assert Combinator(KK, x)()[0] == K, (x, Combinator(KK, x)())
            assert Combinator(KS, x)()[0] == S, (x, Combinator(KS, x)())
            assert Combinator(I, x)()[0] == x, (x, Combinator(I, x)())
            if (x != M) and (x != U) and (x != Y):
                assert Combinator(M, x)()[0] == Combinator(x, x)()[0], \
                       (x, Combinator(M, x)())
            assert Combinator(W, x)()[0] == Combinator(S, x, I), (x, Combinator(W, x)())
            if (x != U) and (x != Y):
                assert Combinator(Bnot_my, x)()[0] == Combinator(x, Combinator(x, I), K)()[0], \
                       (x, Combinator(Bnot_my, x)())

        for x in UNSTABLES:
            assert x(nb=1)[0] != x, (x, x(nb=1))
            assert x(nb=1)[1] == 1, (x, x(nb=1))

        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for x in STABLES:
                for y in STABLES:
                    assert Combinator(K, x, y)()[0] == x, (x, y, Combinator(K, x, y)())
                    assert Combinator(SK, x, y)()[0] == y, (x, y, Combinator(SK, x, y)())
                    assert Combinator(KI, x, y)()[0] == y, (x, y, Combinator(KI, x, y)())
                    assert Combinator(T, K, I, x, y)()[0] == x, (x, y, Combinator(T, K, I, x, y)())
                    assert Combinator(T, K, S, x, y)()[0] == y, (x, y, Combinator(T, K, S, x, y)())
                    if (x in (M, U, Y)) or (y in (M, U, Y)):
                        continue
                    assert Combinator(SS, x, y)()[0] == Combinator(S, y, Combinator(x, y))()[0], \
                           (x, y, Combinator(SS, x, y)())
                    assert Combinator(O, x, y)()[0] == Combinator(y, Combinator(x, y))()[0], \
                           (x, y, Combinator(O, x, y)())
                    assert Combinator(T, x, y)()[0] == Combinator(y, x)()[0], \
                           (x, y, Combinator(T, x, y)())
                    if x != W:
                        assert Combinator(W, x, y)()[0] == Combinator(x, y, y)()[0], \
                               (x, y, Combinator(W, x, y)())
        if debug.assertspeed > debug.ASSERT_NORMAL:
            for x in (K, S, KK, KS, SS, SK, B, C, I, KI, O, T, Bnot_my):
                for y in (K, S, KK, KS, SK, B, C, KI, Bnot_my):
                    for z in STABLES:
                        if (z == U) or (z == Y):
                            continue
                        assert Combinator(S, x, y, z)()[0] \
                               == Combinator(x, z, Combinator(y, z))()[0], \
                               (x, y, z, Combinator(S, x, y, z)())
                        assert Combinator(B, x, y, z)()[0] == Combinator(x, Combinator(y, z))()[0],\
                               (x, y, z, Combinator(B, x, y, z)())
                        assert Combinator(C, x, y, z)()[0] == Combinator(x, z, y)()[0], \
                               (x, y, z, Combinator(C, x, y, z)())

        assert Combinator(I, K)()[0] == K, Combinator(I, K)()
        assert Combinator(Bnot_my, K)()[0] == KI, Combinator(Bnot_my, K)()
        assert Combinator(Bnot_my, Combinator(I, K))()[0] == KI, \
               Combinator(Bnot_my, Combinator(I, K))()[0]
        assert Combinator(Bnot_my, KI)()[0] == K, Combinator(Bnot_my, K)()
        if debug.assertspeed <= debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__cmp__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__contains__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__eq__()...', end=''); sys.stdout.flush()
        for i in range(len(ALLS)):
            x = ALLS[i]
            assert x == x, (i, x)
            for j in range(len(ALLS)):
                y = ALLS[j]
                if i == j:
                    assert y == x, (i, j, x, y)
                else:
                    assert not(y == x), (i, j, x, y)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__getitem__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__hash__()...', end=''); sys.stdout.flush()
        assert hash(K) == hash(K)
        assert hash(S) == hash(S)
        assert hash(K) != hash(S)
        assert hash(S) != hash(K)

        assert hash(K) == hash(Combinator('K'))

        assert hash(S) == hash(Combinator('S'))
        assert hash(S) != hash(Combinator('K'))

        for x in ALLS_VARS:
            for y in ALLS_VARS:
                if x == y:
                    assert hash(x) == hash(y), (hash(x), hash(y), x, y)
                else:
                    assert hash(x) != hash(y), (hash(x), hash(y), x, y)
        print('ok'); sys.stdout.flush()


        print('Combinator.__int__()...', end=''); sys.stdout.flush()
        natural_sys = BARENDREGT
        assert int(I) == 0, int(I)
        assert int(Combinator(V, KI, I)) == 1, int(Combinator(V, KI, I))

        natural_sys = CHURCH
        assert int(KI) == 0, int(KI)
        assert int(Combinator(S, B, KI)) == 1, int(Combinator(S, B, KI))

        for n in range(30):
            natural_sys = BARENDREGT
            x = Combinator(n)
            assert n == int(x), (n, int(x), x)

            natural_sys = CHURCH
            x = Combinator(n)
            assert n == int(x), (n, int(x), x)
        print('ok'); sys.stdout.flush()


        print('Combinator.__len__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__ne__()...', end=''); sys.stdout.flush()
        for i in range(len(ALLS)):
            x = ALLS[i]
            assert not(x != x), (i, x)
            for j in range(len(ALLS)):
                y = ALLS[j]
                if i != j:
                    assert y != x, (i, j, x, y)
                else:
                    assert not(y != x), (i, j, x, y)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__nonzero__()...', end=''); sys.stdout.flush()
        assert K.__nonzero__()
        assert not KI.__nonzero__()

        for x in ALLS_VARS:
            assert isinstance(x.__nonzero__(), bool), (type(x.__nonzero__()), x.__nonzero__())
            if x == K:
                assert x.__nonzero__()
            else:
                assert not x.__nonzero__()
        print('ok'); sys.stdout.flush()


        print('Combinator.__or__()...', end=''); sys.stdout.flush()
        assert (K  | K)()[0]  == K,  (K  | K,  (K  | K)())
        assert (K  | KI)()[0] == K,  (K  | KI, (K  | KI)())
        assert (KI | K)()[0]  == K,  (KI | K,  (KI | K)())
        assert (KI | KI)()[0] == KI, (KI | KI, (KI | KI)())

        assert (K  | True)()[0]  == K,  (K  | True,  (K  | True)())
        assert (K  | False)()[0] == K,  (K  | False, (K  | False)())
        assert (KI | True)()[0]  == K,  (KI | True,  (KI | True)())
        assert (KI | False)()[0] == KI, (KI | False, (KI | False)())

        for x in ALLS_VARS:
            for y in ALLS_VARS + (True, False):
                assert x | y == Combinator(Bor, x, y), (x, y, x | y, Combinator(Bor, x, y))
        print('ok'); sys.stdout.flush()


        print('Combinator.__repr__()...', end=''); sys.stdout.flush()
        assert repr(K) == "Combinator('K')", repr(K)
        assert repr(S) == "Combinator('S')", repr(S)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__str__()...', end=''); sys.stdout.flush()
        assert str(K) == 'K', str(K)
        assert str(S) == 'S', str(S)
        assert str(KK) == 'K K', str(KK)
        assert str(KS) == 'K S', str(KS)
        assert str(I) == 'S K K', str(I)
        assert str(B) == 'S (K S) K', str(B)
        assert str(M) == 'S (S K K) (S K K)', str(M)
        assert str(Y) == 'S (K (S (S K K) (S K K))) (S (S (K S) K) (K (S (S K K) (S K K))))', str(Y)
        assert K.__str__(allparent=True) == 'K', K.__str__(allparent=True)
        assert S.__str__(allparent=True) == 'S', S.__str__(allparent=True)
        assert KK.__str__(allparent=True) == '(K K)', KK.__str__(allparent=True)
        assert KS.__str__(allparent=True) == '(K S)', KS.__str__(allparent=True)
        assert I.__str__(allparent=True) == '((S K) K)', I.__str__(allparent=True)
        assert B.__str__(allparent=True) == '((S (K S)) K)', B.__str__(allparent=True)
        assert M.__str__(allparent=True) == '((S ((S K) K)) ((S K) K))', M.__str__(allparent=True)
        assert Y.__str__(allparent=True) == '((S (K ((S ((S K) K)) ((S K) K))))' \
               + ' ((S ((S (K S)) K)) (K ((S ((S K) K)) ((S K) K)))))', \
               Y.__str__(allparent=True)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.__xor__()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.atomic_is()...', end=''); sys.stdout.flush()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.bnot()...', end=''); sys.stdout.flush()
        assert K.bnot()()[0]  == KI, (K.bnot(),  KI.bnot()())
        assert KI.bnot()()[0] == K,  (KI.bnot(), K.bnot()())

        for x in ALLS_VARS:
            assert x.bnot() == Combinator(Bnot, x), (x, x.bnot(), Combinator(Bnot, x))
        print('ok'); sys.stdout.flush()


        print('Combinator.change_atom()...', end=''); sys.stdout.flush()
        assert K.change_atom(Atom_K, Atom_S) == S, K.change_atome(Atom_K, Atom_S)
        assert K.change_atom(Atom_K, S) == S, K.change_atome(Atom_K, S)
        assert K.change_atom(Atom('K'), S) == S, K.change_atome(Atom('K'), S)
        assert K.change_atom(Atom_K, Atom('S')) == S, K.change_atome(Atom_K, Atom('S'))
        assert K.change_atom(K[0], S) == S, K.change_atome(K[0], S)

        assert K.change_atom(Atom_K, Atom_K) == K, K.change_atome(Atom_K, Atom_K)
        assert K.change_atom(Atom_K, K) == K, K.change_atome(Atom_K, K)

        assert S.change_atom(Atom_S, Atom_K) == K, S.change_atome(Atom_S, Atom_K)
        assert S.change_atom(Atom_S, K) == K, S.change_atome(Atom_S, K)
        assert S.change_atom(Atom('S'), K) == K, S.change_atome(Atom('S'), K)
        assert S.change_atom(Atom_S, Atom('K')) == K, S.change_atome(Atom_S, Atom('K'))
        assert S.change_atom(S[0], Atom('K')) == K, S.change_atome(S[0], Atom('K'))

        assert S.change_atom(Atom_S, Atom_S) == S, S.change_atome(Atom_S, Atom_S)
        assert S.change_atom(Atom_S, S) == S, S.change_atome(Atom_S, S)

        assert str(M.change_atom(Atom_K, S)) == 'S (S S S) (S S S)', str(M.change_atom(Atom_K, S))
        x = M.change_atom(Atom_K, Vx)
        assert str(x) == 'S (S x x) (S x x)', str(x)
        assert str(x.change_atom(Atom_Vx, K)) == 'S (S K K) (S K K)', str(x.change_atom(Atom_Vx, K))
        for x in ALLS:
            if x not in (S, SS):
                y = x.change_atom(Atom_K, Vx)
                assert x != y, (x, y)
                y = y.change_atom(Vx[0], Vz)
                assert x != y, (x, y)
                y = y.change_atom(Vz[0], K)
                assert x == y, (x, y)

            if x not in (K, KK):
                y = x.change_atom(Atom_S, Vz)
                assert x != y, (x, y)
                y = y.change_atom(Vz[0], Vy)
                assert x != y, (x, y)
                y = y.change_atom(Vy[0], S)
                assert x == y, (x, y)
        print('ok'); sys.stdout.flush()


        print('Combinator.depth()...', end=''); sys.stdout.flush()
        assert K.depth() == 0, K.depth()
        assert S.depth() == 0, S.depth()
        assert KK.depth() == 1, KK.depth()
        assert KS.depth() == 1, KS.depth()
        assert SS.depth() == 1, SS.depth()
        assert SK.depth() == 1, SK.depth()

        assert B.depth() == 3, B.depth()
        assert C.depth() == 8, C.depth()
        assert I.depth() == 2, I.depth()
        assert iota.depth() == 6, iota.depth()
        assert KI.depth() == 3, KI.depth()
        assert L.depth() == 6, L.depth()
        assert M.depth() == 4, M.depth()
        assert O.depth() == 3, O.depth()
        assert Omega.depth() == 5, Omega.depth()
        assert R.depth() == 7, R.depth()
        assert T.depth() == 6, T.depth()
        assert V.depth() == 11, V.depth()
        assert W.depth() == 4, W.depth()
        assert Y.depth() == 7, Y.depth()

        assert Bnot.depth() == 7, Bnot.depth()
        assert Band.depth() == 7, Band.depth()
        assert Bor.depth() == 4, Bor.depth()
        assert Bimp.depth() == 6, Bimp.depth()
        assert Bnotrec.depth() == 5, Bnotrec.depth()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.n_to()...', end=''); sys.stdout.flush()
        for n in range(30):
            natural_sys = BARENDREGT
            assert Combinator.n_to(n) == Combinator.n_to_Barendregt(n), \
                   (Combinator.n_to(n), Combinator.n_to_Barendregt(n))

            x = Combinator.n_to(n)
            assert x == Combinator(n), (n, x, Combinator(n))
            assert n == int(x), (n, int(x), x)

            natural_sys = CHURCH
            assert Combinator.n_to(n) == Combinator.n_to_Church(n), \
                   (Combinator.n_to(n), Combinator.n_to_Church(n))

            x = Combinator.n_to(n)
            assert x == Combinator(n), (n, x, Combinator(n))
            assert n == int(x), (n, int(x), x)
        print('ok'); sys.stdout.flush()


        print('Combinator.n_to_Barendregt()...', end=''); sys.stdout.flush()
        assert Combinator.n_to_Barendregt(0) == I, (Combinator.n_to_Barendregt(0), I)
        assert Combinator.n_to_Barendregt(1) == Combinator(V, KI, I), \
               (Combinator.n_to_Barendregt(1), Combinator(V, KI, I))
        assert Combinator.n_to_Barendregt(2) == Combinator(V, KI, Combinator(V, KI, I)), \
               (Combinator.n_to_Barendregt(2), Combinator(V, KI, Combinator(V, KI, I)))
        c = I
        for n in range(30):
            natural_sys = BARENDREGT
            assert Combinator.n_to_Barendregt(n) == c, (n, Combinator.n_to_Barendregt(n), c)
            if n == 0:
                assert Combinator(NBzero_is, c)()[0] == K, (n, Combinator.n_to_Barendregt(n), c)
            else:
                assert Combinator(NBzero_is, c)()[0] == KI, (n, Combinator.n_to_Barendregt(n), c)

            natural_sys = CHURCH
            assert Combinator.n_to_Barendregt(n) == c, (n, Combinator.n_to_Barendregt(n), c)

            c = Combinator(V, KI, c)
        print('ok'); sys.stdout.flush()


        print('Combinator.n_to_Church()...', end=''); sys.stdout.flush()
        assert Combinator.n_to_Church(0) == KI, (Combinator.n_to_Church(0), KI)
        assert Combinator.n_to_Church(1) == Combinator(S, B, KI), \
               (Combinator.n_to_Church(1), Combinator(S, B, KI))
        assert Combinator.n_to_Church(2) == Combinator(S, B, Combinator(S, B, KI)), \
               (Combinator.n_to_Church(2), Combinator(S, B, Combinator(S, B, KI)))
        c = KI
        for n in range(30):
            natural_sys = BARENDREGT
            assert Combinator.n_to_Church(n) == c, (n, Combinator.n_to_Church(n), c)

            natural_sys = CHURCH
            assert Combinator.n_to_Church(n) == c, (n, Combinator.n_to_Church(n), c)

            c = Combinator(S, B, c)
        print('ok'); sys.stdout.flush()


        print('Combinator.nb_atom()...', end=''); sys.stdout.flush()
        assert K.nb_atom() == 1, K.nb_atom()
        assert S.nb_atom() == 1, S.nb_atom()
        assert KK.nb_atom() == 2, KK.nb_atom()
        assert KS.nb_atom() == 2, KS.nb_atom()
        assert SS.nb_atom() == 2, SS.nb_atom()
        assert SK.nb_atom() == 2, SK.nb_atom()

        assert B.nb_atom() == 4, B.nb_atom()
        assert C.nb_atom() == 10, C.nb_atom()
        assert I.nb_atom() == 3, I.nb_atom()
        assert KI.nb_atom() == 4, KI.nb_atom()
        assert M.nb_atom() == 7, M.nb_atom()
        assert O.nb_atom() == 4, O.nb_atom()
        assert Omega.nb_atom() == 14, Omega.nb_atom()
        assert T.nb_atom() == 7, T.nb_atom()
        assert W.nb_atom() == 6, W.nb_atom()

        assert Bnot.nb_atom() == 12, Bnot.nb_atom()
        assert Band.nb_atom() == 11, Band.nb_atom()
        assert Bor.nb_atom() == 6, Bor.nb_atom()
        assert Bimp.nb_atom() == 8, Bimp.nb_atom()
        assert Bnotrec.nb_atom() == 9, Bnotrec.nb_atom()

        for x in ALLS:
            if x.atomic_is():
                assert x.nb_atom() == 1, (x.nb_atom(), x)
            else:
                assert x.nb_atom() > 1, (x.nb_atom(), x)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.stable_is()...', end=''); sys.stdout.flush()
        for x in STABLES + VARS:
            assert x.stable_is()
        for x in UNSTABLES:
            assert not x.stable_is()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Combinator.str_to()...', end=''); sys.stdout.flush()
        assert Combinator.str_to('K') == K, Combinator.str_to('K')
        assert Combinator.str_to('S') == S, Combinator.str_to('S')

        assert Combinator.str_to('K K') == KK, Combinator.str_to('K K')
        assert Combinator.str_to('K S') == KS, Combinator.str_to('K S')
        assert Combinator.str_to('S K') == SK, Combinator.str_to('S K')
        assert Combinator.str_to('S S') == SS, Combinator.str_to('S S')

        assert Combinator.str_to('S K K') == I, Combinator.str_to('S K K')

        assert Combinator.str_to('S (S (K (S (K S) K)) S) (K K)') == C, \
               (Combinator.str_to('S (S (K (S (K S) K)) S) (K K)'), C)

        assert Combinator.str_to(' S (K (S(S K K)(S K K))) '
                                 + '(S (S(K S) K)(K (S (S K K)(S K K)) )) ') == Y, \
               (Combinator.str_to(' S (K (S(S K K)(S K K))) '
                                  + '(S (S(K S) K)(K (S (S K K)(S K K)) )) '), Y)

        for x in (ALLS if debug.assertspeed >= debug.ASSERT_NORMAL else BASICS):
            assert Combinator.str_to(str(x)) == x, (x, Combinator.str_to(str(x)))
            assert Combinator.str_to('({0})'.format(x)) == x, (x, '({0})'.format(x))
            for y in (ALLS if debug.assertspeed > debug.ASSERT_NORMAL else BASICS):
                assert Combinator.str_to('{0}({1})'.format(x, y)) \
                       == Combinator(Combinator.str_to(str(x)), Combinator.str_to(str(y))), \
                       (x, y, Combinator.str_to('{0}({1})'.format(x, y)))
                assert Combinator.str_to('( (({0}) )  ({1}))'.format(x, y)) \
                       == Combinator(Combinator.str_to(str(x)), Combinator.str_to(str(y))), \
                       (x, y, Combinator.str_to('( (({0}) )  ({1}))'.format(x, y)))
        if debug.assertspeed <= debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('???', end='')
        print('ok'); sys.stdout.flush()



        print()
        print('Kxy -> x ...', end=''); sys.stdout.flush()
        assert func_K(Atom('z'), (Vx, ), None) == (Combinator(Atom('z'), Vx), 0), \
               func_K(Atom('z'), (Vx, ), None)
        assert func_K(Atom('z'), (Vx, Vy), None) == (Vx, 1), \
               func_K(Atom('z'), (Vx, Vy), None)

        assert Combinator(K, Vx)() == (Combinator(K, Vx), 0), Combinator(K, Vx)()
        assert Combinator(K, Vx, Vy)() == (Vx, 1), Combinator(K, Vx, Vy)()
        print('ok'); sys.stdout.flush()


        print('Sxyz -> xz(yz) ...', end=''); sys.stdout.flush()
        assert func_S(Atom('z'), (Vx, ), None) == (Combinator(Atom('z'), Vx), 0), \
               func_S(Atom('z'), (Vx, ), None)
        assert func_S(Atom('z'), (Vx, Vy), None) == (Combinator(Atom('z'), Vx, Vy), 0), \
               func_S(Atom('z'), (Vx, Vy), None)
        assert func_S(Atom('z'), (Vx, Vy, Vz), None) \
               == (Combinator(Vx, Vz, Combinator(Vy, Vz)), 1), \
               func_S(Atom('z'), (Vx, Vy, Vz), None)

        assert Combinator(S, Vx)() == (Combinator(S, Vx), 0), Combinator(S, Vx)()
        assert Combinator(S, Vx, Vy)() == (Combinator(S, Vx, Vy), 0), Combinator(S, Vx, Vy)()
        assert Combinator(S, Vx, Vy, Vz)() == (Combinator(Vx, Vz, Combinator(Vy, Vz)), 1), \
               Combinator(S, Vx, Vy, Vz)()
        print('ok'); sys.stdout.flush()


        print('Bxyz -> x(yz) ...', end=''); sys.stdout.flush()
        assert func_B(Atom('z'), (Vx, ), None) == (Combinator(Atom('z'), Vx), 0), \
               func_B(Atom('z'), (Vx, ), None)
        assert func_B(Atom('z'), (Vx, Vy), None) == (Combinator(Atom('z'), Vx, Vy), 0), \
               func_B(Atom('z'), (Vx, Vy), None)
        assert func_B(Atom('z'), (Vx, Vy, Vz), None) == (Combinator(Vx, Combinator(Vy, Vz)), 1), \
               func_B(Atom('z'), (Vx, Vy, Vz), None)

        assert Combinator(B, Vx, Vy, Vz)()[0] == Combinator(Vx, Combinator(Vy, Vz)), \
               Combinator(B, Vx, Vy, Vz)()
        print('ok'); sys.stdout.flush()


        print('Cxyz -> xzy ...', end=''); sys.stdout.flush()
        assert func_C(Atom('z'), (Vx, ), None) == (Combinator(Atom('z'), Vx), 0), \
               func_C(Atom('z'), (Vx, ), None)
        assert func_C(Atom('z'), (Vx, Vy), None) == (Combinator(Atom('z'), Vx, Vy), 0), \
               func_C(Atom('z'), (Vx, Vy), None)
        assert func_C(Atom('z'), (Vx, Vy, Vz), None) == (Combinator(Vx, Vz, Vy), 1), \
               func_C(Atom('z'), (Vx, Vy, Vz), None)

        assert Combinator(C, Vx, Vy, Vz)()[0] == Combinator(Vx, Vz, Vy), Combinator(C, Vx, Vy, Vz)()
        print('ok'); sys.stdout.flush()


        print('Ix -> x ...', end=''); sys.stdout.flush()
        assert func_I(Atom('z'), (Vx, ), None) == (Vx, 1), \
               func_I(Atom('z'), (Vx, ), None)

        assert Combinator(I, Vx)()[0] == Vx, Combinator(I, Vx)()
        print('ok'); sys.stdout.flush()


        print('iota x -> xSK ...', end=''); sys.stdout.flush()
        assert Combinator(iota, Vx)()[0] == Combinator(Vx, S, K), Combinator(iota, Vx)()
        print('ok'); sys.stdout.flush()


        print('Lxy -> x(yy) ...', end=''); sys.stdout.flush()
        assert Combinator(L, Vx, Vy)()[0] == Combinator(Vx, Combinator(Vy, Vy)), \
               Combinator(L, Vx, Vy)()
        print('ok'); sys.stdout.flush()


        print('Mx -> xx ...', end=''); sys.stdout.flush()
        assert Combinator(M, Vx)()[0] == Combinator(Vx, Vx), \
               Combinator(M, Vx)()
        print('ok'); sys.stdout.flush()


        print('Oxy -> y(xy) ...', end=''); sys.stdout.flush()
        assert Combinator(O, Vx, Vy)()[0] == Combinator(Vy, Combinator(Vx, Vy)), \
               Combinator(O, Vx, Vy)()
        print('ok'); sys.stdout.flush()


        print('Omega x -> Omega x ...', end=''); sys.stdout.flush()
##         assert Combinator(Omega, Vx)()[0] == Combinator(Omega, Vx), \
##                Combinator(Omega, Vx)()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Rxyz -> yzx ...', end=''); sys.stdout.flush()
        assert Combinator(R, Vx, Vy, Vz)()[0] == Combinator(Vy, Vz, Vx), Combinator(R, Vx, Vy, Vz)()
        print('ok'); sys.stdout.flush()


        print('Txy -> yx ...', end=''); sys.stdout.flush()
        assert Combinator(T, Vx, Vy)()[0] == Combinator(Vy, Vx), Combinator(T, Vx, Vy)()
        print('ok'); sys.stdout.flush()


        print('Uxy -> y(xxy) ...', end=''); sys.stdout.flush()
        assert Combinator(U, Vx, Vy)()[0] == Combinator(Vy, Combinator(Vx, Vx, Vy)), \
               Combinator(U, Vx, Vy)()
        print('ok'); sys.stdout.flush()


        print('Vxyz -> zxy ...', end=''); sys.stdout.flush()
        assert Combinator(V, Vx, Vy, Vz)()[0] == Combinator(Vz, Vx, Vy), Combinator(V, Vx, Vy, Vz)()
        print('ok'); sys.stdout.flush()


        print('Wxy -> xyy ...', end=''); sys.stdout.flush()
        assert func_W(Atom('z'), (Vx, ), None) == (Combinator(Atom('z'), Vx), 0), \
               func_W(Atom('z'), (Vx, ), None)
        assert func_W(Atom('z'), (Vx, Vy), None) == (Combinator(Vx, Vy, Vy), 1), \
               func_W(Atom('z'), (Vx, Vy), None)

        assert Combinator(W, Vx, Vy)()[0] == Combinator(Vx, Vy, Vy), Combinator(W, Vx, Vy)()
        print('ok'); sys.stdout.flush()


        print('Yx -> x(Yx) ...', end=''); sys.stdout.flush()
##         assert Combinator(Y, Vx)()[0] == Combinator(Vx, Combinator(Y, Vx)), \
##                Combinator(Y, Vx)()
        print('???', end='')
        print('ok'); sys.stdout.flush()



        # Teste l'valuation de divers combinateur
        print()
        print('Evaluation of some combinators...', end='')
        assert Combinator(C, I, Vx, Vy)()[0] == Combinator(T, Vx, Vy)()[0]

        assert Combinator(B,
                          Combinator(B, W),
                          Combinator(B, B, C),
                          Vx,
                          Vy,
                          Vz)()[0] \
                          == Combinator(S, Vx, Vy, Vz)()[0]

        assert Combinator(iota, SK)()[0] == K

        assert Combinator(iota,
                          Combinator(iota,
                                     Combinator(iota, iota)))()[0] == K

        assert Combinator(iota, K)()[0] == S

        assert Combinator(iota,
                          Combinator(iota,
                                     Combinator(iota,
                                                Combinator(iota, iota))))()[0] == S
        print('ok'); sys.stdout.flush()


        # Teste l'valuation par tapes
        print('Evaluation by step...', end='')
        for x in STABLES:
            if x in (Y, ) + VARS:
                continue
            x = Combinator(x, Vx, Vy, Vz)
            x_finish, nb = x()
            assert n > 0, (nb, x, x_finish)
            assert x != x_finish, (nb, x, x_finish)

            x_step = Combinator(x)
            for i in range(nb):
                x_step, nb_step = x_step(nb=1)
                assert nb_step == 1, (x, i, x_step, nb_step, nb, x_finish)

            assert x_step == x_finish, (x, x_step, nb, x_finish)
            assert x_step(nb=1) == (x_finish, 0), (x, x_step(nb=1), nb, x_finish)
            assert x_step() == (x_finish, 0), (x, x_step(), nb, x_finish)

            for k in range(1, nb * 2):
                x_step = Combinator(x)
                nb_total = 0
                for i in range((nb + k - 1)//k):
                    x_step, nb_step = x_step(nb=k)
                    assert 0 < nb_step <= k, (x, i, x_step, nb_step, nb, x_finish)
                    nb_total += nb_step

                assert nb_total == nb,  (x, x_step, nb_total, nb, x_finish)
                assert x_step == x_finish, (x, x_step, nb, x_finish)
                assert x_step(nb=1) == (x_finish, 0), (x, x_step(nb=1), nb, x_finish)
                assert x_step() == (x_finish, 0), (x, x_step(), nb, x_finish)
        print('ok'); sys.stdout.flush()


        # Vrifie que les "constantes" n'ont pas t modifies
        assert str(K) == 'K', str(K)
        assert str(S) == 'S', str(S)
        assert str(KK) == 'K K', str(KK)
        assert str(KS) == 'K S', str(KS)
        assert str(SS) == 'S S', str(SS)
        assert str(SK) == 'S K', str(SK)

        assert str(B) == 'S (K S) K', str(B)
        assert str(C) == 'S (S (K (S (K S) K)) S) (K K)', str(C)
        assert str(I) == 'S K K', str(I)
        assert str(iota) == 'S (S (S K K) (K S)) (K K)', str(iota)
        assert str(KI) == 'K (S K K)', str(KI)
        assert str(L) == 'S (S (K S) K) (K (S (S K K) (S K K)))', str(L)
        assert str(M) == 'S (S K K) (S K K)', str(M)
        assert str(O) == 'S (S K K)', str(O)
        assert str(Omega) == 'S (S K K) (S K K) (S (S K K) (S K K))', str(Omega)
        assert str(R) == 'S (K (S (K S) K)) (S (K (S (S K K))) K)', str(R)
        assert str(T) == 'S (K (S (S K K))) K', str(T)
        assert str(U) == 'S (K (S (S K K))) (S (S K K) (S K K))', str(U)
        assert str(V) == 'S (K (S (S (K (S (K S) K)) S) (K K))) (S (K (S (S K K))) K)', str(V)
        assert str(W) == 'S S (K (S K K))', str(W)
        assert str(Y) == 'S (K (S (S K K) (S K K))) (S (S (K S) K) (K (S (S K K) (S K K))))', str(Y)

        assert str(Bnot) == 'S (S (S K K) (K (K (S K K)))) (K K)', str(Bnot)
        assert str(Bnot_my) == 'S (S (S K K) (S (S K K) (K (S K K)))) (K K)', str(Bnot_my)
        assert str(Band) == 'S (K (S (S K K) (K (K (S K K)))))', str(Band)
        assert str(Bor) == 'S (S K K) (K K)', str(Bor)
        assert str(Bimp) == 'S (K (S (S K K) (K K)))', str(Bimp)
        assert str(Bnotrec) == 'S (S K K) (K (K (S K K)))', str(Bnotrec)

        assert str(NBzero_is) == 'S (K (S (S K K))) K K', str(NBzero_is)
        assert str(NBsucc) == 'S (K (S (S (K (S (K S) K)) S) (K K))) (S (K (S (S K K))) K) ' \
               + '(K (S K K))', str(NBsucc)
        assert str(NBprev) == 'S (K (S (S K K))) K (K (S K K))', str(NBprev)

        assert str(NCsucc) == 'S (S (K S) K)', str(NCsucc)
        assert str(NCadd) == 'S (K S) K S (S (K S) K (S (K S) K))', str(NCadd)
##         assert str(NBadd) == '???', str(NBadd)

        assert str(Vx) == 'x', str(Vx)
        assert str(Vy) == 'y', str(Vy)
        assert str(Vz) == 'z', str(Vz)
        debug.test_end()

    main_test()
##\endcond MAINTEST
