#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.bit32 Bits de naturels sur 32 bits

##\file
# Bits de naturels sur 32 bits

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 30 juillet 2006
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'bit32 --- 2010 March 16'

import DSPython



# ############
# Constantes #
##############
## Reprsentation binaire des naturels tenant sur 8 bits
BIN_8_TUPLE = ('0',        '1',        '10',       '11',
               '100',      '101',      '110',      '111',
               '1000',     '1001',     '1010',     '1011',
               '1100',     '1101',     '1110',     '1111',
               '10000',    '10001',    '10010',    '10011',
               '10100',    '10101',    '10110',    '10111',
               '11000',    '11001',    '11010',    '11011',
               '11100',    '11101',    '11110',    '11111',
               '100000',   '100001',   '100010',   '100011',
               '100100',   '100101',   '100110',   '100111',
               '101000',   '101001',   '101010',   '101011',
               '101100',   '101101',   '101110',   '101111',
               '110000',   '110001',   '110010',   '110011',
               '110100',   '110101',   '110110',   '110111',
               '111000',   '111001',   '111010',   '111011',
               '111100',   '111101',   '111110',   '111111',
               '1000000',  '1000001',  '1000010',  '1000011',
               '1000100',  '1000101',  '1000110',  '1000111',
               '1001000',  '1001001',  '1001010',  '1001011',
               '1001100',  '1001101',  '1001110',  '1001111',
               '1010000',  '1010001',  '1010010',  '1010011',
               '1010100',  '1010101',  '1010110',  '1010111',
               '1011000',  '1011001',  '1011010',  '1011011',
               '1011100',  '1011101',  '1011110',  '1011111',
               '1100000',  '1100001',  '1100010',  '1100011',
               '1100100',  '1100101',  '1100110',  '1100111',
               '1101000',  '1101001',  '1101010',  '1101011',
               '1101100',  '1101101',  '1101110',  '1101111',
               '1110000',  '1110001',  '1110010',  '1110011',
               '1110100',  '1110101',  '1110110',  '1110111',
               '1111000',  '1111001',  '1111010',  '1111011',
               '1111100',  '1111101',  '1111110',  '1111111',
               '10000000', '10000001', '10000010', '10000011',
               '10000100', '10000101', '10000110', '10000111',
               '10001000', '10001001', '10001010', '10001011',
               '10001100', '10001101', '10001110', '10001111',
               '10010000', '10010001', '10010010', '10010011',
               '10010100', '10010101', '10010110', '10010111',
               '10011000', '10011001', '10011010', '10011011',
               '10011100', '10011101', '10011110', '10011111',
               '10100000', '10100001', '10100010', '10100011',
               '10100100', '10100101', '10100110', '10100111',
               '10101000', '10101001', '10101010', '10101011',
               '10101100', '10101101', '10101110', '10101111',
               '10110000', '10110001', '10110010', '10110011',
               '10110100', '10110101', '10110110', '10110111',
               '10111000', '10111001', '10111010', '10111011',
               '10111100', '10111101', '10111110', '10111111',
               '11000000', '11000001', '11000010', '11000011',
               '11000100', '11000101', '11000110', '11000111',
               '11001000', '11001001', '11001010', '11001011',
               '11001100', '11001101', '11001110', '11001111',
               '11010000', '11010001', '11010010', '11010011',
               '11010100', '11010101', '11010110', '11010111',
               '11011000', '11011001', '11011010', '11011011',
               '11011100', '11011101', '11011110', '11011111',
               '11100000', '11100001', '11100010', '11100011',
               '11100100', '11100101', '11100110', '11100111',
               '11101000', '11101001', '11101010', '11101011',
               '11101100', '11101101', '11101110', '11101111',
               '11110000', '11110001', '11110010', '11110011',
               '11110100', '11110101', '11110110', '11110111',
               '11111000', '11111001', '11111010', '11111011',
               '11111100', '11111101', '11111110', '11111111')


## Nombre de Mersenne M<sub>8</sub> == 2<sup>8</sup>
MERSENNE8  = 255
## Nombre de Mersenne M<sub>16</sub> == 2<sup>16</sup>
MERSENNE16 = 65535
## Nombre de Mersenne M<sub>24</sub> == 2<sup>24</sup>
MERSENNE24 = 16777215
## Nombre de Mersenne M<sub>32</sub> == 2<sup>32</sup>
MERSENNE32 = 4294967295


## Nombres de Mersenne M<sub>n</sub> tenant sur 32 bits
MERSENNE_TUPLE = (0,                  1,          3,          7,
                  15,                31,         63,        127,
                  MERSENNE8,        511,       1023,       2047,
                  4095,            8191,      16383,      32767,
                  MERSENNE16,    131071,     262143,     524287,
                  1048575,      2097151,    4194303,    8388607,
                  MERSENNE24,  33554431,   67108863,  134217727,
                  268435455,  536870911, 1073741823, 2147483647,
                  MERSENNE32)


## Nombre de bits  1 pour les naturels tenant sur 8 bits
NBBITS1_8_TUPLE = (0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
                   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
                   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
                   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
                   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
                   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
                   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
                   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
                   1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
                   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
                   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
                   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
                   2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
                   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
                   3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
                   4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8)


## Nombre de bits pour les naturels tenant sur 8 bits
NBBITS_8_TUPLE = (0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4,
                  5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
                  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                  6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                  7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
                  8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8)


## 2<sup>k</sup>, puissances de 2 tenant sur 32 bits
POW2_TUPLE = (1,                 2,          4,          8,
              16,               32,         64,        128,
              256,             512,       1024,       2048,
              4096,           8192,      16384,      32768,
              65536,        131072,     262144,     524288,
              1048576,     2097152,    4194304,    8388608,
              16777216,   33554432,   67108864,  134217728,
              268435456, 536870912, 1073741824, 2147483648)


## Indice du dernier bit  0 pour les naturels tenant sur 8 bits
RSCAN0_8_TUPLE = (None, None, 0, None, 1, 1, 0, None, 2, 2, 2, 2, 1, 1, 0, None,
                  3,    3,    3, 3,    3, 3, 3, 3,    2, 2, 2, 2, 1, 1, 0, None,
                  4,    4,    4, 4,    4, 4, 4, 4,    4, 4, 4, 4, 4, 4, 4, 4,
                  3,    3,    3, 3,    3, 3, 3, 3,    2, 2, 2, 2, 1, 1, 0, None,
                  5,    5,    5, 5,    5, 5, 5, 5,    5, 5, 5, 5, 5, 5, 5, 5,
                  5,    5,    5, 5,    5, 5, 5, 5,    5, 5, 5, 5, 5, 5, 5, 5,
                  4,    4,    4, 4,    4, 4, 4, 4,    4, 4, 4, 4, 4, 4, 4, 4,
                  3,    3,    3, 3,    3, 3, 3, 3,    2, 2, 2, 2, 1, 1, 0, None,
                  6,    6,    6, 6,    6, 6, 6, 6,    6, 6, 6, 6, 6, 6, 6, 6,
                  6,    6,    6, 6,    6, 6, 6, 6,    6, 6, 6, 6, 6, 6, 6, 6,
                  6,    6,    6, 6,    6, 6, 6, 6,    6, 6, 6, 6, 6, 6, 6, 6,
                  6,    6,    6, 6,    6, 6, 6, 6,    6, 6, 6, 6, 6, 6, 6, 6,
                  5,    5,    5, 5,    5, 5, 5, 5,    5, 5, 5, 5, 5, 5, 5, 5,
                  5,    5,    5, 5,    5, 5, 5, 5,    5, 5, 5, 5, 5, 5, 5, 5,
                  4,    4,    4, 4,    4, 4, 4, 4,    4, 4, 4, 4, 4, 4, 4, 4,
                  3,    3,    3, 3,    3, 3, 3, 3,    2, 2, 2, 2, 1, 1, 0, None)


## Indice du premier bit  0 pour les naturels impairs tenant sur 8 bits
SCAN0_ODD8_TUPLE = (1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 5,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 6,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 5,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 7,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 5,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 6,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 5,
                    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1, 8)


## Indice du premier bit  1 pour les naturels pairs tenant sur 8 bits, ou None
SCAN1_EVEN8_TUPLE = (None, 1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     5,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     6,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     5,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     7,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     5,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     6,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1,
                     5,    1, 2, 1, 3, 1, 2, 1, 4, 1, 2, 1, 3, 1, 2, 1)



# ###########
# Fonctions #
#############
## n en binaire dans un string
def bin(n):
    """Renvoie un string reprsentant n en binaire
      (Si n == 0 alors renvoie '0')

    Pre: n naturel < 2**32

    Result: string de 1 <= longueur <= 32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    if n <= MERSENNE8:
        return BIN_8_TUPLE[n]
    else:
        l = [BIN_8_TUPLE[n & MERSENNE8]]
        while n > MERSENNE8:
            n >>= 8
            if len(l[0]) < 8:  # si faut complter l'lment prcdant avec des '0'
                l.insert(0, '0' * (8 - len(l[0])))
            l.insert(0, BIN_8_TUPLE[n & MERSENNE8])
        return ''.join(l)


## Le bit d'indice i de n vaut 1 ?
def bit(n, i):
    """Renvoie True si le bit d'indice i de n vaut 1,
      False sinon

    Pre: n naturel < 2**32
         i naturel < 32

    Result: bool

    O(n) = 1"""
    assert DSPython.nat32_is(n), n
    assert DSPython.natural_is(i), i
    assert i < 32, i

    return bool(n & (1 << i))


## n avec son bit d'indice i  b
def bitset(n, i, b):
    """Si b alors renvoie n avec son bit d'indice i  1,
      sinon renvoie n avec son bit d'indice i  0

    Pre: n naturel < 2**32
         i naturel < 32

    Result: naturel < 2**32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n
    assert DSPython.natural_is(i), i
    assert i < 32, i

    return (n | (1 << i) if b
            else n & ~(1 << i))


## n avec son bit d'indice i  0
def bitset0(n, i):
    """Renvoie n avec son bit d'indice i  0

    Pre: n naturel < 2**32
         i naturel < 32

    Result: naturel < 2**32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n
    assert DSPython.natural_is(i), i
    assert i < 32, i

    return n & ~(1 << i)


## n avec son bit d'indice i  1
def bitset1(n, i):
    """Renvoie n avec son bit d'indice i  1

    Pre: n naturel < 2**32
         i naturel < 32

    Result: naturel < 2**32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n
    assert DSPython.natural_is(i), i
    assert i < 32, i

    return n | (1 << i)


## lg(n) == log<sub>2</sub>(n) == nbbits(n) - 1 == l'indice du dernier bit  1 de n
## &nbsp; &nbsp; (n != 0)
def lg(n):
    """Renvoie lg(n) == log_2(n) == nbbits(n) - 1
      == l'indice du dernier bit  1 de n

    Pre: n: 1 <= naturel < 2**32

    Result: naturel < 32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    return nbbits(n) - 1


## M<sub>k</sub> == 2<sup>k</sup> - 1
def mersenne(k):
    """Renvoie le nime nombre de Mersenne M_k == 2**k - 1

    Pre: k: naturel <= 32

    Result: naturel < 2**32

    O(k) = 1"""
    assert DSPython.natural_is(k), k
    assert k <= 32, k

    return MERSENNE_TUPLE[k]


## n est un nombre de Mersenne ?
def mersenne_is(n):
    """Renvoie True si n est un nombre de Mersenne,
      False sinon

    Pre: n: naturel < 2**32

    Result: bool

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    return MERSENNE_TUPLE[scan0(n)] == n


## Indice du nombre de Mersenne correspondant  n, ou None
def mersenne_to_index(n):
    """Si n == M_k alors renvoie k
                sinon renvoie None

    Pre: n: naturel < 2**32

    Result: naturel <= 32 ou None

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    i = scan0(n)
    return (i if MERSENNE_TUPLE[i] == n
            else None)


## Nombre de bits  0 de n
def nbbits0(n):
    """Renvoie le nombre de bits  0 de n
      (sans tenir compte des 0 aprs le dernier 1)
      (Si n == 0 alors renvoie 0)

    Pre: n: naturel < 2**32

    Result: naturel <= 31

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    return nbbits(n) - nbbits1(n)


## Nombre de bits  1 de n
def nbbits1(n):
    """Renvoie le nombre de bits  1 de n
      (Si n == 0 alors renvoie 0)

    Pre: n: naturel < 2**32

    Result: naturel <= 32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    return (NBBITS1_8_TUPLE[n >> 24] + NBBITS1_8_TUPLE[(n >> 16) & MERSENNE8]
            + NBBITS1_8_TUPLE[(n >> 8) & MERSENNE8] + NBBITS1_8_TUPLE[n & MERSENNE8])


## Nombre de bits de n
def nbbits(n):
    """Renvoie le nombre de bits de n
      == nbbits0(n) + nbbits1(n)
      (sans tenir compte des 0 aprs le dernier 1)
      (Si n == 0 alors renvoie 0)

    Pre: n: naturel < 2**32

    Result: naturel <= 32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    h = n >> 16
    if h != 0:
        if (h>>8) != 0:  # 2**24 <= n < 2**32
            return NBBITS_8_TUPLE[h >> 8] + 24
        else:            # 2**16 <= n < 2**24
            return NBBITS_8_TUPLE[h & MERSENNE8] + 16
    else:
        if (n>>8) != 0:  # 2**8  <= n < 2*16
            return NBBITS_8_TUPLE[n>>8 & MERSENNE8] + 8
        else:            # 0     <= n < 2**8
            return NBBITS_8_TUPLE[n & MERSENNE8]


## 2<sup>k</sup>
def pow2(k):
    """Renvoie la kime puissance de 2 : 2**k

    Pre: k naturel <= 31

    Result: 1 <= naturel <= 2**31

    O(k) = 1"""
    assert DSPython.natural_is(k), k
    assert k <= 31, k

    return POW2_TUPLE[k]


## n est une puissance de 2 ?
def pow2_is(n):
    """Renvoie True si n est une puissance de 2,
      False sinon

    Pre: n: naturel < 2**32

    Result: bool

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    return (n != 0) and (POW2_TUPLE[rscan1(n)] == n)


## Exposant de la puissance de Mersenne correspondant  n, ou None
def pow2_to_index(n):
    """Si n == 2^k alors renvoie k
                 sinon renvoie None

    Pre: n: naturel < 2**32

    Result: naturel <= 32 ou None

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    i = rscan1(n)
    return (i if (n != 0) and (POW2_TUPLE[i] == n)
            else None)


## Indice du dernier bit  0 de n, ou None
def rscan0(n):
    """Renvoie l'indice du dernier bit  0 de n
      ou None si pas de bit  0 contenu dans n

    Pre: n: naturel < 2**32

    Result: naturel <= 32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    h = n >> 16
    if (h&MERSENNE16 != 0) and (h != MERSENNE16):
        if (h>>8 != 0) and (h>>8 != MERSENNE8):
            return RSCAN0_8_TUPLE[h >> 8] + 24
        else:
            return RSCAN0_8_TUPLE[h & MERSENNE8] + 16
    else:
        n &= MERSENNE16
        if (n>>8 != 0) and (h>>8 != MERSENNE8):
            return RSCAN0_8_TUPLE[n >> 8] + 8
        else:
            return RSCAN0_8_TUPLE[n & MERSENNE8]


## Indice du dernier bit  1 de, n == nbbits(n) - 1 ou None
def rscan1(n):
    """Renvoie l'indice du dernier bit  1 de n
      ou None si n == 0

    Pre: n: naturel < 2**32

    Result: naturel <= 31 ou None

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    return (nbbits(n) - 1 if n > 0
            else None)


## Indice du premier bit  0 de n
def scan0(n):
    """Renvoie l'indice du premier bit  0 de n
      (si n == 0         alors renvoie 0)
      (si n == 2**32 - 1 alors renvoie 32)

    Pre: n: naturel < 2**32

    Result: naturel <= 32

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    if n < MERSENNE32:  # n contient au moins un 0
        if (n & MERSENNE16) != MERSENNE16:  # n contient au moins un 0 dans les 16 premiers bits
            if (n & MERSENNE8) != MERSENNE8:  # n contient au moins un 0 dans les 8 premiers bits
                k = 0
            else:
                k = 8
                n >>= 8
        else:                               # n contient au moins un 0 dans les 16 derniers bits
            if (n & (MERSENNE8 << 16)) != (MERSENNE8 << 16):
                k = 16
                n >>= 16
            else:                             # n contient au moins un 0 dans les 8 derniers bits
                k = 24
                n >>= 24
        return (k if n&1 == 0
                else k + SCAN0_ODD8_TUPLE[(n & MERSENNE8) >> 1])
    else:               # n == 11111111111111111111111111111111b
        return 32


## Indice du premier bit  1 de n, ou None
def scan1(n):
    """Renvoie l'indice du premier bit  1 de n
      ou None si n == 0

    Pre: n: naturel < 2**32

    Result: naturel < 32 ou None

    O(n) = 1"""
    assert DSPython.nat32_is(n), n

    if n > 0:  # n contient au moins un 1
        if (n & MERSENNE16) != 0:  # n contient au moins un 1 dans les 16 premiers bits
            if (n & MERSENNE8) != 0:  # n contient au moins un 1 dans les 8 premiers bits
                k = 0
            else:
                k = 8
                n >>= 8
        else:                      # n contient au moins un 1 dans les 16 derniers bits
            if (n & (MERSENNE8 << 16)) != 0:
                k = 16
                n >>= 16
            else:                     # n contient au moins un 1 dans les 8 derniers bits
                k = 24
                n >>= 24
        return (k if n&1 != 0
                else k + SCAN1_EVEN8_TUPLE[(n & MERSENNE8) >> 1])
    else:      # n == 0
        return None



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        import DSPython.debug as debug

        try:
            if not 'profile' in dir():
                import psyco
                psyco.full()
        except ImportError:
            pass

        debug.test_begin(VERSION, __debug__)

        assert len(BIN_8_TUPLE) == 256, len(BIN_8_TUPLE)
        assert len(MERSENNE_TUPLE) == 33, len(MERSENNE_TUPLE)
        assert len(NBBITS1_8_TUPLE) == 256, len(NBBITS1_8_TUPLE)
        assert len(NBBITS_8_TUPLE) == 256, len(NBBITS_8_TUPLE)
        assert len(POW2_TUPLE) == 32, len(POW2_TUPLE)
        assert len(RSCAN0_8_TUPLE) == 256, len(RSCAN0_8_TUPLE)
        assert len(SCAN0_ODD8_TUPLE) == 128, len(SCAN0_ODD8_TUPLE)
        assert len(SCAN1_EVEN8_TUPLE) == 128, len(SCAN1_EVEN8_TUPLE)


        print('bin()...', end=''); sys.stdout.flush()
        assert bin(0) == '0', bin(0)
        assert bin(1) == '1', bin(1)
        for i in range(1, 31):
            assert bin(2**i - 1) == '1'*i, (i, bin(2**i - 1))                    #  1...11
            assert bin(2**i) == '1' + '0'*i, (i, bin(2**i))                      # 10...00
            assert bin(2**i + 1) == '1' + '0'*(i - 1) + '1', (i, bin(2**i + 1))  # 10...01
        assert bin(2**32 - 1) == '1'*32, bin(2**32)
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 16384):
                assert int(bin(n), 2) == n, (n, bin(n))
                assert len(bin(n)) == nbbits(n), (n, bin(n), nbbits(n))
            for n in range(1, 2**32, 10000000):
                assert int(bin(n), 2) == n, (n, bin(n))
                assert len(bin(n)) == nbbits(n), (n, bin(n), nbbits(n))
            for n in range(2**32 - 16384, 2**32):
                assert int(bin(n), 2) == n, (n, bin(n))
                assert len(bin(n)) == nbbits(n), (n, bin(n), nbbits(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('bit()...', end=''); sys.stdout.flush()
        for i in range(32):
            assert bit(2**i, i) == True, (i, 2**i, bit(2**i, i))
            for j in range(32):
                assert bit(0, i)          == False, (i, bit(0, i))
                assert bit(MERSENNE32, i) == True,  (i, bit(MERSENNE32, i))
                if i != j:
                    assert bit(2**i, j) == False, (i, j, 2**i, bit(2**i, j))
        print('ok'); sys.stdout.flush()


        print('bitset()...', end=''); sys.stdout.flush()
        for n in range(2048):
            for i in range(32):
                if bit(n, i):
                    assert bitset(n, i, False) == n - (1<<i), (n, n - (1<<i), bitset(n, i, False))
                    assert bitset(n, i, True) == n, (n, bitset(n, i, True))
                else:
                    assert bitset(n, i, False) == n, (n, bitset(n, i, False))
                    assert bitset(n, i, True) == n + (1<<i), (n, n + (1<<i), bitset(n, i, True))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                for i in range(32):
                    if bit(n, i):
                        assert bitset(n, i, False) == n - (1<<i), (n, n - (1<<i),
                                                                   bitset(n, i, False))
                        assert bitset(n, i, True) == n, (n, bitset(n, i, True))
                    else:
                        assert bitset(n, i, False) == n, (n, bitset(n, i, False))
                        assert bitset(n, i, True) == n + (1<<i), (n, n + (1<<i), bitset(n, i, True))
            for n in range(2**32 - 2048, 2**32):
                for i in range(32):
                    if bit(n, i):
                        assert bitset(n, i, False) == n - (1<<i), (n, n - (1<<i),
                                                                   bitset(n, i, False))
                        assert bitset(n, i, True) == n, (n, bitset(n, i, True))
                    else:
                        assert bitset(n, i, False) == n, (n, bitset(n, i, False))
                        assert bitset(n, i, True) == n + (1<<i), (n, n + (1<<i), bitset(n, i, True))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('bitset0()...', end=''); sys.stdout.flush()
        for n in range(2048):
            for i in range(32):
                if bit(n, i):
                    assert bitset0(n, i) == n - (1<<i), (n, n - (1<<i), bitset0(n, i))
                else:
                    assert bitset0(n, i) == n, (n, bitset0(n, i))
            assert bitset0(n, i) == bitset(n, i, False), (n, bitset(n, i, False), bitset0(n, i))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                for i in range(32):
                    if bit(n, i):
                        assert bitset0(n, i) == n - (1<<i), (n, n - (1<<i), bitset0(n, i))
                    else:
                        assert bitset0(n, i) == n, (n, bitset0(n, i))
                assert bitset0(n, i) == bitset(n, i, False), (n, bitset(n, i, False), bitset0(n, i))
            for n in range(2**32 - 2048, 2**32):
                for i in range(32):
                    if bit(n, i):
                        assert bitset0(n, i) == n - (1<<i), (n, n - (1<<i), bitset0(n, i))
                    else:
                        assert bitset0(n, i) == n, (n, bitset0(n, i))
                assert bitset0(n, i) == bitset(n, i, False), (n, bitset(n, i, False), bitset0(n, i))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('bitset1()...', end=''); sys.stdout.flush()
        for n in range(2048):
            for i in range(32):
                if bit(n, i):
                    assert bitset1(n, i) == n, (n, bitset1(n, i))
                else:
                    assert bitset1(n, i) == n + (1<<i), (n, n + (1<<i), bitset1(n, i))
            assert bitset1(n, i) == bitset(n, i, True), (n, bitset(n, i, True), bitset1(n, i))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                for i in range(32):
                    if bit(n, i):
                        assert bitset1(n, i) == n, (n, bitset1(n, i))
                    else:
                        assert bitset1(n, i) == n + (1<<i), (n, n + (1<<i), bitset1(n, i))
                assert bitset1(n, i) == bitset(n, i, True), (n, bitset(n, i, True), bitset1(n, i))
            for n in range(2**32 - 2048, 2**32):
                for i in range(32):
                    if bit(n, i):
                        assert bitset1(n, i) == n, (n, bitset1(n, i))
                    else:
                        assert bitset1(n, i) == n + (1<<i), (n, n + (1<<i), bitset1(n, i))
                assert bitset1(n, i) == bitset(n, i, True), (n, bitset(n, i, True), bitset1(n, i))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('lg()...', end=''); sys.stdout.flush()
        for n in range(1, 4096):
            assert lg(n) == len(bin(n)) - 1, (n, lg(n), len(bin(n)))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                assert lg(n) == len(bin(n)) - 1, (n, lg(n), len(bin(n)))
            for n in range(2**32 - 4096, 2**32):
                assert lg(n) == len(bin(n)) - 1, (n, lg(n), len(bin(n)))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('mersenne()...', end=''); sys.stdout.flush()
        for k in range(33):
            assert mersenne(k) + 1 == 2**k, (k, mersenne(k))
        print('ok'); sys.stdout.flush()


        print('mersenne_is()...', end=''); sys.stdout.flush()
        assert mersenne_is(0)
        assert mersenne_is(1)
        assert not mersenne_is(2)
        assert mersenne_is(3)
        for k in range(2, 32):
            assert not mersenne_is(mersenne(k) - 1), k
            assert mersenne_is(mersenne(k)), k
            assert not mersenne_is(mersenne(k) + 1), k
        print('ok'); sys.stdout.flush()


        print('mersenne_to_index()...', end=''); sys.stdout.flush()
        assert mersenne_to_index(0) == 0,    mersenne_to_index(0)
        assert mersenne_to_index(1) == 1,    mersenne_to_index(1)
        assert mersenne_to_index(2) == None, mersenne_to_index(2)
        assert mersenne_to_index(3) == 2,    mersenne_to_index(3)
        for k in range(2, 32):
            assert mersenne_to_index(mersenne(k) - 1) == None, \
                   (k, mersenne_to_index(mersenne(k) - 1))
            assert mersenne_to_index(mersenne(k)) == k, \
                   (k, mersenne_to_index(mersenne(k)    ))
            assert mersenne_to_index(mersenne(k) + 1) == None, \
                   (k, mersenne_to_index(mersenne(k) + 1))
        print('ok'); sys.stdout.flush()


        print('nbbits0()...', end=''); sys.stdout.flush()
        assert nbbits0(0) == 0
        for n in range(1, 4096):
            nb = 0
            for i in range(rscan1(n)):
                if not bit(n, i):
                    nb += 1
            assert nbbits0(n) == nb, (n, nb, nbbits0(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                nb = 0
                for i in range(rscan1(n)):
                    if not bit(n, i):
                        nb += 1
                assert nbbits0(n) == nb, (n, nb, nbbits0(n))
            for n in range(2**32 - 4096, 2**32):
                nb = 0
                for i in range(rscan1(n)):
                    if not bit(n, i):
                        nb += 1
                assert nbbits0(n) == nb, (n, nb, nbbits0(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('nbbits1()...', end=''); sys.stdout.flush()
        assert nbbits1(0) == 0
        for n in range(4096):
            nb = 0
            for i in range(32):
                if bit(n, i):
                    nb += 1
            assert nbbits1(n) == nb, (n, nb, nbbits1(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                nb = 0
                for i in range(32):
                    if bit(n, i):
                        nb += 1
                assert nbbits1(n) == nb, (n, nb, nbbits1(n))
            for n in range(2**32 - 4096, 2**32):
                nb = 0
                for i in range(32):
                    if bit(n, i):
                        nb += 1
                assert nbbits1(n) == nb, (n, nb, nbbits1(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('nbbits()...', end=''); sys.stdout.flush()
        assert nbbits(0) == 0
        assert nbbits(1) == 1
        assert nbbits(2) == 2
        assert nbbits(3) == 2
        assert nbbits(4) == 3
        for n in range(1, 4096):
            l = nbbits(n)
            assert (1<<(l - 1)) <= n, (n, l)
            assert n < (1<<l),        (n, l)
            assert nbbits(n) == len(bin(n)), (n, nbbits(n), bin(n))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                l = nbbits(n)
                assert (1<<(l - 1)) <= n, (n, l)
                assert n < (1<<l),        (n, l)
                assert nbbits(n) == len(bin(n)), (n, nbbits(n), bin(n))
            for n in range(2**32-4096, 2**32):
                l = nbbits(n)
                assert (1<<(l - 1)) <= n, (n, l)
                assert n < (1<<l),        (n, l)
                assert nbbits(n) == len(bin(n)), (n, nbbits(n), bin(n))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('pow2()...', end=''); sys.stdout.flush()
        for k in range(32):
            assert pow2(k) == 2**k, (k, pow2(k))
        print('ok'); sys.stdout.flush()


        print('pow2_is()...', end=''); sys.stdout.flush()
        assert not pow2_is(0)
        assert pow2_is(1)
        assert pow2_is(2)
        assert not pow2_is(3)
        assert pow2_is(4)
        assert not pow2_is(5)
        for k in range(2, 32):
            assert not pow2_is(pow2(k) - 1), k
            assert pow2_is(pow2(k)), k
            assert not pow2_is(pow2(k) + 1), k
        print('ok'); sys.stdout.flush()


        print('pow2_to_index()...', end=''); sys.stdout.flush()
        assert pow2_to_index(0) == None, pow2_to_index(0)
        assert pow2_to_index(1) == 0,    pow2_to_index(1)
        assert pow2_to_index(2) == 1,    pow2_to_index(2)
        assert pow2_to_index(3) == None, pow2_to_index(3)
        assert pow2_to_index(4) == 2,    pow2_to_index(4)
        assert pow2_to_index(5) == None, pow2_to_index(5)
        for k in range(2, 32):
            assert pow2_to_index(pow2(k) - 1) == None, \
                   (k, pow2_to_index(pow2(k) - 1))
            assert pow2_to_index(pow2(k)) == k, \
                   (k, pow2_to_index(pow2(k)    ))
            assert pow2_to_index(pow2(k) + 1) == None, \
                   (k, pow2_to_index(pow2(k) + 1))
        print('ok'); sys.stdout.flush()


        print('rscan0()...', end=''); sys.stdout.flush()
        assert rscan0(0)          == None
        assert rscan0(1)          == None
        assert rscan0(2)          == 0
        assert rscan0(3)          == None
        assert rscan0(4)          == 1
        assert rscan0(5)          == 1
        assert rscan0(MERSENNE32) == None
        for n in range(4096):
            k = None
            for i in range(nbbits(n) - 1, -1, 1):
                if not bit(n, i):
                    k = i
                    break
                assert rscan0(n) == k, (n, rscan0(n), k)
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                k = None
                for i in range(nbbits(n) - 1, -1, 1):
                    if not bit(n, i):
                        k = i
                        break
                    assert rscan0(n) == k, (n, rscan0(n), k)
            for n in range(2**32 - 4096, 2**32):
                k = None
                for i in range(nbbits(n) - 1, -1, 1):
                    if not bit(n, i):
                        k = i
                        break
                    assert rscan0(n) == k, (n, rscan0(n), k)
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('rscan1()...', end=''); sys.stdout.flush()
        assert rscan1(0)          == None
        assert rscan1(1)          == 0
        assert rscan1(2)          == 1
        assert rscan1(3)          == 1
        assert rscan1(4)          == 2
        assert rscan1(5)          == 2
        assert rscan1(MERSENNE32) == 31
        for n in range(1, 4096):
            assert rscan1(n) == len(bin(n)) - 1, (n, rscan1(n), len(bin(n)))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 10000000):
                assert rscan1(n) == len(bin(n)) - 1, (n, rscan1(n), len(bin(n)))
            for n in range(2**32 - 4096, 2**32):
                assert rscan1(n) == len(bin(n)) - 1, (n, rscan1(n), len(bin(n)))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('scan0()...', end=''); sys.stdout.flush()
        assert scan0(0)         == 0,  scan0(0)
        assert scan0(2**32 - 1) == 32, scan0(2**32 - 1)
        for i in range(33):
            assert scan0(2**i - 1) == i, (i, scan0(2**i - 1))
        for n in range(32768):
            if n&1 == 0:
                assert scan0(n) == 0, (n, scan0(n))
            else:
                assert scan0(n) == scan0(n>>1) + 1, (n, scan0(n), scan0(n>>1))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 1000000):
                if n&1 == 0:
                    assert scan0(n) == 0, (n, scan0(n))
                else:
                    assert scan0(n) == scan0(n>>1) + 1, (n, scan0(n), scan0(n>>1))
            for n in range(2**32 - 32768, 2**32):
                if n&1 == 0:
                    assert scan0(n) == 0, (n, scan0(n))
                else:
                    assert scan0(n) == scan0(n>>1) + 1, (n, scan0(n), scan0(n>>1))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('scan1()...', end=''); sys.stdout.flush()
        assert scan1(0) == None, scan1(0)
        for i in range(32):
            assert scan1(2**i) == i, (i, scan1(2**i))
        for n in range(1, 32768):
            if n&1 == 1:
                assert scan1(n) == 0, (n, scan1(n))
            else:
                assert scan1(n) == scan1(n>>1) + 1, (n, scan1(n), scan1(n>>1))
        if debug.assertspeed >= debug.ASSERT_NORMAL:
            for n in range(1, 2**32, 1000000):
                if n&1 == 1:
                    assert scan1(n) == 0, (n, scan1(n))
                else:
                    assert scan1(n) == scan1(n>>1) + 1, (n, scan1(n), scan1(n>>1))
            for n in range(2**32 - 32768, 2**32):
                if n&1 == 1:
                    assert scan1(n) == 0, (n, scan1(n))
                else:
                    assert scan1(n) == scan1(n>>1) + 1, (n, scan1(n), scan1(n>>1))
        else:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
