#!/usr/bin/env python
# -*- coding: latin-1 -*-
##\package DSPython.baset Base tree : arbre binaire &quot;de base b&quot;
# !!! Work in progress !!!

##\file
# Base tree : arbre binaire &quot;de base b&quot;
# !!! Work in progress !!!

# (c) Olivier Pirson --- DragonSoft
# http://www.opimedia.be/DS/
# Dbut le 13 novembre 2007
####################################
from __future__ import print_function

## Date du dernier changement pour ce module
VERSION = 'baset --- 2010 March 16'

import types

import DSPython
import DSPython.nbsystem as nbsystem



# ########
# Classe #
##########
##\brief Abre binaire de base b :
# arbre binaire donc chaque noeud peut contenir un naturel < base.
# Les feulles de l'arbre binaire ne sont jamais nuls.
# Les indices parcourent l'arbre binaire comme suit :
#          0
#    1            2
#  3   4       5     6
# 7 8 9 10   11 12  ...
class Baset:
    """Abre binaire de base b :
    arbre binaire donc chaque noeud peut contenir un naturel < base.
    Les feulles de l'arbre binaire ne sont jamais nuls.
    Les indices parcourent l'arbre binaire comme suit :
             0
       1            2
     3   4       5     6
    7 8 9 10   11 12  ..."""

    ## Constucteur
    def __init__(self, n=0, b=2):
        """Cre un Baset lie  la base b et l'initialise avec n
        Si n est un string il reprsente la valeur de n en base b

        Pre: n: naturel ou string ne contenant que des chiffres < b <= 36
             b: naturel >= 2

        O(n, b) = ..."""
        assert DSPython.natural_is(n) or isinstance(n, str), (type(n), n)
        assert DSPython.natural_is(b), b
        assert 2 <= b, b

        ## Le nombre en base b reprsentant l'arbre
        self.n = (nbsystem.str_to(n, b) if isinstance(n, str)
                  else n)
        ## Base lie  ce Baset
        self.b = b


    ## self[key]
    def __getitem__(self, key):
        """Renvoie l'lment d'indice key

        Pre: key: naturel ou slice

        Result: naturel < self.b

        O(key) = ..."""
        assert DSPython.natural_is(key) or isinstance(key, slice), type(key)

        if DSPython.natural_is(key):  # indice simple
            return nbsystem.fig(self.n, key, self.b)
        else:                         # slice
            n = 0
            p = 1
            if key.step == None:
                for i in range(key.start, key.stop):
                    n += p*self[i]
                    p *= self.b
            else:
                for i in range(key.start, key.stop, key.step):
                    n += p*self[i]
                    p *= self.b
            return n


    ## Nombre d'lments
    def __len__(self):
        """Renvoie le nombre d'lments

        Result: naturel

        O() = ..."""
        nb = 0
        n = self.n
        while n > 0:
            nb += 1
            n //= self.b
        return nb


    ## Renvoie un string quot 'Baset(n, b)'
    def __repr__(self):
        """Renvoie un string quot 'Baset(n, b)'

        Result: string quot

        O() = ..."""
        return ("'Baset({0})'".format(self.n) if self.b == 2
                else "'Baset({0}, b={1})'".format(self.n, self.b))


    ## self[key] = value
    def __setitem__(self, key, value):  # ??? traiter aussi les slice
        """Modifie l'lment d'indice key par value

        Pre: key: naturel
             value: naturel < self.b

        O(key, value) = ..."""
        assert DSPython.natural_is(key), type(key)
        assert DSPython.natural_is(value), type(value)
        assert value < self.b, (value, self.b)

        p = self.b**key
        (q, r) = divmod(self.n, p)
        # "reste" : chiffres qui prcdent
        q //= self.b
        # "quotient" : chiffres qui suivent
        self.n = q*(p*self.b) + (value*p + r)


    ## Renvoie True ssi l'lment d'indice key est une feuille
    def leaf_is(self, key):
        """Renvoie True si l'lment d'indice key est une feuille
        (cd que l'lment est non nul et qu'il ne possde pas de fils),
        False sinon

        Pre: key: naturel

        Result: bool

        O(key) = ..."""
        assert DSPython.natural_is(key), type(key)

        return (self[key] != 0) and ((self.subtree(key) if key > 0
                                      else self).nb_not0() == 1)


    ## Nombre d'lments non nul
    def nb_not0(self):
        """Renvoie le nombre d'lments non nul

        Result: naturel

        O() = ..."""
        nb = 0
        n = self.n
        while n > 0:
            (n, e) = divmod(n, self.b)
            if e > 0:
                nb += 1
        return nb


    ## Renvoie True ssi l'lment d'indice key est un noeud
    def node_is(self, key):
        """Renvoie True si l'lment d'indice key est un noeud
        (cd qu'il est non nul, ou nul mais avec une descendance non nulle),
        False sinon

        Pre: key: naturel

        Result: bool

        O(key) = ..."""
        assert DSPython.natural_is(key), type(key)

        return (self[key] != 0) or ((self.subtree(key) if key > 0
                                     else self).nb_not0() != 0)


    ## Renvoie True ssi tous les noeuds internes de l'arbre sont nuls
    def normal_is(self):
        """Renvoie True si tous les noeuds internes de l'arbre sont nuls
        (cd si que pour chaque feuille ces anctres sont nuls),
        False sinon

        Result: bool

        O() = ..."""

        for k in range(len(self)):
            if self.leaf_is(k) and (not self.orphanleaf_is(k)):
                return False
        return True


    ## Renvoie True ssi l'lment d'indice key est un noeud dont tous les anctre sont nuls
    def orphan_is(self, key):
        """Renvoie True si l'lment d'indice key est un noeud dont tous les anctre sont nuls,
        False sinon

        Pre: key: naturel

        Result: bool

        O(key) = ..."""
        assert DSPython.natural_is(key), type(key)

        if self.node_is(key):
            if key > 0:
                key = (key - 1) // 2
                while (key > 0) and (self[key] == 0):
                    key = (key - 1) // 2
                return self[key] == 0
            else:
                return True
        else:
            return False


    ## Renvoie True ssi l'lment d'indice key est une feuille dont tous les anctres sont nuls,
    # False sinon
    def orphanleaf_is(self, key):
        """Renvoie True si l'lment d'indice key est une feuille dont tous les anctre sont nuls,
        False sinon

        Pre: key: naturel

        Result: bool

        O(key) = ..."""
        assert DSPython.natural_is(key), type(key)

        return self.leaf_is(key) and self.orphan_is(key)


    ## Envoie une reprsentation de l'arbre sur la sortie standard
    def print_tree(self): #???  changer
        """???

        O() = ..."""
        n = self.n
        p = 1
        while n > 0:
            for i in range(p):
                (n, e) = divmod(n, self.b)
                print(e, end='')
            print()
            p *= self.b


    ## Renvoi le sous-arbre dont la racine est l'lment d'indice key
    def subtree(self, key):
        """Renvoi le sous-arbre dont la racine est l'lment d'indice key

        Pre: key: naturel

        Result: Baset

        O(key) = ..."""
        assert DSPython.natural_is(key), type(key)

        if key != 0:
            r = 0   # naturel du sous-arbre rsultat
            p = 1   # puissance de b
            nb = 1  # nombre de chiffres  "extraire"
            n = self.n // self.b**key  # reste du nombre  traiter
            while n > 0:
                c = self[key:key + nb]  # paquet de nb chiffres
                r += c*p
                p *= self.b**nb
                key = key*2 + 1  # indice du premier chiffre du paquet suivant
                n //= self.b**nb
                nb *= 2
            return Baset(r, self.b)
        else:
            return Baset(self.n, self.b)



# ######\cond MAINTEST
# Main #
########
if __name__ == '__main__':
    def main_test():
        """Test du module"""
        import sys

        import DSPython.debug as debug

        debug.test_begin(VERSION, __debug__)

        print('Baset()...', end=''); sys.stdout.flush()
        t = Baset()
        assert t.n == 0, (t.n, t.b)
        assert t.b == 2, (t.n, t.b)

        t = Baset(b=10)
        assert t.n == 0,  (t.n, t.b)
        assert t.b == 10, (t.n, t.b)

        t = Baset(n=123, b=10)
        assert t.n == 123, (t.n, t.b)
        assert t.b == 10,  (t.n, t.b)

        t = Baset(n='1011',)
        assert t.n == 11, (t.n, t.b)
        assert t.b == 2, (t.n, t.b)

        t = Baset(n='123', b=10)
        assert t.n == 123, (t.n, t.b)
        assert t.b == 10,  (t.n, t.b)
        print('ok'); sys.stdout.flush()


        print('Baset.__getitem__()...', end=''); sys.stdout.flush()
        t = Baset()
        for i in range(100):
            assert t[i] == 0, (i, t[i], t)

        t = Baset(b=10)
        for i in range(100):
            assert t[i] == 0, (i, t[i], t)

        t = Baset('543210', b=10)
        for i in range(5):
            assert t[i:i+2] == nbsystem.str_to('012345'[i + 1] + '012345'[i], b=10), \
                   (t, i, t[i:i+2], nbsystem.str_to('012345'[i + 1] + '012345'[i], b=10))

        for b in range(2, 37):
            t = Baset('1'*100, b=b)
            for i in range(1, 100):
                assert t[i] == 1, (b, i, t[i], t)

        for b in range(2, 37):
            for k in range(b):
                c = nbsystem.fig_to_chr(k)
                t = Baset(c*50, b=b)
                for i in range(1, 50 if debug.assertspeed >= debug.ASSERT_NORMAL else 5):
                    assert t[i] == nbsystem.chr_to_fig(c), \
                           (b, k, c, nbsystem.chr_to_fig(c), i, t[i], t)
                assert t[50] == 0, (b, k, t[50], t)
        print('???', end='')
        if debug.assertspeed < debug.ASSERT_NORMAL:
            print(debug.assertspeed_str(), end='')
        print('ok'); sys.stdout.flush()


        print('Baset.__len__()...', end=''); sys.stdout.flush()
        t = Baset()
        assert len(t) == 0, (len(t), t)
        for i in range(100):
            t[i] = i%2
        assert len(t) == 100, (len(t), t)

        t = Baset()
        assert len(t) == 0, (len(t), t)
        for i in range(99):
            t[i] = i%2
        assert len(t) == 98, (len(t), t)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.__repr__()...', end=''); sys.stdout.flush()
        t = Baset(n='123', b=10)
        assert repr(t) == "'Baset({0}, b={1})'".format(t.n, t.b), (repr(t), t)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.__setitem__()...', end=''); sys.stdout.flush()
        t = Baset()
        for i in range(100):
            t[i] = i%2
        for i in range(100):
            assert t[i] == i%2, (i, t[i], t)


        t = Baset(b=10)
        for i in range(100):
            t[i] = i%10
        for i in range(100):
            assert t[i] == i%10, (i, t[i], t)

        t = Baset(b=10)
        for i in range(100,0,-1):
            t[i] = i%10
        for i in range(100,0,-1):
            assert t[i] == i%10, (i, t[i], t)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.leaf_is()...', end=''); sys.stdout.flush()
        t = Baset()
        for k in range(100):
            assert not t.leaf_is(k), (k, t[k])

        t = Baset('9876543010', 10)
        assert not t.leaf_is(0), t[0]
        assert not t.leaf_is(1), t[1]
        assert not t.leaf_is(2), t[2]
        assert not t.leaf_is(3), t[3]
        assert not t.leaf_is(4), t[4]
        assert t.leaf_is(5), t[5]
        assert t.leaf_is(6), t[6]
        assert t.leaf_is(7), t[7]
        assert t.leaf_is(8), t[8]
        assert t.leaf_is(9), t[9]
        for k in range(10, 100):
            assert not t.leaf_is(k), (k, t[k])

        t = Baset('9876543210', 10)
        assert not t.leaf_is(0), t[0]
        assert not t.leaf_is(1), t[1]
        assert not t.leaf_is(2), t[2]
        assert not t.leaf_is(3), t[3]
        assert not t.leaf_is(4), t[4]
        assert t.leaf_is(5), t[5]
        assert t.leaf_is(6), t[6]
        assert t.leaf_is(7), t[7]
        assert t.leaf_is(8), t[8]
        assert t.leaf_is(9), t[9]
        for k in range(10, 100):
            assert not t.leaf_is(k), (k, t[k])
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.nb_not0()...', end=''); sys.stdout.flush()
        t = Baset()
        assert t.nb_not0() == 0, (t.nb_not0(), t)
        for i in range(100):
            t[i] = i%2
        assert t.nb_not0() == 50, (t.nb_not0(), t)

        t = Baset()
        assert t.nb_not0() == 0, (t.nb_not0(), t)
        for i in range(99):
            t[i] = i%2
        assert t.nb_not0() == 49, (t.nb_not0(), t)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.node_is()...', end=''); sys.stdout.flush()
        t = Baset()
        for k in range(100):
            assert not t.node_is(k), (k, t[k])

        t = Baset('9876543210', 10)
        for k in range(10):
            assert t.node_is(k), (k, t[k])
        for k in range(10, 100):
            assert not t.node_is(k), (k, t[k])

        t = Baset('9876543010', 10)
        for k in range(10):
            assert t.node_is(k), (k, t[k])
        for k in range(10, 100):
            assert not t.node_is(k), (k, t[k])
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.normal_is()...', end=''); sys.stdout.flush()
        assert Baset().normal_is(), Baset()
        assert Baset(1).normal_is(), Baset(1)
        assert Baset('10').normal_is(), Baset('10')
        assert not Baset('11').normal_is(), Baset('11')
        assert Baset('100').normal_is(), Baset('100')
        assert not Baset('101').normal_is(), Baset('101')
        assert Baset('110').normal_is(), Baset('110')
        assert not Baset('111').normal_is(), Baset('111')

        assert Baset('3200010', 10).normal_is(), Baset('3200010', 10)
        assert not Baset('3205010', 10).normal_is(), Baset('3205010', 10)

        assert not Baset('9876543210', 10).normal_is(), Baset('9876543210', 10)
        assert not Baset('9876543010', 10).normal_is(), Baset('9876543010', 10)
        assert Baset('9876500000', 10).normal_is(), Baset('9876500000', 10)
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.orphan_is()...', end=''); sys.stdout.flush()
        t = Baset()
        for k in range(100):
            assert not t.orphan_is(k), (k, t[k])

        t = Baset('9876543210', 10)
        assert t.orphan_is(0), t[0]
        assert t.orphan_is(1), t[1]
        assert t.orphan_is(2), t[2]
        for k in range(3, 100):
            assert not t.orphan_is(k), (k, t[k])

        t = Baset('9876543010', 10)
        assert t.orphan_is(0), t[0]
        assert t.orphan_is(1), t[1]
        assert t.orphan_is(2), t[2]
        assert not t.orphan_is(3), t[3]
        assert not t.orphan_is(4), t[4]
        assert t.orphan_is(5), t[5]
        assert t.orphan_is(6), t[6]
        for k in range(7, 100):
            assert not t.orphan_is(k), (k, t[k])
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.orphanleaf_is()...', end=''); sys.stdout.flush()
        t = Baset()
        for k in range(100):
            assert not t.orphanleaf_is(k), (k, t[k])

        t = Baset('9876543210', 10)
        for k in range(100):
            assert not t.orphanleaf_is(k), (k, t[k])

        t = Baset('9876543010', 10)
        assert not t.orphanleaf_is(0), t[0]
        assert not t.orphanleaf_is(1), t[1]
        assert not t.orphanleaf_is(2), t[2]
        assert not t.orphanleaf_is(3), t[3]
        assert not t.orphanleaf_is(4), t[4]
        assert t.orphanleaf_is(5), t[5]
        assert t.orphanleaf_is(6), t[6]
        for k in range(7, 100):
            assert not t.orphanleaf_is(k), (k, t[k])
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.print_tree()...', end=''); sys.stdout.flush()
        t = Baset()
        t.print_tree()
        for i in range(100):
            t[i] = i%2
#        t.print_tree()
        print('???', end='')
        print('ok'); sys.stdout.flush()


        print('Baset.subtree()...', end=''); sys.stdout.flush()
        for b in range(2, 37):
            for k in range(100):
                t = Baset('9876543210', b)
                assert t.subtree(k).b == b, (k, t.subtree(k).b, b)
        t = Baset('9876543210', 10)
        assert t.subtree(0).n  == t.n,    t.subtree(0).n
        assert t.subtree(1).n  == 987431, t.subtree(1).n
        assert t.subtree(2).n  == 652,    t.subtree(2).n
        assert t.subtree(3).n  == 873,    t.subtree(3).n
        assert t.subtree(4).n  == 94,     t.subtree(4).n
        assert t.subtree(5).n  == 5,      t.subtree(5).n
        assert t.subtree(9).n  == 9,      t.subtree(9).n
        assert t.subtree(10).n == 0,      t.subtree(10).n
        print('???', end='')
        print('ok'); sys.stdout.flush()
        debug.test_end()

    main_test()
##\endcond MAINTEST
